///////////////////////////////////////////////////////////////////////////////
// $Id: maptoolrectangle.cpp,v 1.2 2004/10/03 19:30:45 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    maptoolrectangle.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the rectangle tool class for maps
*/

// local includes
#include "mapdocument.h"
#include "maptoolrectangle.h"

///////////////////////////////////////////////////////////////////////////////

MapToolRectangle::MapToolRectangle(MapDocument* doc) : MapTool(doc)
{
}

///////////////////////////////////////////////////////////////////////////////

MapToolRectangle::~MapToolRectangle()
{
}

///////////////////////////////////////////////////////////////////////////////

void MapToolRectangle::activate()
{
	// clear rect
	m_rect = QRect();
	m_start = QPoint();
	m_end = QPoint();
}

///////////////////////////////////////////////////////////////////////////////

bool MapToolRectangle::usedMouseRelease(uint row, uint col, int button)
{
	if (button == Qt::RightButton)
	{
		activate();
		return true;
	}
	
	if (button != Qt::LeftButton) return false;

	// sanity checks
	if (m_doc == 0) return false;
	if (static_cast<int>(row) >= m_doc->numRows() ||
	    static_cast<int>(col) >= m_doc->numCols()) return false;

	// Do not draw on invisible layers
	if (!m_doc->isLayerVisible(m_doc->currentLayer())) return false;
	
	// if the rectangle is already valid, this is the second click
	// which finishes it
	if (m_rect.isNull())
	{
		m_rect = QRect(col, row, 1, 1);
		m_start = QPoint(col, row);
	}
	else
	{
		m_end = QPoint(col, row);
		m_rect = QRect(m_start, m_end).normalize();
		m_rect.setWidth(m_rect.width() + 1);
		m_rect.setHeight(m_rect.height() + 1);

		for (int r = m_rect.y(); r < m_rect.bottom(); ++r)
		{
			for (int c = m_rect.x(); c < m_rect.right(); ++c)
			{
				m_doc->drawCurrentTileAt(r, c);
			}
		}
		m_rect = QRect();
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapToolRectangle::usedMouseMove(uint row, uint col, int)
{
	// sanity checks
	if (m_doc == 0) return false;
	
	// Do not draw on invisible layers
	if (!m_doc->isLayerVisible(m_doc->currentLayer())) return false;
	
	if (m_rect.isNull()) return false;

	m_end = QPoint(col, row);
	m_rect = QRect(m_start, m_end).normalize();

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapToolRectangle::coversTileAt(uint row, uint col) const
{
	return m_rect.contains(col, row);
}

///////////////////////////////////////////////////////////////////////////////

QRect MapToolRectangle::coverArea() const
{
	return m_rect;
}

// End of file
