# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: DNA/RNA/protein sequence converter.

# load libs
import wx

# load modules
import count

class mConv(wx.Dialog):
    """DNA/RNA/protein sequence converter."""

    # ----
    def __init__(self, parent):
        wx.Dialog.__init__(self, parent, -1, "Sequence Converter", style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        # init module variables
        self.data = ''

        # make gui items
        sequences = self.makeSequenceBox()
        buttons = self.makeButtons()

        # pack items
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(sequences, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALIGN_RIGHT|wx.ALL, 10)
        else:
            mainSizer.Add(sequences, 1, wx.EXPAND|wx.ALL, 5)
            mainSizer.Add(buttons, 0, wx.ALIGN_RIGHT)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeSequenceBox(self):
        """ Make main boxes for sequences. """

        # make items
        sourceLabel = wx.StaticText(self, -1, "Source sequence: ")
        self.source = wx.TextCtrl(self, -1, '', size=(500, 150), style=wx.TE_MULTILINE)
        result_label = wx.StaticText(self, -1, "Converted sequence: ")
        self.result = wx.TextCtrl(self, -1, '', size=(500, 150), style=wx.TE_MULTILINE|wx.TE_READONLY)

        # pack items
        sequenceBox = wx.BoxSizer(wx.VERTICAL)
        sequenceBox.Add(sourceLabel, 0)
        sequenceBox.Add(self.source, 1, wx.EXPAND|wx.TOP, 5)
        sequenceBox.Add(result_label, 0, wx.EXPAND|wx.TOP, 5)
        sequenceBox.Add(self.result, 1, wx.EXPAND|wx.TOP, 5)

        # set events
        self.source.Bind(wx.EVT_TEXT, self.seqConverter)

        return sequenceBox
    # ----


    # ----
    def makeSwitch(self):
        """ Make box with sequence type switch. """

        # make items
        self.switch1to1_radio = wx.RadioButton(self, -1, "1 -> 1 filter", style=wx.RB_GROUP)
        self.switch1to1_radio.SetToolTip(wx.ToolTip("Filter 1-letter protein sequence"))
        self.switch3to1_radio = wx.RadioButton(self, -1, "3 -> 1")
        self.switch3to1_radio.SetToolTip(wx.ToolTip("3-letter protein to 1-letter protein"))
        self.switch1to3_radio = wx.RadioButton(self, -1, "1 -> 3")
        self.switch1to3_radio.SetToolTip(wx.ToolTip("1-letter protein to 3-letter protein"))
        self.switchNAto1_radio = wx.RadioButton(self, -1, "DNA/RNA -> 1")
        self.switchNAto1_radio.SetToolTip(wx.ToolTip("DNA/RNA to 1-letter protein"))

        # pack items
        switchBox = wx.BoxSizer(wx.HORIZONTAL)
        switchBox.Add(self.switch1to1_radio, 0, wx.LEFT|wx.RIGHT, 5)
        switchBox.Add(self.switch3to1_radio, 0, wx.LEFT|wx.RIGHT, 5)
        switchBox.Add(self.switch1to3_radio, 0, wx.LEFT|wx.RIGHT, 5)
        switchBox.Add(self.switchNAto1_radio, 0, wx.LEFT|wx.RIGHT, 5)

        # set events
        self.switch1to1_radio.Bind(wx.EVT_RADIOBUTTON, self.seqConverter)
        self.switch3to1_radio.Bind(wx.EVT_RADIOBUTTON, self.seqConverter)
        self.switch1to3_radio.Bind(wx.EVT_RADIOBUTTON, self.seqConverter)
        self.switchNAto1_radio.Bind(wx.EVT_RADIOBUTTON, self.seqConverter)

        # set defaults
        self.switch1to1_radio.SetValue(True)

        return switchBox
    # ----


    # ----
    def makeButtons(self):
        """ Make box with buttons. """

        # make items
        self.Send_button = wx.Button(self, wx.ID_OK, "Send")
        self.Send_button.SetToolTip(wx.ToolTip("Send sequence to sequence editor"))
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(self.makeSwitch(), 0, wx.ALL, 5)
        buttonBox.Add(self.Send_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set defaults
        self.Send_button.Enable(False)

        return buttonBox
    # ----


    # ----
    def seqConverter(self, evt):
        """ Get and convert sequence. """
    
        # get source sequence
        sourceSeq = self.source.GetValue()

        # get switch and count
        if self.switch1to1_radio.GetValue():
            switch = '1to1'
        elif self.switch3to1_radio.GetValue():
            switch = '3to1'
        elif self.switch1to3_radio.GetValue():
            switch = '1to3'
        elif self.switchNAto1_radio.GetValue():
            switch = 'NAto1'

        # convert sequence
        resultSeq = count.converter(sourceSeq, switch)

        # show result
        if resultSeq:
            self.result.SetValue(resultSeq)
            self.data = resultSeq
        else:
            self.result.SetValue("Bad sequence or converter type")
            self.data = ''

        # enable/disable send button
        if (switch == '1to1' or switch == '3to1' or switch == 'NAto1') and resultSeq:
            self.Send_button.Enable(True)
        else:
            self.Send_button.Enable(False)
    # ----
