# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Main application frame.

# load libs
import sys
import wx

# load modules
from nucleus.images import getSplashBitmap
from nucleus.docframe import docFrame
from nucleus.config import mConfig


class mMass(wx.App):
    """Raise application"""

    # ----
    def OnInit(self):
        """" Raise new application. """

        # show splash screen
        splash = wx.SplashScreen(getSplashBitmap(),
                    wx.SPLASH_CENTRE_ON_SCREEN|wx.SPLASH_TIMEOUT,
                    2000, None, -1)
        splash.Show()

        # load app configs
        self.config = mConfig()

        # init document container
        self.documents = []

        # raise application frame
        frame = docFrame(self)

        # add frame to document container
        self.documents.append(frame)

        # update file history
        self.updateFileHistory()

        # open file from commandline
        if len(sys.argv) == 2:
            frame.onFileOpen(path=sys.argv[1])

            # destroy splash screen
            try:
                splash.Destroy()
            except:
                pass

        return True
    # ----


    # ----
    def OnExit(self):
        """" Save config before exit. """
        self.config.saveCfgOnExit()
    # ----


    # ----
    def MacOpenFile(self, path):
        """" Enable drag/drop under Mac. """

        # open file
        self.documents[0].onFileOpen(path=path)
    # ----


    # ----
    def appMonitor(self, event, value=None):
        """" Main application monitor. """

        # open blank document
        if event == 'onFileNew':
            self.onFileNew()

        # open document
        elif event == 'onFileOpen':
            self.onFileOpen(value)

        # open document
        elif event == 'onFileSave':
            self.updateFileHistory(value)

        # document closed
        elif event == 'onFileClose':
            return self.onFileClose(value)

        # exit application
        elif event == 'onFileExit':
            self.onFileExit()

        # check if document already open
        elif event == 'checkOpenPath':
            return self.checkOpenPath(value)

        # check applied modification in all documents
        elif event == 'checkModifications':
            return self.checkModifications()
    # ----


    # ----
    def onFileNew(self):
        """" Open blank document. """

        # check for empty frame
        if len(self.documents) == 1 and self.documents[0].isEmpty():
            document = self.documents[0]

        # raise new document frame
        else:
            document = docFrame(self)
            self.documents.append(document)

        # make document
        document.makeDocument()
        document.setTitle('Untitled')
    # ----


    # ----
    def onFileOpen(self, data):
        """" Open selected document. """

        # check for empty frame
        if len(self.documents) == 1 and self.documents[0].isEmpty():
            document = self.documents[0]

        # raise new document frame
        else:
            document = docFrame(self)
            self.documents.append(document)

        # make document
        document.makeDocument(data)
        document.setTitle()

        # update file history
        self.updateFileHistory(data['path'])
    # ----


    # ----
    def onFileClose(self, document):
        """" Close document. """

        # remove document from container if not last
        if len(self.documents) > 1:
            self.documents.remove(document)
            return True

        elif document:
            document.setTitle()
            return False
    # ----


    # ----
    def onFileExit(self):
        """" Close all opened documents. """

        while self.documents:
            doc = self.documents[0]
            if doc.isEmpty() or doc.onFileClose() == False:
                return
    # ----


    # ----
    def checkOpenPath(self, path):
        """" Check if document already open. Check also if any scanID is set
            to allow different scan to be loaded."""

        for doc in self.documents:
            if doc.checkSameDocument(path):
                doc.Raise()
                return True

        return False
    # ----


    # ----
    def checkModifications(self):
        """" Check applied modifications in all documents. """

        valid = True
        for doc in self.documents:
            if not doc.checkAppliedModifications():
                valid = False

        return valid
    # ----


    # ----
    def updateFileHistory(self, path=False):
        """ Manage list of previously loaded files. """

        # get current history list
        historyList = []
        for x in range(10):
            historyList.append(self.config.cfg['history']['f'+str(x)])

        # update history list
        if path:

            # remove file if exists
            if path in historyList:
                historyList.remove(path)

            # add file to list
            historyList.insert(0, path)

            # check length
            if len(historyList) > 10:
                del historyList[-1]

            # update config
            for x, item in enumerate(historyList):
                self.config.cfg['history']['f'+str(x)] = item

        # update main menu in all opened documents
        for doc in self.documents:
            doc.docMonitor('updateFileHistory', historyList)
    # ----

# --------------------------------------------------

if __name__ == '__main__':

    # run application
    app = mMass(0)
    app.MainLoop()
