/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

// 	$Id$	

#ifndef _mira_parameters_H
#define _mira_parameters_H

#include "stdinc/defines.H"

#include <iostream>
#include <fstream>

#include <sstream>
#include <algorithm>                     // STL min max template

// BOOST string functions
#include <boost/algorithm/string.hpp> 

#include "errorhandling/errorhandling.H"

#include "mira/structs.H"

#undef yyFlexLexer
#define yyFlexLexer MPFlexLexer
#include <FlexLexer.h>

class MIRAParameters
{
private:

  enum qsj_jobareas_t {JA_QUALITY, JA_TYPE, JA_METHOD, JA_TECH};

  static string MP_currentparametersection;
  static bool MP_errorinparams;
  static vector<string>        MP_loadfilename;


  special_parameters    mp_special_params;
  skim_parameters       mp_skim_params;
  align_parameters      mp_align_params;
  dynamic_parameters    mp_dynamic_params;
  assembly_parameters   mp_assembly_params;
  pathfinder_parameters mp_pathfinder_params;
  contig_parameters     mp_contig_params;
  directory_parameters  mp_directory_params;
  edit_parameters       mp_edit_params;

  bool MP_parsedsomesettings;

  //Functions
private:
  void foolCompiler();

  template <class T>
  void checkMin(const char * parameter, T & var, const T & minval, const T & defaultval)
    {
      if(var < minval){
	cout << parameter << " is " << var << ", but must be at least " << minval << ". Setting to " << defaultval << endl; 
	var=defaultval;
      }
    }
  template <class T>
  void checkMax(const char * parameter, T & var, const T & maxval, const T & defaultval)
    {
      if(var > maxval){
	cout << parameter << " is " << var << ", but must be no more than " << maxval << ". Setting to " << defaultval << endl; 
	var=defaultval;
      }
    }

  // doubled from Read class ... to circumvent include of read.H in this header file
  // is there a better way?
  enum read_sequencing_type {SEQTYPE_SANGER=0, SEQTYPE_454GS20, SEQTYPE_IONTORRENT, SEQTYPE_PACBIO, SEQTYPE_SOLEXA, SEQTYPE_ABISOLID, SEQTYPE_END};
  
  template <class T>
  static void multiParamPrint(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr, T & varInPv0, const char * indent, const char * desc, int32 fieldlength){
    multiParamPrint_(Pv, indexesInPv, ostr, varInPv0, 
		     indent, desc, fieldlength,
		     false, false);
  }
  template <class T>
  static void multiParamPrintNumericChar(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr, T & varInPv0, const char * indent, const char * desc, int32 fieldlength){
    multiParamPrint_(Pv, indexesInPv, ostr, varInPv0, 
		     indent, desc, fieldlength,
		     false, true);
  }
  template <class T>
  static void multiParamPrintBool(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr, T & varInPv0, const char * indent, const char * desc, int32 fieldlength){
    multiParamPrint_(Pv, indexesInPv, ostr, varInPv0, 
		     indent, desc, fieldlength,
		     true, false);
  }

  template <class T>
  static void multiParamPrint_(vector<MIRAParameters> & Pv, const vector<int> & indexesInPv, ostream & ostr, T & varInPv0, const char * indent, const char * desc, int32 fieldlength, bool asboolean, bool numericcast){

    int32 offsetOfVarInPv0=static_cast<int32>(reinterpret_cast<char *>(&varInPv0) - reinterpret_cast<char *>(&(Pv[0])));

    ostr << indent << desc;

    for(int32 firstpad=fieldlength-static_cast<int32>(strlen(desc));firstpad>1;firstpad--) ostr << ' ';

    ostr << ':';
    if(indexesInPv.size()==1){
      char * addrPv0= reinterpret_cast<char *>(&Pv[indexesInPv[0]]);
      char * valInPv0= reinterpret_cast<char *>(&(addrPv0[offsetOfVarInPv0])); 
      if(asboolean) {
	if(*(reinterpret_cast<bool *>(valInPv0))) {
	  ostr << " yes\n";
	}else{
	  ostr << " no\n";
	}
      }else if(numericcast) {
	ostr << ' ';
	charAsNum(ostr, valInPv0);
	ostr << '\n';
	//ostr << ' ' << showpos << varInPv0 << '\n';
      }else{
	ostr << ' ' << *(reinterpret_cast<T*>(valInPv0)) << '\n';
      }
    }else{
      //ostr << '\n';
      for(uint32 ipvindex=0; ipvindex<indexesInPv.size(); ipvindex++){
	if(ipvindex>0) {
	  ostr << indent;
	  for(int32 pad=fieldlength;pad>0;pad--) ostr << ' ';
	}
	if(indexesInPv[ipvindex] >= static_cast<int>(Pv.size())) {
	  cerr << "BARF! MIRAParameters broken, wanted to read elem " << indexesInPv[ipvindex] << " for vector of size " << Pv.size() << "\n";
	  abort();
	}
	switch(indexesInPv[ipvindex]){
	case SEQTYPE_SANGER:{
	  ostr << "  [san] ";
	  break;
	}
	case SEQTYPE_454GS20:{
	  ostr << "  [454] ";
	  break;
	}
	case SEQTYPE_IONTORRENT:{
	  ostr << "  [ion] ";
	  break;
	}
	case SEQTYPE_PACBIO:{
	  ostr << "  [pbs] ";
	  break;
	}
	case SEQTYPE_SOLEXA:{
	  ostr << "  [sxa] ";
	  break;
	}
	case SEQTYPE_ABISOLID:{
	  ostr << "  [sid] ";
	  break;
	}
	default : {
	  cerr << "BARF! MIRAParameters broken, wanted readtype " << indexesInPv[ipvindex] << "\n";
	  abort();
	}
	}

	char * valInPvX;
	{
	  char * addrPvX= reinterpret_cast<char *>(&Pv[indexesInPv[ipvindex]]);
	  valInPvX = reinterpret_cast<char *>(&(addrPvX[offsetOfVarInPv0]));
	}

	ostr << ' ';
	if(asboolean) {
	  if(*(reinterpret_cast<bool*>(valInPvX))==true) {
	    ostr << "yes\n";
	  }else{
	    ostr << "no\n";
	  }
	}else if(numericcast) {
	  charAsNum(ostr, valInPvX);
	  ostr << '\n';
	}else{
	  ostr << *(reinterpret_cast<T*>(valInPvX)) << '\n';
	}
      }
    }
  }

  static void charAsNum(ostream & ostr, char * c){
    char x=*c;
    ostr << static_cast<int>(x);
  }

  static int32 gimmeAnInt(FlexLexer * lexer, stringstream & errstream);
  static double gimmeADouble(FlexLexer * lexer, stringstream & errstream);
  static int32 getFixedStringMode(FlexLexer * lexer, stringstream & errstream);
  static void checkCOMMON(const string & currentst,
			  FlexLexer * lexer, 
			  stringstream & errstream);
  static void checkNONCOMMON(const string & currentst,
			     FlexLexer * lexer, 
			     stringstream & errstream);
			  

  static void interpretJobDefs(vector<MIRAParameters> & Pv, 
			       vector<vector<uint32> > & jobdefs, 
			       stringstream & errstream);

  static void saveParsedSettingsValues(vector<MIRAParameters> & Pv,
				       MIRAParameters * actpar, 
				       vector<bool> & saved);
  static void restoreParsedSettingsValues(vector<MIRAParameters> & Pv,
					  MIRAParameters * actpar, 
					  vector<bool> & saved);

public:
  MIRAParameters();
  ~MIRAParameters();
  friend ostream & operator<<(ostream &ostr, MIRAParameters const &mp);

  static void setupStdMIRAParameters(vector<MIRAParameters> & Pv, 
				     bool verbose=false);
  static void postParsingChanges(vector<MIRAParameters> & Pv);

  static void dumpAllParams(vector<MIRAParameters> & Pv, 
			    ostream & ostr);
  static void dumpAllParams(vector<MIRAParameters> & Pv, 
			    const vector<int> & indexesInPvm,
			    ostream & ostr);
  static void dumpAlignParams(vector<MIRAParameters> & Pv, 
			      const vector<int> & indexesInPvm,
			      ostream & ostr);

  static void dumpGeneralParams(vector<MIRAParameters> & Pv, 
				const vector<int> & indexesInPvm,
				ostream & ostr);
  static void dumpMiscParams(vector<MIRAParameters> & Pv, 
			     const vector<int> & indexesInPvm,
			     ostream & ostr);
  static void dumpLoadParams(vector<MIRAParameters> & Pv, 
			     const vector<int> & indexesInPvm,
			     ostream & ostr);
  //static void dumpShortReadParams(vector<MIRAParameters> & Pv, 
  //			   const vector<int> & indexesInPvm,
  //			   ostream & ostr);
  static void dumpAssemblyParams(vector<MIRAParameters> & Pv, 
				 const vector<int> & indexesInPvm,
				 ostream & ostr);
  static void dumpStrainBackboneParams(vector<MIRAParameters> & Pv, 
				       const vector<int> & indexesInPvm,
				       ostream & ostr);
  static void dumpDataProcessingParams(vector<MIRAParameters> & Pv, 
				       const vector<int> & indexesInPvm,
				       ostream & ostr);
  static void dumpClippingParams(vector<MIRAParameters> & Pv, 
				 const vector<int> & indexesInPvm,
				 ostream & ostr);
  static void dumpSkimParams(vector<MIRAParameters> & Pv, 
			     const vector<int> & indexesInPvm,
			     ostream & ostr);
  static void dumpPathfinderParams(vector<MIRAParameters> & Pv, 
				   const vector<int> & indexesInPvm,
				   ostream & ostr);
  
  static void dumpContigParams(vector<MIRAParameters> & Pv, 
			       const vector<int> & indexesInPvm,
			       ostream & ostr);
  static void dumpEditParams(vector<MIRAParameters> & Pv, 
			     const vector<int> & indexesInPvm,
			     ostream & ostr);
  static void dumpDirectoryParams(vector<MIRAParameters> & Pv, 
				  const vector<int> & indexesInPvm,
				  ostream & ostr);
  static void dumpFileInParams(vector<MIRAParameters> & Pv, 
			       const vector<int> & indexesInPvm,
			       ostream & ostr);
  static void dumpFileOutParams(vector<MIRAParameters> & Pv, 
				const vector<int> & indexesInPvm,
				ostream & ostr);
  static void dumpFileTempParams(vector<MIRAParameters> & Pv, 
				 const vector<int> & indexesInPvm,
				 ostream & ostr);
  static void dumpOutputCustomisationParams(vector<MIRAParameters> & Pv, 
					    const vector<int> & indexesInPvm,
					    ostream & ostr);
  static void dumpFileDirectoryOutNamesParams(vector<MIRAParameters> & Pv, 
					      const vector<int> & indexesInPvm,
					      ostream & ostr);

  
  void dumpGeneralParams(ostream &ostr) const;
  void dump454Params(ostream &ostr) const;
  void dumpShortReadParams(ostream &ostr) const;
  void dumpAssemblyParams(ostream &ostr) const;
  void dumpStrainBackboneParams(ostream &ostr) const;
  void dumpDataProcessingParams(ostream &ostr) const;
  void dumpClippingParams(ostream &ostr) const;
  void dumpSkimParams(ostream &ostr) const;
  void dumpPathfinderParams(ostream &ostr) const;
  void dumpAlignParams(ostream &ostr) const;
  void dumpContigParams(ostream &ostr) const;
  void dumpEditParams(ostream &ostr) const;
  void dumpDirectoryParams(ostream &ostr) const;
  void dumpFileInParams(ostream &ostr) const;
  void dumpFileOutParams(ostream &ostr) const;
  void dumpFileTempParams(ostream &ostr) const;
  void dumpOutputCustomisationParams(ostream &ostr) const;
  void dumpFileDirectoryOutNamesParams(ostream &ostr) const;

  template <class T>
  static void multiParamPrint(vector<MIRAParameters> & Pv, const T & var, const string & desc);

  void consistencyCheck();

  inline special_parameters const & getSpecialParams() const 
    { return mp_special_params; }
  inline skim_parameters const & getSkimParams() const
    { return mp_skim_params; }
  inline align_parameters const & getAlignParams() const
    { return mp_align_params; }
  inline dynamic_parameters const & getDynamicParams() const
    { return mp_dynamic_params; }
  inline assembly_parameters const & getAssemblyParams() const
    { return mp_assembly_params; }
  inline pathfinder_parameters const & getPathfinderParams() const
    { return mp_pathfinder_params; }
  inline contig_parameters const & getContigParams() const
    { return mp_contig_params; }
  inline directory_parameters const & getDirectoryParams() const
    { return mp_directory_params; }
  inline edit_parameters const & getEditParams() const
    { return mp_edit_params; }

  inline special_parameters & getNonConstSpecialParams() 
    { return mp_special_params; }
  inline skim_parameters & getNonConstSkimParams()
    { return mp_skim_params; }
  inline align_parameters & getNonConstAlignParams()
    { return mp_align_params; }
  inline dynamic_parameters & getNonConstDynamicParams()
    { return mp_dynamic_params; }
  inline assembly_parameters & getNonConstAssemblyParams()
    { return mp_assembly_params; }
  inline pathfinder_parameters & getNonConstPathfinderParams()
    { return mp_pathfinder_params; }
  inline contig_parameters & getNonConstContigParams()
    { return mp_contig_params; }
  inline directory_parameters & getNonConstDirectoryParams()
    { return mp_directory_params; }
  inline edit_parameters & getNonConstEditParams()
    { return mp_edit_params; }

  static void loadParams(const string & pfile, vector<MIRAParameters> & Pv);
  static void parse(int argc, char **argv, vector<MIRAParameters> & Pv);
  static void parse(const char * params, vector<MIRAParameters> & Pv);
  static void parse(const string & params, vector<MIRAParameters> & Pv) {parse(params.c_str(),Pv);};
  static void parse(istream & is, 
		    vector<MIRAParameters> & Pv,
		    MIRAParameters * mp);
  static void parseQuickmode(const char * params, 
			     const char * qm,
			     vector<MIRAParameters> & Pv,
			     bool verbose=true,
			     MIRAParameters * mp=NULL);
  static void parseQuickmodeNoTechSettingsChange(
    const char * params, 
    const char * qm,
    vector<MIRAParameters> & Pv,
    bool verbose=true,
    MIRAParameters * mp=NULL);

  static void generateProjectNames(vector<MIRAParameters> & Pv, string name="");
  static void generateProjectInNames(vector<MIRAParameters> & Pv, string name="");
  static void generateProjectOutNames(vector<MIRAParameters> & Pv, string name="");
  static void correctTmpDirectory(vector<MIRAParameters> & Pv);

  static void setBorgMode(vector<MIRAParameters> & Pv, 
			  bool verbose);
  static void setQuickmodeGenomeDraft(vector<MIRAParameters> & Pv,
				      bool verbose);
  static void setQuickmodeGenomeNormal(vector<MIRAParameters> & Pv, 
				       bool verbose);
  static void setQuickmodeGenomeAccurate(vector<MIRAParameters> & Pv,
					 bool verbose);

  int32 setSpecialParams(special_parameters & sp);
  int32 setSkimParams(skim_parameters & sk);
  int32 setAlignParams(align_parameters & ap);
  int32 setDynamicParams(dynamic_parameters & dp);
  int32 setAssemblyParams(assembly_parameters & ap);
  int32 setPathfinderParams(pathfinder_parameters & pp);
  int32 setContigParams(contig_parameters & cp);
  int32 setDirectoryParams(directory_parameters & cp);
  int32 setEditParams(edit_parameters & cp);

  void setPathfinderuseGenomicAlgorithms(bool b);
  void setPathfinderMaxContigTime(uint32 t); 
  void setPathfinderNextReadMaxTimeToESS(uint32 t); 

  void setContigRODIRS(uint32 r);
  void setContigAssumeSNPInsteadofRepeats(bool b);
  void setContigDisregardSpuriousRMBMismatches(bool b);
  void setContigMinReadsPerGroup(uint32 i);
  void setContigEndReadMarkExclusionArea(uint32 r);
  void setContigMinNeighbourQuality(base_quality_t q);
  void setContigMinGroupQuality(base_quality_t q);
  void setContigMarkGapBases(bool b);
  void setContigMarkMulticolumnGapBases(bool b);

  void setContigAlignmentOutputTextLineLen(int32 l);
  void setContigAlignmentOutputHTMLLineLen(int32 l);
  void setContigAlignmentOutputTextGapPad(char c);
  void setContigAlignmentOutputHTMLGapPad(char c);
  void setContigForceNonIUPAC(bool perseq, bool amongseq);

  void setAlignMaxCutoff(uint32 mco);
  void setAlignMinScore(uint32 ms);
  void setAlignMinRelScore(uint32 mrs);
  void setAlignMinOverlap(uint32 mo);
  void setAlignBandwidthMin(uint32 bm);
  void setAlignBandwidthMax(uint32 bm);
  void setAlignBandwidthInPercent(uint32 bip);
  void setAlignGapPenaltyLevel(uint32 level);

  void setEditAutomaticContigEditing(bool b);
  void setEditStrictEditingMode(bool b);
  void setEditConfirmationThreshold(uint32 r);

  void setAssemblyProjectName(const string & s);
  void setAssemblyInfileFOFNEXP( const string & s);
  void setAssemblyInfileFASTA( const string & s);
  void setAssemblyInfileFASTAQUAL( const string & s);
  void setAssemblyInfileCAF( const string & s);
  void setAssemblyInfileStrainData( const string & s);
  void setAssemblyOutfileCAF( const string & s);
  void setAssemblyOutdirGAP4DA( const string & s);

  void setAssemblyUseTemplateInformation(bool u);
  void setAssemblyPatternLength(uint32 pl);
  void setAssemblyNumEndErrors(uint32 nee);
  void setAssemblyNumMiddleErrors(uint32 nme);
  void setAssemblyNumMiddleRuns(uint32 nmr);
  void setAssemblyNumRMBBreakLoops(int32 i);
  void setAssemblyLoadBackbone(bool b);
  void setAssemblyStartBackboneUsageInPass(int32 i);
  void setAssemblyNumPasses(int32 i);
  void setAssemblyInfileWhich(string & iw);
  void setAssemblyLogUnusedIDs(const string & s);
  void setAssemblyLogADS( const string & s);
  void setAssemblyReadExtension(bool b);
  void setAssemblyMarkRepeats(bool b);
  void setAssemblyMinLength(uint32 r);
  void setAssemblyLoadStrainData(bool b);
  void setAssemblyClipPossibleVectors(bool b);
  void setAssemblyPutAssembledWithMIRATags(bool b);
  void setAssemblyFASTQQualOffset(base_quality_t b);

};

#endif




