#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005-2007 by Enthought, Inc.
#  All rights reserved.
#
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

"""
The node type associated with our project resources.

"""

# Standard library imports.
import logging

# Enthought library imports
from enthought.envisage import get_application
from enthought.envisage.resource import ResourceNodeType
from enthought.pyface.api import ImageResource

# Local imports.
from plugin_definition import PROJECT_MENU_ID
from project_node_monitor import ProjectNodeMonitor


# Setup a logger for this module.
logger = logging.getLogger(__name__)


class ProjectNodeType(ResourceNodeType):
    """
    The node type associated with our project resources.

    """

    ##########################################################################
    # Traits
    ##########################################################################

    #### public 'ResourceNodeType' interface #################################

    # If set, then we build our context menu by querying for actions, groups,
    # and menus, within ActionSets, which are configured for locations starting
    # with this path.
    action_location_root = PROJECT_MENU_ID


    #### public 'NodeType' interface #########################################

    # The image used to represent nodes of this type that are NOT expanded.
    closed_image = ImageResource('folder')

    # The image used to represent nodes of this type that ARE expanded.
    open_image = ImageResource('folder-open')


    ##########################################################################
    # 'NodeType' interface
    ##########################################################################

    #### public interface ####################################################

    def get_monitor(self, node):
        """
        Returns a monitor that detects changes to a node.

        """

        monitor = ProjectNodeMonitor(node=node)

        return monitor


    def is_collapsible(self, node):
        """
        Overridden to ensure that our projects don't collapse.

        """

        return False


    ##########################################################################
    # 'ResourceNodeType' interface
    ##########################################################################

    #### protected interface #################################################

    def _get_action_sets(self):
        """
        Returns all action set contributions.

        """

        from enthought.envisage.single_project.plugin_definition import \
            SingleProjectActionSet
        action_sets = get_application().load_extensions(SingleProjectActionSet)
        logger.debug('Found contribution action sets:')
        for set in action_sets:
            logger.debug('  Name: [%s]', set.name)

        return action_sets


#### EOF #####################################################################
