""" An adapter factory proxy. """


# Enthought library imports.
from enthought.envisage.api import Application
from enthought.traits.api import Instance, Property, Any
from enthought.type_manager.api import AdapterFactory


class AdapterFactoryProxy(AdapterFactory):
    """ A proxy for an adapter factory.

    The actual adapter factory is created lazily.

    """

    #### 'AdapterFactoryProxy' interface ######################################

    # The application that the proxy is part of.
    application = Instance(Application)

    # The extension that defines the factory.
    extension = Any

    # The factory that this object is a proxy for (lazily loaded).
    factory = Property(Instance(AdapterFactory))

    #### Private interface ####################################################

    # The factory that this object a proxy for (lazily loaded).
    _factory = Instance(AdapterFactory)
    
    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, application, extension):
        """ Creates a new proxy. """

        super(AdapterFactoryProxy, self).__init__(
            application=application, extension=extension
        )

        return
    
    ###########################################################################
    # 'AdapterFactory' interface.
    ###########################################################################

    def adapt(self, adaptee, target_class, *args, **kw):
        """ Returns an adapter that adapts an object to the target class.

        Returns '''None''' if the factory cannot produce such an adapter.

        """

        # This causes the factory to be loaded.
        return self._factory.adapt(adaptee, target_class, *args, **kw)

    ###########################################################################
    # 'AdapterFactoryProxy' interface.
    ###########################################################################

    #### Properties ###########################################################
    
    def _get_factory(self):
        """ Property getter. """
        
        return self._factory

    ###########################################################################
    # Private interface.
    ###########################################################################

    #### Initializers #########################################################

    def __factory_default(self):
        """ Trait initializer for the '''_factory''' trait. """

        # Shorthand.
        extension     = self.extension
        import_symbol = self.application.import_symbol

        # We are about to import the classes defined in the extension, so make
        # sure that the plugin that the factory was defined in has been
        # started.
        self.application.start_plugin(extension._definition_.id)

        # If a specific factory class was specified then create an instance of
        # it.
        if len(extension.class_name) > 0:
            klass   = import_symbol(extension.class_name)
            factory = klass(adapter_manager=self.adapter_manager)

        # Otherwise, create an instance of the default adapter factory class.
        else:
            factory = AdapterFactory(
                adapter_manager = self.adapter_manager,
                adaptee_class   = import_symbol(extension.adaptee_class_name),
                adapter_class   = import_symbol(extension.adapter_class_name),
                target_class    = import_symbol(extension.target_class_name)
            )
            
        return factory

    #### Methods ##############################################################

    def _get_class_name(self, klass):
        """ Returns the full class name for a class. """

        return "%s.%s" % (klass.__module__, klass.__name__)
            
#### EOF ######################################################################
