""" Base class for all object actions. """


# Standard library imports.
import inspect
import logging

# Enthought library imports.
from enthought.pyface.action.api import Action
from enthought.traits.api import Str


# Setup a logger for this module.
logger = logging.getLogger(__name__)


class ObjectAction(Action):
    """ Base class for all object actions.

    This class is designed such that implementors can easily override the
    the perform() method without losing the ability to execute the UOL
    configured action.

    """

    #### 'ObjectAction' interface #############################################

    # The universal object locator (UOL). This string is used to locate an
    # instance to invoke a method on.
    #
    # UOLs can currently have one of the following forms:
    #
    # * ``'service://a_service_identifier'``
    # * ``'name://a/path/through/the/naming/service'``
    # * ``'file://the/pathname/of/a/file/containing/a/UOL'``
    # * ``'http://a/URL/pointing/to/a/text/document/containing/a/UOL'``
    uol = Str

    # The name of the method to invoke on the object.
    method_name = Str

    ###########################################################################
    # 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Performs the action.

        This implementation simply performs the action specified by **uol**
        and **method_name**.

        Override this method to add additional work to the performance of this
        action.

        """

        self._perform_uol_action(event)

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _perform_uol_action(self, event):
        """ Called to perform the configured UOL action. """

        # Find the object.
        object = event.application.lookup_application_object(self.uol)
        if object is not None:
            method = getattr(object, self.method_name)

            # If the only argument is 'self' then don't pass the event. This
            # makes it easier to hook up actions that invoke NON-UI methods.
            #
            # fixme: Should we check the argument spec of the method more
            # closely and only pass the event iff there is exactly one argument
            # called 'event'?
            args, varargs, varkw, dflts = inspect.getargspec(method)
            if len(args) == 1:
                method()

            else:
                method(event)

        else:
            logger.error("Cannot resolve UOL: %s" % self.uol)

        return

#### EOF ######################################################################
