(* Copyright (C) 2003-2005, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://cs.unibo.it/helm/.
 *)

module C    = Cic
module Pp   = CicPp
module I    = CicInspect
module S    = CicSubstitution
module DTI  = DoubleTypeInference
module HEL  = HExtlib
module PEH  = ProofEngineHelpers
module TC   = CicTypeChecker 
module Un   = CicUniv

module H    = ProceduralHelpers
module Cl   = ProceduralClassify

(* term preprocessing: optomization 1 ***************************************)

let defined_premise = "DEFINED"

let get_type msg c bo =
try   
   let ty, _ = TC.type_of_aux' [] c bo Un.oblivion_ugraph in
   ty
with e -> failwith (msg ^ ": " ^ Printexc.to_string e)

let define c v =
   let name = C.Name defined_premise in
   let ty = get_type "define" c v in
   C.LetIn (name, v, ty, C.Rel 1)

let clear_absts m =
   let rec aux k n = function
      | C.Lambda (s, v, t) when k > 0 -> 
         C.Lambda (s, v, aux (pred k) n t)
      | C.Lambda (_, _, t) when n > 0 -> 
         aux 0 (pred n) (S.lift (-1) t)
      | t                  when n > 0 ->
         Printf.eprintf "CicPPP clear_absts: %u %s\n" n (Pp.ppterm t);
	 assert false 
      | t                                 -> t
   in 
   aux m

let rec add_abst k = function 
   | C.Lambda (s, v, t) when k > 0 -> C.Lambda (s, v, add_abst (pred k) t)
   | t when k > 0 -> assert false
   | t -> C.Lambda (C.Anonymous, C.Implicit None, S.lift 1 t)

let rec opt1_letin g es c name v w t =
   let name = H.mk_fresh_name c name in
   let entry = Some (name, C.Def (v, w)) in
   let g t =
      if DTI.does_not_occur 1 t then begin	    
         let x = S.lift (-1) t in
	 HLog.warn "Optimizer: remove 1"; opt1_proof g true c x 
      end else 
      let g = function
         | C.LetIn (nname, vv, ww, tt) when H.is_proof c v ->
	    let eentry = Some (nname, C.Def (vv, ww)) in
	    let ttw = get_type "opt1_letin 1" (eentry :: c) tt in
	    let x = C.LetIn (nname, vv, ww,
             C.LetIn (name, tt, ttw, S.lift_from 2 1 t)) in
	    HLog.warn "Optimizer: swap 1"; opt1_proof g true c x 
         | v when H.is_proof c v && H.is_atomic v     ->
	    let x = S.subst v t in
	    HLog.warn "Optimizer: remove 5"; opt1_proof g true c x 
	 | v                                           ->
	    g (C.LetIn (name, v, w, t))
      in
      if es then opt1_term g es c v else g v
   in
   if es then opt1_proof g es (entry :: c) t else g t

and opt1_lambda g es c name w t =
   let name = H.mk_fresh_name c name in
   let entry = Some (name, C.Decl w) in
   let g t = g (C.Lambda (name, w, t)) in
   if es then opt1_proof g es (entry :: c) t else g t

and opt1_appl g es c t vs =
   let g vs = 
      let g = function      
         | C.LetIn (mame, vv, tyty, tt) ->
            let vs = List.map (S.lift 1) vs in
	    let x = C.LetIn (mame, vv, tyty, C.Appl (tt :: vs)) in
	    HLog.warn "Optimizer: swap 2"; opt1_proof g true c x
         | C.Lambda (name, ww, tt) ->
	    let v, vs = List.hd vs, List.tl vs in
            let w = get_type "opt1_appl 1" c v in
	    let x = C.Appl (C.LetIn (name, v, w, tt) :: vs) in
	    HLog.warn "Optimizer: remove 2"; opt1_proof g true c x
	 | C.Appl vvs              ->
            let x = C.Appl (vvs @ vs) in
	    HLog.warn "Optimizer: nested application"; opt1_proof g true c x
	 | t                       ->
	    let rec aux d rvs = function
	       | [], _                   -> 
	          let x = C.Appl (t :: List.rev rvs) in
		  if d then opt1_proof g true c x else g x
	       | v :: vs, (cc, bb) :: cs ->
	          if H.is_not_atomic v && I.S.mem 0 cc && bb then begin 
		     HLog.warn "Optimizer: anticipate 1"; 
                     aux true (define c v :: rvs) (vs, cs)
		  end else 
		     aux d (v :: rvs) (vs, cs)
	       | _, []                   -> assert false
	    in
	    let h () =
	       let classes, conclusion = Cl.classify c (H.get_type c t) in
	       let csno, vsno = List.length classes, List.length vs in
	       if csno < vsno then
	          let vvs, vs = HEL.split_nth csno vs in
		  let x = C.Appl (define c (C.Appl (t :: vvs)) :: vs) in
	          HLog.warn "Optimizer: anticipate 2"; opt1_proof g true c x
	       else match conclusion, List.rev vs with
	          | Some _, rv :: rvs when csno = vsno && H.is_not_atomic rv ->
		     let x = C.Appl (t :: List.rev rvs @ [define c rv]) in
		     HLog.warn "Optimizer: anticipate 3"; opt1_proof g true c x
	          | _ (* Some _, _ *)                                             ->
		     g (C.Appl (t :: vs))
(*		  | None, _                                                ->
	             aux false [] (vs, classes)
*)	    in
	    let rec aux h prev = function
	       | C.LetIn (name, vv, tyty, tt) :: vs ->
	          let t = S.lift 1 t in
                  let prev = List.map (S.lift 1) prev in
                  let vs = List.map (S.lift 1) vs in
		  let y = C.Appl (t :: List.rev prev @ tt :: vs) in
                  let ww = get_type "opt1_appl 2" c vv in
		  let x = C.LetIn (name, vv, ww, y) in  
		  HLog.warn "Optimizer: swap 3"; opt1_proof g true c x
	       | v :: vs                      -> aux h (v :: prev) vs
	       | []                           -> h ()
	    in 
	    aux h [] vs
      in
      if es then opt1_proof g es c t else g t
   in
   if es then H.list_map_cps g (fun h -> opt1_term h es c) vs else g vs

and opt1_mutcase g es c uri tyno outty arg cases =
   let eliminator = H.get_default_eliminator c uri tyno outty in
   let lpsno, (_, _, _, constructors) = H.get_ind_type uri tyno in
   let ps, sort_disp = H.get_ind_parameters c arg in
   let lps, rps = HEL.split_nth lpsno ps in
   let rpsno = List.length rps in
   let predicate = clear_absts rpsno (1 - sort_disp) outty in   
   let is_recursive t =
      I.S.mem tyno (I.get_mutinds_of_uri uri t) 
   in
   let map2 case (_, cty) = 
      let map (h, case, k) (_, premise) = 
         if h > 0 then pred h, case, k else
	 if is_recursive premise then 
	    0, add_abst k case, k + 2 
	 else
	    0, case, succ k
      in
      let premises, _ = PEH.split_with_whd (c, cty) in
      let _, lifted_case, _ =
         List.fold_left map (lpsno, case, 1) (List.rev (List.tl premises))
      in
      lifted_case
   in
   let lifted_cases = List.map2 map2 cases constructors in
   let args = eliminator :: lps @ predicate :: lifted_cases @ rps @ [arg] in
   let x = H.refine c (C.Appl args) in
   HLog.warn "Optimizer: remove 3"; opt1_proof g es c x

and opt1_cast g es c t w =
   let g t = HLog.warn "Optimizer: remove 4"; g t in
   if es then  opt1_proof g es c t else g t

and opt1_other g es c t = g t 

and opt1_proof g es c = function 
   | C.LetIn (name, v, ty, t)   -> opt1_letin g es c name v ty t
   | C.Lambda (name, w, t)      -> opt1_lambda g es c name w t
   | C.Appl (t :: v :: vs)      -> opt1_appl g es c t (v :: vs)
   | C.Appl [t]                 -> opt1_proof g es c t
   | C.MutCase (u, n, t, v, ws) -> opt1_mutcase g es c u n t v ws
   | C.Cast (t, w)              -> opt1_cast g es c t w
   | t                          -> opt1_other g es c t

and opt1_term g es c t = 
   if H.is_proof c t then opt1_proof g es c t else g t

(* term preprocessing: optomization 2 ***************************************)

let expanded_premise = "EXPANDED"

let eta_expand g tys t =
   assert (tys <> []);
   let name i = Printf.sprintf "%s%u" expanded_premise i in 
   let lambda i ty t = C.Lambda (C.Name (name i), ty, t) in
   let arg i = C.Rel (succ i) in
   let rec aux i f a = function
      | []            -> f, a 
      | (_, ty) :: tl -> aux (succ i) (H.compose f (lambda i ty)) (arg i :: a) tl
   in
   let n = List.length tys in
   let absts, args = aux 0 H.identity [] tys in
   let t = match S.lift n t with
      | C.Appl ts -> C.Appl (ts @ args)
      | t         -> C.Appl (t :: args)
   in
   g (absts t)

let rec opt2_letin g c name v w t =
   let entry = Some (name, C.Def (v, w)) in
   let g t = 
      let g v = g (C.LetIn (name, v, w, t)) in
      opt2_term g c v
   in
   opt2_proof g (entry :: c) t

and opt2_lambda g c name w t =
   let entry = Some (name, C.Decl w) in
   let g t = g (C.Lambda (name, w, t)) in
   opt2_proof g (entry :: c) t

and opt2_appl g c t vs =
   let g vs =
      let x = C.Appl (t :: vs) in
      let vsno = List.length vs in
      let _, csno = PEH.split_with_whd (c, H.get_type c t) in
      if vsno < csno then 
         let tys, _ = PEH.split_with_whd (c, H.get_type c x) in
	 let tys = List.rev (List.tl tys) in
	 let tys, _ = HEL.split_nth (csno - vsno) tys in
         HLog.warn "Optimizer: eta 1"; eta_expand g tys x
      else g x 
   in
   H.list_map_cps g (fun h -> opt2_term h c) vs

and opt2_other g c t =
   let tys, csno = PEH.split_with_whd (c, H.get_type c t) in
   if csno > 0 then begin
      let tys = List.rev (List.tl tys) in      
      HLog.warn "Optimizer: eta 2"; eta_expand g tys t 
   end else g t

and opt2_proof g c = function 
   | C.LetIn (name, v, w, t) -> opt2_letin g c name v w t
   | C.Lambda (name, w, t)   -> opt2_lambda g c name w t
   | C.Appl (t :: vs)        -> opt2_appl g c t vs
   | t                       -> opt2_other g c t

and opt2_term g c t = 
   if H.is_proof c t then opt2_proof g c t else g t

(* object preprocessing *****************************************************)

let optimize_obj = function
   | C.Constant (name, Some bo, ty, pars, attrs) ->
      let bo, ty = H.cic_bc [] bo, H.cic_bc [] ty in 
      let g bo = 
         Printf.eprintf "Optimized : %s\nPost Nodes: %u\n" 
	    (Pp.ppterm bo) (I.count_nodes 0 bo);
	 let _ = H.get_type [] (C.Cast (bo, ty)) in
	 C.Constant (name, Some bo, ty, pars, attrs)
      in
      Printf.eprintf "BEGIN: %s\nPre Nodes : %u\n" 
         name (I.count_nodes 0 bo);
      begin try opt1_term g (* (opt2_term g []) *) true [] bo
      with e -> failwith ("PPP: " ^ Printexc.to_string e) end
   | obj                                         -> obj
