/* This file is part of the KDE project
   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   Original copied from showfoto:
    Copyright 2005 by Gilles Caulier <caulier.gilles@free.fr>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// QT includes.

#include <qlayout.h>
#include <qvbox.h>
#include <qvgroupbox.h>
#include <qhgroupbox.h>
#include <qlabel.h>
#include <qlineedit.h>

#include <qwhatsthis.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>

// sure
#include <qpushbutton.h>

// KDE includes.

#include <klocale.h>
#include <kdialog.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kuser.h>
#include <kapplication.h>
#include <klineedit.h>
#include <ktextedit.h>
#include <kmessagebox.h>
#include <kpassdlg.h>

#include <kemailsettings.h>
#include <kwallet.h>
using KWallet::Wallet;

//sure
#include <klistbox.h>
#include <ktabwidget.h>

// Local includes.
#include "global.h"
#include "setupidentities.h"

namespace Mailody {

NoDragKListBox::NoDragKListBox(QWidget* parent)
    : KListBox(parent) {}

void NoDragKListBox::mouseMoveEvent( QMouseEvent *e )
{   e->accept(); /* eat this */  }

void NoDragKListBox::keyPressEvent( QKeyEvent *e )
{   e->accept(); /* eat this */  }


SetupIdentities::SetupIdentities(QWidget* parent )
           : QWidget(parent),
           m_currentIdentity(0)
{
    QVBoxLayout *mainLayout = new QVBoxLayout(parent);
    QGridLayout* mainGrid = new QGridLayout(this,2,2,2,2);

    m_identybox = new NoDragKListBox(this);
    m_identybox->setAcceptDrops(false);
    connect(m_identybox, SIGNAL(selectionChanged(QListBoxItem*)),
            SLOT(slotSelectionChanged(QListBoxItem*)));

    mainGrid->addMultiCellWidget( m_identybox, 0, 0, 0, 1 );

    QPushButton* add = new QPushButton(i18n("+"), this);
    connect(add, SIGNAL(clicked()), SLOT(slotAdd()));
    mainGrid->addWidget( add, 1, 0 );

    QPushButton* del = new QPushButton(i18n("-"), this);
    connect(del, SIGNAL(clicked()), SLOT(slotDel()));
    mainGrid->addWidget( del, 1, 1 );

    m_tabWidget = new KTabWidget(this);

    mainGrid->addMultiCellWidget( m_tabWidget, 0, 1, 2, 2 );
    mainGrid->setRowStretch( 0, 0 );

    // -------------- page 1 ---------------------- //

    QWidget* page1 = new QWidget(this);
    QGridLayout* m1 = new QGridLayout(page1,3,2,7,7);

    QLabel *l0 = new QLabel(i18n("Identity name:") + ' ', page1 );
    l0->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    m1->addWidget(l0,0,0);
    QWhatsThis::add( l0, i18n("The display name") );
    m_identityName = new KLineEdit(page1);
    connect(m_identityName, SIGNAL(textChanged( const QString & )),
            SLOT( slotIdentityNameEdit( const QString & )));
    m1->addWidget(m_identityName,0,1);
    l0->setBuddy(m_identityName);

    QLabel *l1 = new QLabel(i18n("Full name:") + ' ', page1 );
    l1->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    m1->addWidget(l1,1,0);
    QWhatsThis::add( l1, i18n("The display name") );
    m_fullName = new KLineEdit(page1);
    m1->addWidget(m_fullName,1,1);
    l1->setBuddy(m_fullName);

    QLabel *l2 = new QLabel(i18n("Email address:") + ' ', page1 );
    l2->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    m1->addWidget(l2,2,0);
    QWhatsThis::add( l2, i18n("Your Emailadress") );
    m_emailAddress = new KLineEdit(page1);
    m1->addWidget(m_emailAddress,2,1);
    l2->setBuddy(m_emailAddress);

    QLabel *l3 = new QLabel(i18n("SMTP-server:") + ' ', page1 );
    m1->addWidget(l3,3,0);
    l3->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    QWhatsThis::add( l3, i18n("SSL is safe SMTP over port 465, TLS will "
                              "will operate on 25 and switch to a secure "
                              "connection, directly after connecting, None "
                              "will connect to 25 but will not switch to a "
                              "secure connection. This setting will probably "
                              "be needed for most providers.") );
    m_smtpServer = new KLineEdit(page1);
    m1->addWidget(m_smtpServer,3,1);
    l3->setBuddy(m_smtpServer);

    QLabel *l4 = new QLabel(i18n("Safety:") + ' ', page1);
    m1->addWidget(l4,4,0);
    l4->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    QWhatsThis::add( l4, i18n("Indicate how safe you want to send") );
    m_safeSMTP = new QButtonGroup(3, Qt::Horizontal, page1);
    m_safeSMTP->setExclusive(true);
    m_safeSMTP->setInsideMargin(3);
    new QRadioButton(i18n("None"), m_safeSMTP);
    QRadioButton* sslRadioS = new QRadioButton(i18n("SSL"), m_safeSMTP);
    QRadioButton* tlsRadioS = new QRadioButton(i18n("TLS"), m_safeSMTP);
    m1->addWidget(m_safeSMTP,4,1);
    l4->setBuddy(m_safeSMTP);

    m_smtpAuthBox = new QVGroupBox(page1);
    m_smtpAuthBox->setTitle(i18n("Use authentication"));
    m_smtpAuthBox->setCheckable( true );

    QHBox *s1 = new QHBox(m_smtpAuthBox);
    QLabel *l5 = new QLabel(i18n("Username:") + ' ', s1 );
    l5->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    QWhatsThis::add( l5, i18n("The username") );
    m_smtpUserName = new KLineEdit(s1);
    l5->setBuddy(m_smtpUserName);

    QHBox *s2 = new QHBox(m_smtpAuthBox);
    QLabel *l6 = new QLabel(i18n("Password:") + ' ', s2 );
    l6->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    QWhatsThis::add( l6, i18n("The password") );
    m_smtpPassword = new KPasswordEdit(s2);
    l6->setBuddy(m_smtpPassword);

    m1->addMultiCellWidget(m_smtpAuthBox,5,5,0,1);

    QLabel *htmlp = new QLabel(i18n("Include auto html part:") + ' ', page1 );
    m1->addWidget(htmlp,6,0);
    htmlp->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    QWhatsThis::add( htmlp, i18n("Mailody is able to convert your plain "
                                "message to a html message and include that "
                                "in the outgoing message, this means the "
                                "receiver will also have clickable links and "
                                "colored quote levels") );
    m_htmlPart = new QButtonGroup(2, Qt::Horizontal, page1);
    m_htmlPart->setExclusive(true);
    m_htmlPart->setInsideMargin(3);
    new QRadioButton(i18n("No"), m_htmlPart);
    new QRadioButton(i18n("Yes"), m_htmlPart);
    m1->addWidget(m_htmlPart,6,1);
    htmlp->setBuddy(m_htmlPart);

    QSpacerItem* spacer1 = new QSpacerItem( 20, 20, QSizePolicy::Minimum,
                                          QSizePolicy::Expanding );
    m1->addItem( spacer1, 7, 1 );

    m_tabWidget->addTab(page1,i18n("General"));

    // ------------------- page 2 -----------------------//
    QWidget* page2 = new QWidget(this);
    QGridLayout* m2 = new QGridLayout(page2,1,1,5,5);

    m_sigBox = new QGroupBox(page2);
    m_sigBox->setCheckable(true);
    m_sigBox->setTitle(i18n("Use Signature"));
    m_sigBox->setColumnLayout(0, Qt::Vertical);
    m_sigBox->setInsideMargin(3);
    m_sigBox->setInsideSpacing(3);
    m_sigBox->setFlat(true);



    QVBoxLayout* f2 = new QVBoxLayout(m_sigBox->layout());
    f2->setAlignment( Qt::AlignTop );
    m_signature = new KTextEdit(m_sigBox);
    QWhatsThis::add( m_signature, i18n("This is the mail signature."));

    m_sigPos = new QButtonGroup(2, Qt::Horizontal, i18n("Position"), m_sigBox);
    m_sigPos->setExclusive(true);
    m_sigPos->setFlat(true);

    new QRadioButton(i18n("Bottom"), m_sigPos);
    new QRadioButton(i18n("Top"), m_sigPos);
    QWhatsThis::add( m_sigPos, i18n("Select the placement of the signature in "
                                    "the composer. Top means above the text "
                                    "in the message you reply to or forward. "
                                    "Bottom means below the text in the "
                                    "message you reply to or forward."));

    m_tabWidget->addTab(page2,i18n("Signature"));
    m2->addWidget(m_sigBox, 0, 0);
    f2->addWidget(m_signature);
    f2->addWidget(m_sigPos);

    //---------------------------------------------------//


    mainLayout->addWidget(this);

    m_wallet = Wallet::openWallet(Wallet::NetworkWallet(), this->winId());
    if (!m_wallet)
    {
        m_smtpPassword->setEnabled(false);
    }
    else if (m_wallet && m_wallet->isOpen())
    {
        if (!m_wallet->hasFolder("mailody"))
            m_wallet->createFolder( "mailody" );

        m_wallet->setFolder( "mailody" );
    }

    //------------------------------------------------/

    readSettings();

    if(!Global::cryptoConnectionSupported())
    {
        tlsRadioS->setEnabled(false);
        sslRadioS->setEnabled(false);
        m_safeSMTP->setButton(0);
    }

    m_tabWidget->setEnabled(false);
}

SetupIdentities::~SetupIdentities()
{
}

void SetupIdentities::slotAdd()
{
    // kdDebug() << k_funcinfo << endl;

    int i = 0;
    QString name;

    while (true)
    {
        ++i;
        name = i18n("Identity #%1").arg(i);
        QListBoxItem* find = m_identybox->findItem(name);
        if (!find)
            break;
    }

    // Create a new identity based on KDE settings.
    KUser* currentUser = new KUser();
    KEMailSettings esetting;

    QString server = esetting.getSetting(KEMailSettings::OutServer);
    if (server.isEmpty() && m_identityList.count() > 0)
        server = (m_identityList.first())->smtpServer();

    QString email = esetting.getSetting(KEMailSettings::EmailAddress);
    if (email.isEmpty() && m_identityList.count() > 0)
        email = (m_identityList.first())->emailAddress();

    Identity* id = new Identity(
                                name,
                                server,
                                currentUser->fullName(),
                                email,
                                currentUser->loginName(),
                                QString::null,
                                0,
                                false,
                                false,
                                QString::null,
                                0,
                                0
                              );

    m_identityList.append(id);
    m_identybox->setSelected( new QListBoxText(m_identybox, name), true );
    m_identityName->setFocus();
    m_identityName->selectAll();
}

void SetupIdentities::slotDel()
{
    QListBoxItem* item = m_identybox->selectedItem();
    if (!item)
        return;

    delete item;

    if (!m_currentIdentity)
        return;

    m_identityList.remove(m_currentIdentity);

    delete m_currentIdentity;
    m_currentIdentity=0;

    m_tabWidget->setEnabled(false);
}

void SetupIdentities::slotSelectionChanged(QListBoxItem*)
{
    // kdDebug() << k_funcinfo << endl;

    // Save current settings, but only when widget is enabled, deleting
    // an identity disables the widget, but does not clear them...
    if (m_tabWidget->isEnabled())
        saveCurrent();

    m_tabWidget->setEnabled(true);

    // Now handle what the user wants. find the identity clicked on and
    // display the values.

    if (!m_identityList.count() || !m_identybox->selectedItem() ||
         m_identybox->currentText().isEmpty())
        return;

    QValueList<Identity*>::iterator it;
    for ( it = m_identityList.begin(); it != m_identityList.end(); ++it )
    {
        if (!(*it) || (*it)->identityName() != m_identybox->currentText())
            continue;

        // we found it. Now set all the fields.
        m_currentIdentity = (*it);
        m_identityName->setText(m_currentIdentity->identityName());
        m_smtpServer->setText(m_currentIdentity->smtpServer());
        m_fullName->setText(m_currentIdentity->fullName());
        m_emailAddress->setText(m_currentIdentity->emailAddress());
        m_smtpUserName->setText(m_currentIdentity->smtpUserName());
        m_smtpPassword->erase();
        m_smtpPassword->insert(m_currentIdentity->smtpPassword());
        m_safeSMTP->setButton(m_currentIdentity->safeSMTP());
        m_smtpAuthBox->setChecked(m_currentIdentity->smtpNeedAuth());
        m_sigBox->setChecked(m_currentIdentity->sigEnabled());
        m_signature->setText(m_currentIdentity->signature());
        m_sigPos->setButton(m_currentIdentity->sigPos());
        m_htmlPart->setButton(m_currentIdentity->htmlPart());
    }
}

void SetupIdentities::saveCurrent()
{
    // Delete the current identity object.
    if (m_currentIdentity)
    {
        m_identityList.remove(m_currentIdentity);
        delete m_currentIdentity;
        m_currentIdentity = 0;
    }

    // Create a new identity object, based on current values.
    if (!m_identityName->text().isEmpty())
    {
        Identity* i = new Identity(
                                    m_identityName->text(),
                                    m_smtpServer->text(),
                                    m_fullName->text(),
                                    m_emailAddress->text(),
                                    m_smtpUserName->text(),
                                    m_smtpPassword->password(),
                                    m_safeSMTP->selectedId(),
                                    m_smtpAuthBox->isChecked(),
                                    m_sigBox->isChecked(),
                                    m_signature->text(),
                                    m_sigPos->selectedId(),
                                    m_htmlPart->selectedId()
                                  );
        m_identityList.append(i);

    }
}

void SetupIdentities::slotIdentityNameEdit( const QString& textt )
{
    // kdDebug() << k_funcinfo << endl;

    QListBoxItem* item = m_identybox->item(m_identybox->currentItem());
    if (!item)
        return;

    // dont emit a selectionchanged!
    m_identybox->blockSignals(true);
    m_identybox->changeItem(textt, m_identybox->index(item));
    m_identybox->blockSignals(false);
}

void SetupIdentities::applySettings()
{
    // kdDebug() << k_funcinfo << endl;

    // Save current settings, but only when widget is enabled, deleting
    // an identity disables the widget, but does not clear them...
    if (m_tabWidget->isEnabled())
        saveCurrent();

    /**
     * iterate through the identity objects and write out all identities
     * delete all other existing identities after the correct ones are
     * written.
     */

    KConfig* config = kapp->config();
    QStringList existingGroups = Global::identities(true);

    QValueList<Identity*>::iterator it =  m_identityList.begin();
    while ( it != m_identityList.end() )
    {
        existingGroups.remove((*it)->identityName());
        config->setGroup("identity_" + (*it)->identityName() );
        config->writeEntry("smtpServer", (*it)->smtpServer() );
        config->writeEntry("fullName", (*it)->fullName());
        config->writeEntry("emailAddress", (*it)->emailAddress() );
        config->writeEntry("smtpUserName", (*it)->smtpUserName() );
        config->writeEntry("safeSMTP", (*it)->safeSMTP() );
        config->writeEntry("smtpNeedAuth", (*it)->smtpNeedAuth() );
        config->writeEntry("sigEnabled", (*it)->sigEnabled() );
        config->writeEntry("signature", (*it)->signature() );
        config->writeEntry("sigPos", (*it)->sigPos() );
        config->writeEntry("htmlPart", (*it)->htmlPart() );
        if (m_wallet)
            m_wallet->writePassword("smtpidentity_" + (*it)->identityName(),
                                    (*it)->smtpPassword());
        it = m_identityList.remove( it );
    }

    QStringList::iterator its =  existingGroups.begin();
    while ( its != existingGroups.end() )
    {
        config->deleteGroup("identity_" + (*its));
        ++its;
    }

    config->sync();
}

void SetupIdentities::readSettings()
{
    // kdDebug() << k_funcinfo << endl;

    /**
     * All identities in the config file have a identity_ prefix.
     * So here we iterate through them and create objects for each identity
     *
     * general is used mailody <= 0.2.0
     */

    KConfig* config = kapp->config();
    QStringList groups = config->groupList();
    for ( QStringList::Iterator it = groups.begin(); it != groups.end(); ++it )
    {
        if (!(*it).startsWith("identity_"))
            continue;

        config->setGroup(*it);

        QString pass;
        if (m_wallet)
            m_wallet->readPassword("smtp"+ (*it), pass);

        QString name = (*it).mid(9);
        Identity* i = new Identity(
                                    name,
                                    config->readEntry("smtpServer"),
                                    config->readEntry("fullName"),
                                    config->readEntry("emailAddress"),
                                    config->readEntry("smtpUserName"),
                                    pass,
                                    config->readNumEntry("safeSMTP"),
                                    config->readBoolEntry("smtpNeedAuth"),
                                    config->readBoolEntry("sigEnabled"),
                                    config->readEntry("signature"),
                                    config->readNumEntry("sigPos"),
                                    config->readNumEntry("htmlPart")
                                  );

        m_identityList.append(i);
        new QListBoxText(m_identybox, i->identityName());
    }
}

}

#include "setupidentities.moc"
