/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/


/*! \file SpotDecoder.h
    \brief Implementation of the Template class SpotDecoder.
    
    Magics Team - ECMWF 2005
    
    Started: Mon 19-Sep-2005
    
    Changes:
    
*/
 

#include <locale>
#include "WrepJSon.h"
#include "MetaData.h"
#include "CustomisedPoint.h"
#include "DateTime.h"
#include "TextVisitor.h"
#include "json_spirit.h"
#include "IntervalMap.h"
#include "EfiLegendEntry.h"

using namespace magics;
using namespace json_spirit;


WrepJSon::WrepJSon() 
{
	methods_["date"] = &WrepJSon::date;
	methods_["time"] = &WrepJSon::time;
	methods_["epsz"] = &WrepJSon::epsz;
	methods_["detz"] = &WrepJSon::detz;
	methods_["tracker"] = &WrepJSon::ignore;
	methods_["location"] = &WrepJSon::location;
	methods_["location"] = &WrepJSon::location;
	
	decoders_["eps"] = &WrepJSon::eps;
	decoders_["efi"] = &WrepJSon::efi;
	decoders_["cdf"] = &WrepJSon::cdf;
	decoders_["basic"] = &WrepJSon::basic;
	
	transformationHandlers_["eps"] = &WrepJSon::eps;
	transformationHandlers_["cdf"] = &WrepJSon::cdf;
	transformationHandlers_["efi"] = &WrepJSon::efi;
	
	minx_ = std::numeric_limits<double>::max();
	maxx_ = -std::numeric_limits<double>::max();
	miny_ = std::numeric_limits<double>::max();
	maxy_ = -std::numeric_limits<double>::max();
}

WrepJSon::~WrepJSon()
{
	
}
void WrepJSon::visit(Transformation& transformation)
{
	decode();
	map<string,  TransformationHandler>::iterator handler = transformationHandlers_.find(family_);

	if ( handler !=  transformationHandlers_.end() ) {
			       (this->*handler->second)(transformation);
	}  		
	else {
		transformation.setDataMinX(minx_);
		transformation.setDataMaxX(maxx_);
		transformation.setDataMinY(miny_);
		transformation.setDataMaxY(maxy_);
	}
}

    
void WrepJSon::eps(Transformation& transformation)
{
	transformation.setDataMinX(minx_  * 3600, base_);
	transformation.setDataMaxX(maxx_ * 3600, base_);
	transformation.setDataMinY(miny_);
	transformation.setDataMaxY(maxy_);
}
void WrepJSon::cdf(Transformation& transformation)
{
	transformation.setDataMinX(minx_);
	transformation.setDataMaxX(maxx_);
	transformation.setDataMinY(0);
	transformation.setDataMaxY(100);
}
void WrepJSon::efi(Transformation& transformation)
{
	transformation.setDataMinX(-100);
	transformation.setDataMaxX(+100);
	transformation.setDataMinY(miny_);
	transformation.setDataMaxY(maxy_);
}
void WrepJSon::decode()
{	
	  map<string,  Decoder>::iterator decoder = decoders_.find(family_);

	  if ( decoder != decoders_.end() ) {
		       (this->*decoder->second)();
		        		    	    }  		
	  else 
		  basic();
	 
}
void WrepJSon::eps()
{
	if ( !points_.empty()) return;
	
	shift_ = 12;
	methods_[param_] = &WrepJSon::parameter;
	methods_[keyword_] = &WrepJSon::dig;
	     
    scaling_factor_ = param_scaling_factor_;
    offset_factor_ = param_offset_factor_;    
    file_ = path_;
    current_ = &values_;
    
    basic();

    
   
    base_ = DateTime(date_, time_);
   
    if ( values_.steps_.empty() ) {
    	 Log::fatal() << "Could not find data for parameter: " << param_ << endl;
    	 abort();
    } 
    minx_ = 0;
    maxx_ = values_.steps_.back()+shift_;
    Log::dev() << "minx= " <<  minx_ << "->maxx= " << maxx_ << endl;
	for (unsigned int i = 0; i < values_.steps_.size(); i++) {
		CustomisedPoint* point = new CustomisedPoint();		
		point->longitude(values_.steps_[i] * 3600);		 
		point->latitude(values_.steps_[i] * 3600);		 
		(*point)["step"]    = values_.steps_[i] * 3600;
		(*point)["shift"] = 0;
		(*point)["width"]    = 1 * 3600;
		point->base(base_);
		map<string, vector<double>  >& values = values_.values_;
		
		for ( map<string, vector<double>  >::iterator val = values.begin(); val != values.end(); ++val ) 
			(*point)[val->first] = (val->second)[i];
	
		points_.push_back(point);
	}
	
}

void WrepJSon::customisedPoints(const std::set<string>&, CustomisedPointsList& out)
{
	Log::dev() << "WrepJSon::customisedPoints" <<endl;
	
	decode();

	for (vector<CustomisedPoint*>::const_iterator point = points_.begin(); point != points_.end(); ++point) {
			out.push_back(*point);	
			cout << **point << endl;
			
	}
}


void WrepJSon::basic()
{
	  ifstream is( file_.c_str());
	    	 
	     json_spirit::Value value;
	         try {
	        	  json_spirit::read_or_throw( is, value );
	        	  Object object = value.get_value< Object >();
	        		        
	        	  for (vector<Pair>::const_iterator entry = object.begin(); entry !=  object.end(); ++entry) {
	        		  map<string,  Method >::iterator method = methods_.find(entry->name_);
	        		    	    if ( method != methods_.end() ) {
	        		    	    	   ( (this->*method->second)(entry->value_) );
	        		    	    }  		
	        			    		
	        	  }
	         }
	         catch (exception e) 
	         {
	        	 Log::fatal() << "Could not processed the file: " << file_ << ": " << e.what() << endl;
	        	 abort();
	         }	
}
void WrepJSon::print(ostream& out) const
{
	out << "WrepJSon[";
	out << "]";
}





void WrepJSon::location(const json_spirit::Value& value )
{
	assert( value.type() == array_type );
	Array location = value.get_value< Array>();
	
	assert(location.size() == 2); 
	
	latitude_=location[0].get_value<double>();
	longitude_=location[1].get_value<double>();
	
	Log::dev() << "found -> lat= " << location[0].get_value<double>() << endl;
	Log::dev() << "found -> lon= " << location[1].get_value<double>() << endl;
}
void WrepJSon::epsz(const json_spirit::Value& value)
{
	assert( value.type() == real_type);
	Log::dev() << "found -> epsz= " <<  value.get_value< double>() << endl;
}

void WrepJSon::detz(const json_spirit::Value& value )
{
	assert( value.type() == real_type);
	Log::dev() << "found -> detz= " <<  value.get_value< double>() << endl;
}
void WrepJSon::date(const json_spirit::Value& value)
{
	assert( value.type() == str_type);
	Log::dev() << "found -> date= " <<  value.get_value<string>() << endl;
	date_ =  value.get_value<string>();
	
}
void WrepJSon::time(const json_spirit::Value& value)
{
	assert( value.type() == str_type);
	Log::dev() << "found -> time= " <<  value.get_value<string>() << endl;
	time_ =  value.get_value<string>();
}

void WrepJSon::parameter(const json_spirit::Value& value)
{
	assert( value.type() == obj_type );
	Object param = value.get_value< Object >();
	        		        
	for (vector<Pair>::const_iterator info = param.begin(); info !=  param.end(); ++info) {
				assert (info->value_.type() == array_type);
	        	Array values = info->value_.get_value<Array>();
	        	if ( info->name_ == "steps" ) {
	        		for (unsigned int i = 0; i < values.size(); i++) {
	        				current_->steps_.push_back( tonumber(values[i].get_value<string>()));
	        			  
	        	     }
	        	}
	        	else {
	        		map<string, vector<double> >& xv = current_->values_;
	        		xv.insert(make_pair(info->name_, vector<double>()));
	        		vector<double>& vals =  xv[info->name_];
	        		for (unsigned int i = 0; i < values.size(); i++) {
	        			double val = values[i].get_value<double>();
	        			val = (val * scaling_factor_) + offset_factor_;
	        			if ( val < miny_) miny_ = val;
	        			if ( val > maxy_) maxy_ = val;
	        			vals.push_back(val);
	        			
	        			
	        			
	        		}
	        	}
	}
}
void WrepJSon::dig(const json_spirit::Value& value)
{
	assert( value.type() == obj_type );
	Object object = value.get_value< Object >();
	  for (vector<Pair>::const_iterator entry = object.begin(); entry !=  object.end(); ++entry) {
      		  map<string,  Method >::iterator method = methods_.find(entry->name_);
      		    	    if ( method != methods_.end() ) {
      		    	    	   ( (this->*method->second)(entry->value_) );
      		    	    }  		
      			    		
      	  }       		        
	
}

MatrixHandler<UserPoint>& WrepJSon::matrix()
{
	decode();
	vector<double> values;
	vector<double> steps;
	// Find the min and the max ..
	for (vector<CustomisedPoint*>::const_iterator point = points_.begin(); point != points_.end(); ++point)
	{
		 Log::dev() << **point << endl;
		 Log::dev() << **point << endl;
		 for ( int s = 0; s != 50; s++) {		    
				string key = tostring(s);
			    map<string, double>::const_iterator member = (*point)->find(key);
				if ( member != (*point)->end() ) 
					values.push_back(member->second );
		 }

		 steps.push_back((**point)["step"]);
	}


	double from = maground(*min_element(values.begin(), values.end())) - 1;
	double to = maground(*max_element(values.begin(), values.end())) + 1 ;
	Log::dev() << "min->" << *min_element(values.begin(), values.end()) << " = " << from << endl;
	Log::dev() << "max->" << *max_element(values.begin(), values.end()) << " = " << to << endl;

	IntervalMap<int> array;
	//double step = parameter_->plumesInterval();
	double step= plumes_;
	for ( double a = from; a <= to; a = a + step ) {
			array.insert(make_pair(Interval(a, a+step), 0));
	}


	cout << "dim-->" << points_.size() << array.size() << endl; 
	matrix_.set(array.size(), points_.size());
	for (IntervalMap<int>::iterator interval = array.begin(); interval!= array.end(); ++interval) {
		double row = (interval->first.max_ - interval->first.min_)/2 +interval->first.min_ ;
		cout << "row-> " << row << endl;
		matrix_.rowsAxis().push_back(row);
	}
	for (vector<double>::iterator s = steps.begin(); s != steps.end(); ++s) {				
		matrix_.columnsAxis().push_back(*s);
	}

	matrix_.setMapsAxis();
	int column = 0;

	for (vector<CustomisedPoint*>::const_iterator point = points_.begin(); point != points_.end(); ++point)
	{

		for (IntervalMap<int>::iterator interval = array.begin(); interval!= array.end(); ++interval) {
			interval->second = 0;
		}
		for ( int s = 0; s != 50; s++) {		    
			string key = tostring(s);

			map<string, double>::const_iterator step = (*point)->find(key);
			if (step != (*point)->end() ) {

			  IntervalMap<int>::iterator interval = array.get(step->second);
			  interval->second++;
		     }

			}
		int row = 0;
		cout << "Array ->" << array.size() << "row-->" << row  << endl;
		for (IntervalMap<int>::iterator interval = array.begin(); interval!= array.end(); ++interval) {
			cout << "index = " << column+row*steps.size() << interval->second << endl;
			matrix_[column+row*steps.size()] = interval->second*2; // in percentage!
			row++;
		}
	    column++;
	}


	
	matrixHandlers_.push_back(new  MatrixHandler<UserPoint>(matrix_));
	return *(matrixHandlers_.back());
}

void WrepJSon::cdf() 
{
	if ( !clim_.empty() ) return;
	
	
	 file_ = path_;
	 methods_["clim"] =  &WrepJSon::clim;
	 methods_["eps"] = &WrepJSon::eps;
	 methods_["efi"] = &WrepJSon::efi;
	 methods_[param_] =  &WrepJSon::parameter;
	 scaling_factor_ = param_scaling_factor_;
	 	offset_factor_ = param_offset_factor_;
	 
	    
	basic();
	// setminmax ...
	minx_ = std::numeric_limits<double>::max();
		maxx_ = -std::numeric_limits<double>::max();
		for (map<string,  InputData>::iterator info = eps_.begin(); info !=  eps_.end(); ++info) {				
							
							// find the index in the steps..
							InputData& data = info->second;
							int index = data.index(tonumber(info->first));
							map<string, vector<double>  >& values = data.values_;					
						
							for (map<string,  vector<double> >::iterator val = values.begin(); val !=  values.end(); ++val) {			
								if ( minx_ > val->second[index] ) minx_ = val->second[index];
								if ( maxx_ < val->second[index] ) maxx_ = val->second[index];
									
							}
							
			}
			int index = clim_.index(36);
			for (map<string,  vector<double> >::iterator val = clim_.values_.begin(); val !=  clim_.values_.end(); ++val) {	
								
									if ( minx_ > val->second[index] ) minx_ = val->second[index];
									if ( maxx_ < val->second[index] ) maxx_ = val->second[index];
			}
			

			int steps = 1;
			CustomisedPoint* point = new CustomisedPoint();		
			points_.push_back(point);
			
			for ( vector<int>::iterator step = steps_.begin(); step != steps_.end(); ++step) {
				map<string,  InputData>::iterator info = eps_.find(tostring(*step));
				if ( info != eps_.end() ) {
						
							// find the index in the steps..
							InputData& data = info->second;
							int s = tonumber(info->first);
							int index = data.index(tonumber(info->first));
							map<string, vector<double>  >& values = data.values_;		
							ostringstream key;			
							key << steps << "_step";
							(*point)[key.str()] = s +12;
							for (map<string,  vector<double> >::iterator val = values.begin(); val !=  values.end(); ++val) {			
									ostringstream key;
									int i = 2*tonumber(val->first);
									key << steps << "_" << i;
									(*point)[key.str()] = val->second[index];			
							}
							steps++;
				}
			}
			

			
			index = clim_.index(36);
			for (map<string,  vector<double> >::iterator val = clim_.values_.begin(); val !=  clim_.values_.end(); ++val) {		
									ostringstream key;									
									key << "clim_" << val->first;
									(*point)[key.str()] = val->second[index];			
			}
}
void WrepJSon::efi()
{
	if ( !efi_.empty() ) return;
	
	
	 file_ = path_;
	 methods_["clim"] =  &WrepJSon::ignore;
	 methods_["eps"] = &WrepJSon::ignore;
	 methods_["efi"] = &WrepJSon::efi;
	 methods_[param_] =  &WrepJSon::parameter;

	basic();
	CustomisedPoint* point = new CustomisedPoint();		
				points_.push_back(point);

			int steps = 1;
			

			(*point)["steps"] = steps_.size();
			
			miny_ = steps_.size()+1;
			maxy_ =  1;
			
			for ( vector<int>::iterator step = steps_.begin(); step != steps_.end(); ++step) {
					int x = *step +12;
					map<string,  InputData>::iterator info = efi_.find(tostring(x));
					if ( info != efi_.end() ) {
								cout << "efi->" << info->first << endl;
								// find the index in the steps..
								InputData& data = info->second;						
								int index = data.index(tonumber(info->first));
								map<string, vector<double>  >& values = data.values_;		
								ostringstream key;			
								key << "efi" << steps << "_step";
								(*point)[key.str()] = (tonumber(info->first)) +12;
								for (map<string,  vector<double> >::iterator val = values.begin(); val !=  values.end(); ++val) {			
										ostringstream key;
										key << "efi" << steps << "_value";  
										(*point)[key.str()] = val->second[index];		
										cout << "EFI-->INFO--> " <<  key.str() << " --> " << val->second[index] << endl;
								}
								steps++;
					}
				}
			
}

void WrepJSon::clim(const json_spirit::Value& value)
{
	
	current_ = &clim_;
	scaling_factor_ = param_scaling_factor_;
	offset_factor_ = param_offset_factor_;
	dig(value);
	clim_.print();

}
void WrepJSon::efi(const json_spirit::Value& value)
{
	cout << " Youpi found efi" << endl;
	assert( value.type() == obj_type );
	Object param = value.get_value< Object >();
	scaling_factor_ = 100;
	offset_factor_ = 0;
	        		        
	for (vector<Pair>::const_iterator info = param.begin(); info !=  param.end(); ++info) {				
	        	cout << info->name_ << " in efi " << endl;
	        	efi_.insert(make_pair(info->name_, InputData()));
	        	current_ = &(efi_[info->name_]);
	        	dig(info->value_);   
	  }
	        
	for (map<string,  InputData>::iterator info = efi_.begin(); info !=  efi_.end(); ++info) {				
		cout << "Efi->" << info->first;
		info->second.print();
	}
	        
	
}

void WrepJSon::eps(const json_spirit::Value& value)
{
	cout << " Youpi found eps" << endl;
	scaling_factor_ = param_scaling_factor_;
		offset_factor_ = param_offset_factor_;
		assert( value.type() == obj_type );
		Object param = value.get_value< Object >();
		        		        
		for (vector<Pair>::const_iterator info = param.begin(); info !=  param.end(); ++info) {				
			        	
			        	eps_.insert(make_pair(info->name_, InputData()));
			        	current_ = &(eps_[info->name_]);
			        	dig(info->value_);   
			  }
			        
			for (map<string,  InputData>::iterator info = eps_.begin(); info !=  eps_.end(); ++info) {				
				cout << "eps->" << info->first;
				info->second.print();
			}
}
	 

