/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file MgQPlotScene.cc
    \brief Definition of MgQPlotScene
    \author Graphics Section, ECMWF

    Started: March 2010
*/


#include "MgQPlotScene.h"

#include <QDebug>
#include <QGraphicsItem>
#include <QImage>
#include <QPixmap>
#include <QPainter>
#include <QProgressDialog>
#include <QStyleOptionGraphicsItem>

#include "MgQDriverObject.h"
#include "MgQHistoNode.h"
#include "MgQLayerNode.h"
#include "MgQLayoutNode.h"

MgQStepMetaData::MgQStepMetaData(QStringList keys) : keys_(keys)
{
	for(int i=0; i < keys_.count(); i++)
	{
		stepData_[i]=QStringList();
	}
}

void MgQStepMetaData::clear()
{
	keys_.clear();
	stepData_.clear();
}

int MgQStepMetaData::stepNum()
{
	if(keys_.count() > 0)
		return stepData_[0].count();
	else
		return 0;
}

QStringList MgQStepMetaData::stepData(int index)
{
	return stepData_[index];
}

void MgQStepMetaData::addStep(QString value)
{
	foreach(int index,stepData_.keys())
	{
		stepData_[index].append(value);
	}
}

void MgQStepMetaData::appendToStepData(QString key, int step, QString value)
{	
	int index=keys_.indexOf(key,0);
	while(index != -1)
	{
		QStringList& lst=stepData_[index];
		//qDebug() << key << index << step << lst;
		lst[step].append(value);

		index=keys_.indexOf(key,index+1);
	}
} 

MgQPlotScene::MgQPlotScene(QObject *parent) : MgQScene(parent)
{
	prevCurrentStep_=0;
	driverObject_=0;

	sceneLayerNode_=0;
	previewLayoutNode_=0;

	cacheDevice_=new QPixmap(900,700);
	cachePainter_=new QPainter(cacheDevice_);
	cacheDevice_->fill(qRgba(255,255,255,255));

	antialias_=false;
	cachePainter_->setRenderHint(QPainter::Antialiasing,antialias_);

	stepNum_=0;
}
 
MgQPlotScene::~MgQPlotScene()
{
	if(driverObject_)
		delete driverObject_;

	foreach(MgQLayerState *st,previousLayerState_)
	{
		delete st;
	}	
}

void MgQPlotScene::clearBeforeNewRequest()
{
	if(rootNode_)
	{
		removeItem(rootNode_);
		delete rootNode_;
		rootNode_=0;
	}

	//Animation
	prevCurrentStep_=currentStep_;


	layerNodes_.clear();
	sceneLayerNode_=0;
	previewLayoutNode_=0;
	stepNum_=0;
}

void MgQPlotScene::saveStateBeforeNewRequest()
{
	foreach(MgQLayerState *st,previousLayerState_)
	{
		delete st;
	}	
	previousLayerState_.clear();

	foreach(MgQLayerNode *node,layerNodes_)
	{
		MgQLayerState *st=new MgQLayerState;
		node->saveLayerState(st);
		previousLayerState_.push_back(st);		
	}

}

MgQLayoutNode* MgQPlotScene::projectorNode(QPointF pos)
{
	//QGraphicsItem* out=0;

	if(sceneLayerNode_)
	{
		MgQLayoutNode *item=sceneLayerNode_->projectorNode();

		if(item->contains(item->mapFromScene(pos)))
		{			
			//qDebug() << "children:" << item->childItems().size()  << layout->isVisible();
			return item;
		}	
	}
	
	return 0;

	/*foreach(QGraphicsItem *item, items())
	{
		if(item->data(0).toString() == "Layout" && item->data(1).toString() == "Zoomable")
		{
			//MgQLayoutItem *layout=static_cast<MgQLayoutItem*>(item);
			//if(layout->layout().isZoomable()== true &&
			qDebug() << pos;
			qDebug() << item->mapFromScene(pos);
			if(item->contains(item->mapFromScene(pos)))
			{
				MgQLayoutNode *layout=static_cast<MgQLayoutNode*>(item);
				qDebug() << "children:" << layout->childItems().size()  << layout->isVisible();
				return layout;
			}
		}
	}

	return 0;*/		
}

MgQLayoutNode* MgQPlotScene::projectorNode()
{
	if(sceneLayerNode_)
	{
		return sceneLayerNode_->projectorNode();
	}

	return 0;

	/*
	foreach(QGraphicsItem *item, items())
	{
		if(item->data(0).toString() == "Layout" && item->data(1).toString() == "Zoomable")
		{
			
			MgQLayoutNode *layout=static_cast<MgQLayoutNode*>(item);
			return layout;
		}
	}

	return 0;*/		
}


MgQLayoutNode* MgQPlotScene::findPreviewLayout()
{
	return 0;

	/*foreach(QGraphicsItem *item, items())
	{
		if(item->data(MgQ::NodeType).toInt() == MgQ::PreviewLayoutNode)
		{
			
			MgQLayoutNode *layout=static_cast<MgQLayoutNode*>(item);
			return layout;
		}
	}

	return 0;*/		
}

MgQLayoutNode* MgQPlotScene::findMagnifierLayout()
{
	foreach(QGraphicsItem *item, items())
	{
		if(checkNodeType(item,MgQ::MagnifierLayoutNode))
		{			
			MgQLayoutNode *layout=static_cast<MgQLayoutNode*>(item);
			return layout;
		}
	}

	return 0;		
}


//-------------------------------------
// Animation
//-------------------------------------

void MgQPlotScene::updateAnimation()
{
	//Regenarete the cache
	updateCache();
	
	//Repaint everything
	update();
}

void MgQPlotScene::selectCurrentStepForAnimation()
{
	setCurrentStep(prevCurrentStep_);
}

int MgQPlotScene::currentStep()
{
	return currentStep_;
}

int MgQPlotScene::stepNum()
{
	return stepNum_;
}

void MgQPlotScene::setCurrentStep(int step)
{
	currentStep_=step;

	if(currentStep_< 0 || currentStep_ >= stepNum_)
		return;

	for(int i=0; i < stepNum_; i++)
	{
		if(i!= currentStep_)
		{
			setStepVisible(i,false);
		}
		else
		{
			setStepVisible(i,true);	
			updateCache();
		}
	}

        updateAnimation();
}

void MgQPlotScene::setStepVisible(int step,bool visible)
{
	if(visible == true && stepCached(step) ==false)
	{
		//Notify scene node about the new step
		//sceneLayerNode_->sceneLayer().reset();
		sceneLayerNode_->sceneLayer().getReady(step);
		foreach(MgQLayerNode *node,layerNodes_)
		{
			if(node->stepNum() > 0 && node->stepCached(step) == false)
			{
				driverObject_->driver().executeStep(step,node);
				//node->setStepVisible(step,visible);
			}
		}
		
	}

	foreach(MgQLayerNode *node,layerNodes_)
	{
		if(node->stepNum() > 0)
		{
			node->setStepVisible(step,visible);
		}
	}
}

bool MgQPlotScene::stepCached(int step)
{
	foreach(MgQLayerNode *node,layerNodes_)
	{
		if(node->stepNum() > 0 && node->stepCached(step) == false)
		{
			return false;
		}
		
	}

	return true;
}

void MgQPlotScene::stepMetaData(MgQStepMetaData* metaData)
{
	if(!sceneLayerNode_)
		return;

	const magics::SceneLayer& snl=sceneLayerNode_->sceneLayer(); 	
	MetaDataCollector stepData;

	for(int step=0; step < stepNum_; step++)
	{	
		metaData->addStep("");

		for(vector<Layer*>::iterator it = snl.beginLayer(step); it != snl.endLayer(step); ++it) 
		{
			//Layer& l=*it;

			foreach(QString str,metaData->keys())
			{
				stepData[str.toStdString()]="";
			}

			(*it)->collect(stepData);
			
			for(map<string,string>::iterator itK=stepData.begin(); itK!= stepData.end(); itK++)
			{
				QString keyStr=QString::fromStdString(itK->first);
				QString valueStr=QString::fromStdString(itK->second);
				if(valueStr != "")
				{
					valueStr+="\n";

					/*if(keyStr == "MV_Format")
					{
						valueStr="GRIB";
						valueStr+="\n";
					}*/

					metaData->appendToStepData(keyStr,step,valueStr);
				}
			}
		}
	}

	/*

	foreach(MgQLayerNode *node,layerNodes_)
	{
		if(node->stepNum() > 0)
		{
			int step=0;
			Layer& l=node->layer();
			StepLayer *layer=static_cast<StepLayer*>(&l);
			for(vector<SingleLayer*>::iterator it=layer->firstStep(); it != layer->endStep(); it++)
			{
				foreach(QString str,metaData->keys())
				{
					stepData[str.toStdString()]="";
				}				
				
				(*it)->metadata(stepData);
				
				if(firstLayer || metaData->stepNum() <= step)
				{
					metaData->addStep("");
				}

				for(map<string,string>::iterator itK=stepData.begin(); itK!= stepData.end(); itK++)
				{
					QString keyStr=QString::fromStdString(itK->first);
					QString valueStr=QString::fromStdString(itK->second);
					valueStr+="\n";


					if(keyStr == "MV_Format")
					{
						valueStr="GRIB";
						valueStr+="\n";
					}


					metaData->appendToStepData(keyStr,step,valueStr);
				}

				step++;
			}
			
			firstLayer=false;
		
		}
	}	*/
}


//-------------------------------------
// Layers
//-------------------------------------

void MgQPlotScene::updateLayers()
{
	//Regenarete the cache
	updateCache();
	
	//Repaint everything
	update();
}
 
void MgQPlotScene::addLayerNode(MgQLayerNode* node) 
{
	layerNodes_.push_back(node);
	//node->setStackLevel(layerNodes_.count()-1);
	assert(node->layer().zindex() >= 0);
	node->setStackLevel(node->layer().zindex());
}


MgQLayerNode* MgQPlotScene::layerNode(const Layer& layer)
{
	foreach(MgQLayerNode *node,layerNodes_)
	{
		if(&node->layer() == &layer)
		{
			return node;
		}
	}
	
	return 0;
}

void MgQPlotScene::restoreLayerState()
{
	if(previousLayerState_.count() == 0)
		return;

	return;
	
	QList<MgQLayerNode*> newNodes;
	QMap<int,MgQLayerNode*> stStackLevelToNode;	
	QSet<MgQLayerState*> stToNode;
	
	foreach(MgQLayerNode *node,layerNodes_)
	{
		bool found=false;
		foreach(MgQLayerState *st,previousLayerState_)
		{
			if(stToNode.contains(st) ==  false &&
			   st->id_ == QString::fromStdString(node->layer().id()))
			{
				stToNode.insert(st);
				stStackLevelToNode[st->stackLevel_]=node;
				node->setLayerAlpha(st->alpha_);
				node->setLayerVisibility(st->visible_);
				found=true;
				break;
			}
		}
		if(!found)
		{
			newNodes.push_back(node);
		}
	}	

	//Set stack level for the "old" nodes
	int currenStackLevel=0;
	foreach(int level,stStackLevelToNode.keys())
	{
		//qDebug() << "restore" << QString::fromStdString(stStackLevelToNode[level]->layer().id()) << level << currenStackLevel;
		stStackLevelToNode[level]->setStackLevel(currenStackLevel);
		currenStackLevel++;
	}
	
	//Set stack level for the "new" nodes 
	foreach(MgQLayerNode *node,newNodes)
	{
		node->setStackLevel(currenStackLevel);
		currenStackLevel++;
	}
}	

void MgQPlotScene::collectLayerData(QList<QPointF> &pos,QMap<int,QList<QStringList> > &val,bool addLayerName)
{
	for(int step=0; step < stepNum_; step++)
	{	
		collectLayerData(pos,val[step],step,addLayerName);
	}
}

void MgQPlotScene::collectLayerDataForCurrentStep(QList<QPointF> &pos,QList<QStringList> &val,bool addLayerName)
{
	collectLayerData(pos,val,currentStep_,addLayerName);
}

void MgQPlotScene::collectLayerData(QList<QPointF> &pos,QList<QStringList> &val,int step,bool addLayerName)
{
	if(!sceneLayerNode_)
		return;

	ValuesCollector layerData;
	const magics::SceneLayer& snl=sceneLayerNode_->sceneLayer(); 	
	QString str;

	for(vector<Layer*>::iterator it = snl.beginLayer(step); it != snl.endLayer(step); ++it) 
	{		
		layerData.clear();	

		if((*it)->visibility() == true)
		{
			foreach(QPointF pp,pos)
			{		
				std::pair<double,double> mp(pp.x(),pp.y());
				layerData[mp]=0;
			}		
	
			(*it)->collect(layerData);
	
			for(int i=0; i < pos.count(); i++)
			{		
				std::pair<double,double> mp(pos[i].x(),pos[i].y());
				if(layerData.find(mp) != layerData.end())
				{				
					str.clear();
					if(addLayerName)
					{
						QStringList lst=QString::fromStdString((*it)->name()).split("/");
						if(!lst.isEmpty())
						{
							str=lst.last();
						}
	
						str+="=";
					} 
					str+=QString::number(layerData[mp]);
					val[i] << str;
				}	
			}
		}	
	}

}

const map<string,string>& MgQPlotScene::getLayerInfoForCurrentStep(MgQLayerNode *node)
{
	return getLayerInfo(node,currentStep_);
}

const map<string,string>& MgQPlotScene::getLayerInfo(MgQLayerNode *node,int step)
{
	static const map<string,string> emptyMap;

	if(!node || !sceneLayerNode_ || 
	   !layerNodes_.contains(node))
		return emptyMap;

	if(node->stepNum() == 0)
	{
		return node->layer().getInfos();
	}
	else
	{
		const magics::SceneLayer& snl=sceneLayerNode_->sceneLayer(); 		
		Layer *stepLayer=snl.findLayer(&node->layer(),step);
		if(stepLayer)
			return stepLayer->getInfos();
		else
			return emptyMap;
	}

	return emptyMap;
}

QPixmap MgQPlotScene::getLayerInfoImageForCurrentStep(MgQLayerNode *node)
{
	return getLayerInfoImage(node,currentStep_);
}

QPixmap MgQPlotScene::getLayerInfoImage(MgQLayerNode *node,int step)
{
	if(!node || !sceneLayerNode_ || 
	   !layerNodes_.contains(node))
		return QPixmap();


	MgQHistoNode *histoNode=node->histoNode(step);
	if(!histoNode)	
		return QPixmap();

	if(!histoNode->cached())
	{
		Layer *stepLayer;

		if(node->stepNum() == 0)
		{
			stepLayer=&node->layer();
		}
		else
		{		
			const magics::SceneLayer& snl=sceneLayerNode_->sceneLayer(); 		
			stepLayer=snl.findLayer(&node->layer(),step);
		}

		if(!stepLayer)
			return QPixmap();
			
		driverObject_->driver().executeHisto(stepLayer,histoNode);

		histoNode->setCached(true);

	}	

	return node->histoPixmap(step,QSize(200,200));
}


void MgQPlotScene::renderLayerPreview()
{
	foreach(MgQLayerNode *node,layerNodes_)
	{
		node->renderPreview();
	}
}

//----------------------------------------
// Magnifier
//---------------------------------------

MgQMagnifierLayoutNode* MgQPlotScene::updateMagnifier(float zoomFactor)
{
	//magnifierZoom_=zoom;
	
	MgQLayerNode *magLayerNode=0;	

	foreach(MgQLayerNode *node,layerNodes_)
	{
		if(node->stepNum() > 0) //&& node->stepCached(currentStep_) == false)
		{
			magLayerNode=node;
			break;
			//driverObject_->driver().executeMagnifier(currentStep_,node);
				//node->setStepVisible(step,visible);
		}
	}

	if(!magLayerNode) 
		return 0;

	const magics::SceneLayer& snl=sceneLayerNode_->sceneLayer(); 	
	QString str;
	
	Layer *magLayer=snl.findLayer(&magLayerNode->layer(),currentStep_);

	MgQMagnifierLayoutNode *magLayoutNode=magLayerNode->magnifierLayoutNode();

	if(!magLayoutNode)
	{
		return 0;
	}		

	magLayoutNode->clearPlotContents(); 

	/*if(zoomFactor  != node->zoomFactor())
	{
		node->clearPlotContents();
	}
	else
	{ 
		return;
	}*/	

	magLayoutNode->setZoomFactor(zoomFactor);

	/*QVector<QPointF> pp;

	pp.push_back(QPointF(node->layout().minX(),node->layout().minY()));
	pp.push_back(QPointF(node->layout().maxX(),node->layout().minY()));
	pp.push_back(QPointF(node->layout().maxX(),node->layout().maxY()));
	pp.push_back(QPointF(node->layout().minX(),node->layout().maxY()));

	node->setArea(pp);*/

	float textWidth = 70.+20;
	float textHeight = 40.;
		
	magLayoutNode->setResolutionX(textWidth/zoomFactor);
	magLayoutNode->setResolutionY(textHeight/zoomFactor);

	driverObject_->driver().executeMagnifier(magLayer,magLayoutNode);

	return magLayoutNode;


	/*if(!animation_)
		return;

	node->setStep(animation_->currentStepObject());
		
	//Log::dev() << "GridValArea>  dx: " << fabsf(ldx) << " dy: " << fabsf(ldy) << endl;  
	for(int i=0; i <  pp.count() ; i++)
	{
		Log::dev() << " " << pp[i].x() << " " << pp[i].y() << endl;
	}	

	animation_->driver().redisplayMagnifier(node);*/
}

void MgQPlotScene::clearMagnifier()
{
	/*MgQLayoutNode *lnode=findMagnifierLayout();
	if(!lnode) return;

	MgQMagnifierLayoutNode *node=static_cast<MgQMagnifierLayoutNode*>(lnode);

	node->clearPlotContents();*/
}

/*void MgQPlotScene::drawItems(QPainter *painter, int numItems,
                             QGraphicsItem *items[],
                             const QStyleOptionGraphicsItem options[],
                             QWidget *widget)
 {
     for (int i = 0; i < numItems; ++i) {
          // Draw the item
          painter->save();
          painter->setMatrix(items[i]->sceneMatrix(), true);
          items[i]->paint(painter, &options[i], widget);
          painter->restore();
      }
 }
*/

void MgQPlotScene::updateCache()
{
	if ( !rootNode_ )
		return;
	QPainter *painter=cachePainter_;
	QStyleOptionGraphicsItem options;

	QRectF sourceRect = sceneRect();
        QRectF targetRect(0, 0, painter->device()->width(), painter->device()->height());

	cacheDevice_->fill(qRgba(255,255,255,255));

	painter->save();

	QTransform painterTransform;
    	painterTransform *= QTransform()
                        .translate(targetRect.left(), targetRect.top())
                        //.scale(xratio, yratio)
                        .translate(-sourceRect.left(), -sourceRect.top());
    	painter->setWorldTransform(painterTransform, true);

	rootNode_->setVisible(false);

	renderItemRecursively(rootNode_,painter,&options);

	rootNode_->setVisible(false);

	/*foreach(QGraphicsItem *item, items(Qt::AscendingOrder))
	{
		if(item->zValue() < 1.2 && item->isVisible())
		{
			painter->save();
          		painter->setTransform(item->sceneTransform(), true);
          		item->paint(painter, &options, 0);
          		painter->restore();			
		}
	}*/

	painter->restore();

	//cacheDevice_->save("/var/tmp/cgr/test.png");

	//rootNode_->setFlag(QGraphicsItem::ItemHasNoContents,true);
	rootNode_->setVisible(false); 


	//setBackgroundBrush(QPixmap::fromImage(*cacheDevice_));

}

void MgQPlotScene::drawBackground ( QPainter * painter, const QRectF & rect ) 
{	
	qDebug() << "bg" << rect << sceneRect();

	QRectF targetRect(0, 0, sceneRect().width(), sceneRect().height());
	//painter->drawPixmap(sceneRect(),*cacheDevice_,targetRect);

	painter->drawPixmap(rect,*cacheDevice_,rect);
}

void MgQPlotScene::renderForMagnifier(QPainter *painter, const QRectF &targetRect, const QRectF &sourceRect)
{
	QStyleOptionGraphicsItem option;
	painter->setRenderHint(QPainter::Antialiasing,antialias_);
	renderContents(painter,&option,targetRect,sourceRect,true);
}

void MgQPlotScene::renderForPrinter(QPainter *painter)
{
	QStyleOptionGraphicsItem option;
	
	QRectF sourceRect = sceneRect();
        QRectF targetRect(0, 0, painter->device()->width(), painter->device()->height());

	painter->setRenderHint(QPainter::Antialiasing,antialias_);
	renderContents(painter,&option,targetRect,sourceRect);
}

void MgQPlotScene::renderForVideo(QPainter *painter,QProgressDialog *progress,QString path,QStringList &files)
{	
	QPixmap *pix=static_cast<QPixmap*>(painter->device());
	QStyleOptionGraphicsItem option;
	
	QRectF sourceRect = sceneRect();
        QRectF targetRect(0, 0, painter->device()->width(), painter->device()->height());
		
	int oriCurrentStep=currentStep_;

	for(int i=0; i < stepNum_ && i < files.count() ; i++)
	{		
		progress->setValue(i);

		pix->fill();
		
		bool oriCached=stepCached(i);

		setStepVisible(i,true);	
		for(int j=0; j < stepNum_; j++)
		{
			if(j != i)
				setStepVisible(j,false);
		}

		renderContents(painter,&option,targetRect,sourceRect);
		
		pix->save(path + files[i]);
		
		if(oriCached == false)
		{
			foreach(MgQLayerNode *node,layerNodes_)
			{
				node->clearStep(i);
			}
		}
	}
	
	currentStep_=-1;
	setCurrentStep(oriCurrentStep);
}

void MgQPlotScene::setEnableAntialias(bool status)
{
	if(antialias_!=status)
	{
		antialias_=status;
		cachePainter_->setRenderHint(QPainter::Antialiasing,antialias_);

		updateAnimation();
	}
}
		