/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <pthread.h>
#include <errno.h>
#include <sys/sysctl.h>
#include <net/if_dl.h>
#include <net/route.h>
#include <stdlib.h>
#include <string.h>

#include "libsysactivity.h"
#include "utils.h"

pthread_key_t buffer_key;
pthread_key_t buffer_size_key;

static void create_keys();
static int helper(struct sa_network_interface* dst_array, uint16_t dst_size, uint16_t* written,
		char* name);

int sa_open_network() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_network_interfaces(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	return helper(NULL, 0, number, NULL);
}

int sa_get_network_interface(char* name, struct sa_network_interface* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	return helper(dst, 0, NULL, name);
}

int sa_get_network_interfaces(struct sa_network_interface* dst, uint16_t dst_size,
		uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	return helper(dst, dst_size, written, NULL);
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pthread_key_create(&buffer_key, free_keys);
	pthread_key_create(&buffer_size_key, NULL);

	keys_created = 1;
}

static int helper(struct sa_network_interface* dst_array, uint16_t dst_size, uint16_t* written,
		char* name) {
	int mib[6] = { CTL_NET, AF_ROUTE, 0, 0, NET_RT_IFLIST, 0 };
	size_t size;

	if (sysctl(mib, 6, NULL, &size, NULL, 0) == -1)
		return ENOSYS;

	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	void* buffer = pthread_getspecific(buffer_key);
	if (buffer_size < size) {
		buffer = realloc(buffer, size);
		if (pthread_setspecific(buffer_key, buffer))
			return ENOSYS;
		if (buffer == NULL)
			return ENOMEM;
		buffer_size = size;
		if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
			return ENOSYS;
	}

	if (sysctl(mib, 6, buffer, &size, NULL, 0) == -1)
		return ENOSYS;

	struct if_msghdr* if_hdr;
	struct rt_msghdr* rt_hdr;
	struct if_data* if_data;
	struct sockaddr_dl* sdl;
	struct sa_network_interface* dst = dst_array;
	size_t offset;

	if (written != NULL)
		*written = 0;

	for (offset = 0; offset < size; offset += rt_hdr->rtm_msglen) {
		if (dst_size > 0) {
			if (*written == dst_size)
				return 0;
			dst = &dst_array[*written];
		}

		rt_hdr = (struct rt_msghdr*) (buffer + offset);
		if (rt_hdr->rtm_version != RTM_VERSION || rt_hdr->rtm_type != RTM_IFINFO)
			continue;
		if_hdr = (struct if_msghdr*) (buffer + offset);
		if (!(if_hdr->ifm_addrs & (1 << RTAX_IFP)))
			continue;
		sdl = (struct sockaddr_dl*) (if_hdr + 1);
		if (sdl == NULL || sdl->sdl_family != AF_LINK)
			continue;

		if (dst != NULL) {
			if (name != NULL && strncmp(sdl->sdl_data, name, sizeof sdl->sdl_nlen) != 0)
				continue;

			strlcpy(dst->name, sdl->sdl_data, sizeof dst->name);
			if_data = &if_hdr->ifm_data;
			dst->received_bytes = if_data->ifi_ibytes;
			dst->received_packets = if_data->ifi_ipackets;
			dst->received_errors = if_data->ifi_ierrors;
			dst->received_drop = if_data->ifi_iqdrops;
			dst->received_multicast = if_data->ifi_imcasts;
			dst->sent_bytes = if_data->ifi_obytes;
			dst->sent_packets = if_data->ifi_opackets;
			dst->sent_errors = if_data->ifi_oerrors;
			dst->sent_multicast = if_data->ifi_omcasts;

			if (name != NULL)
				return 0;
		}

		if (written != NULL)
			(*written)++;
	}

	if (dst_size > 0 && *written < dst_size - 1)
		return ENODEV;
	if (name != NULL)
		return ENODEV;

	return 0;
}
