/* MouseHandler.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.mouse;

import java.awt.Graphics;
import java.awt.event.MouseEvent;
import javax.swing.JComponent;

/**
 * Handles certain types of mouse events and provides visual feedback. Mouse handlers
 * are intended to be registered with a {@link MouseTool}. 
 * The <i>parent component</i>, i.e., the component for which
 * the handler acts as a controller, is handed as an extra parameter to each method. 
 * This eases sharing of handler objects between components and in most cases 
 * should allow mouse handlers to be singleton objects.
 *
 * Note that this component is not necessarily the source of the mouse events.
 * <p>
 * Handlers are linked in a chain of responsibility. To indicate that
 * a handler is responsible
 * for a given mouse gesture it should consume the mouse pressed event
 * at the end of the methods {@link #doMousePressed},
 * {@link #doMouseClicked} and {@link #doPopup}.
 * The mouse gesture ends with the subsequent release of the mouse button.
 * <p>
 * Note that to be able to distinguish between a 'mouse clicked' and a 'mouse down'
 * the method {@link #doMousePressed} is only invoked as soon as the mouse is first
 * dragged.
 */
public interface MouseHandler {
    
    /**
     * Mouse handler that does not do anything. (For use in
     * the 'null object' design pattern.)
     */
    public static final MouseHandler NULL_MOUSE_HANDLER
            = new NullMouseHandler ();
    
    /**
     * Invoked when the mouse was clicked, i.e. the button was pressed 
     * and released without moving. If the click was a popup gesture for
     * the current platform, then {@link #doPopup} is invoked instead.
     * <p>
     * Handlers should consume the event if they take responsibility for this
     * mouse gesture.
     * @param parent Component for which this handler acts as a controller
     * @param mouseEvent Corresponding mouse event
     */
    public void doMouseClicked (JComponent parent, MouseEvent mouseEvent);
    
    /**
     * Invoked when a popup gesture was invoked and the
     * mouse did not actually move while the button was down.
     * <p>
     * Handlers should consume the event if they take responsibility for this
     * mouse gesture.
     * @param parent Component for which this handler acts as a controller
     * @param mouseEvent Corresponding mouse event
     */
    public void doPopup (JComponent parent, MouseEvent mouseEvent);
    
    /**
     * Invoked when the mouse was pressed down and then dragged
     * for the first time.
     * <p>
     * Handlers should consume the event if they take responsibility for the
     * mouse gesture started by this press (and ending with the subsequent
     * release of the button).
     * @param parent Component for which this handler acts as a controller
     * @param mouseEvent Corresponding mouse event
     */
    public void doMousePressed (JComponent parent, MouseEvent mouseEvent);
    
    /**
     * Invoked when the mouse pointer was dragged, i.e., moved
     * while the button is down. Will not be called when the 
     * corresponding {@link #doMousePressed} event
     * was not consumed by this object.
     * @param parent Component for which this handler acts as a controller
     * @param mouseEvent Corresponding mouse event
     * @param pressedEvent Mouse event that initiated this gesture
     * @param previousEvent Pressed or dragged event before this event.
     */
    public void doMouseDragged (JComponent parent, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent);
    
    /**
     * Invoked when the mouse pointer was released after it was dragged. 
     * Will not be called when the corresponding {@link #doMousePressed} event
     * was not consumed by this object.
     * @param parent Component for which this handler acts as a controller
     * @param mouseEvent Corresponding mouse event
     * @param pressedEvent Mouse event that initiated this gesture
     */
    public void doMouseReleased (JComponent parent, MouseEvent mouseEvent, MouseEvent pressedEvent);
    
    /**
     * Paint visual feedback while dragging the mouse.  
     * Will not be called when the corresponding {@link #doMousePressed} event
     * was not consumed by this object.
     * @param parent Component for which this handler acts as a controller
     * @param pressedEvent Mouse event that initiated this gesture
     * @param mouseEvent Mouse event corresponding to the last drag
     * @param g Graphics context onto which the visual feedback should be drawn
     */
    public void paintDragging (JComponent parent, Graphics g, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent);
    
    /**
     * Check whether this handler can work on the given component.
     */
    public boolean canHandle(JComponent parent);
    
}
