#include "osl/move.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

using namespace osl;

class MoveTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(MoveTest);
  CPPUNIT_TEST(testPass);
  CPPUNIT_TEST(testValid);
  CPPUNIT_TEST(testDeclareWin);
  CPPUNIT_TEST(testNormal);
  CPPUNIT_TEST(testSetFrom);
  CPPUNIT_TEST(testCapturePtpeO);
  CPPUNIT_TEST(testIgnoreUnpromote);
  CPPUNIT_TEST(testHasIgnoredUnpromote);
  CPPUNIT_TEST(testRotate180);
  CPPUNIT_TEST_SUITE_END();
public:
  void testValid() {
    const Move drop(Position(5,5), KNIGHT, WHITE);
    CPPUNIT_ASSERT(drop.isValid()); 

    const Move bug = Move::makeDirect(-58698240);
    CPPUNIT_ASSERT(! bug.isValid()); // KINGを取る55桂打
  }
  void testNormal()
  {
    const Move m76fu(Position(7,7),Position(7,6),PAWN,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(! m76fu.isPass());
    CPPUNIT_ASSERT(m76fu.isNormal());
    CPPUNIT_ASSERT(! m76fu.isInvalid());

    const Move m34fu(Position(3,3),Position(3,4),PAWN,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(! m34fu.isPass());
    CPPUNIT_ASSERT(m34fu.isNormal());
    CPPUNIT_ASSERT(! m34fu.isInvalid());
  }
  void testPass()
  {
    const Move pass_black = Move::PASS(BLACK);
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_black.ptype());
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_black.oldPtype());
    CPPUNIT_ASSERT_EQUAL(Position::STAND(), pass_black.from());
    CPPUNIT_ASSERT_EQUAL(Position::STAND(), pass_black.to());
    CPPUNIT_ASSERT_EQUAL(BLACK, pass_black.player());

    CPPUNIT_ASSERT(pass_black.isPass());
    CPPUNIT_ASSERT(! pass_black.isNormal());
    CPPUNIT_ASSERT(! pass_black.isInvalid());

    const Move pass_white = Move::PASS(WHITE);
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_white.ptype());
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_white.oldPtype());
    CPPUNIT_ASSERT_EQUAL(Position::STAND(), pass_white.from());
    CPPUNIT_ASSERT_EQUAL(Position::STAND(), pass_white.to());
    CPPUNIT_ASSERT_EQUAL(WHITE, pass_white.player());

    CPPUNIT_ASSERT(pass_white.isPass());
    CPPUNIT_ASSERT(! pass_white.isNormal());
    CPPUNIT_ASSERT(! pass_white.isInvalid());
  }
  void testDeclareWin()
  {
    CPPUNIT_ASSERT(Move::DeclareWin().isInvalid());
    CPPUNIT_ASSERT(! Move::DeclareWin().isNormal());
  }
  void testSetFrom()
  {
    const Position from(7,7);
    const Position to(7,6);
    const Ptype ptype = GOLD;
    const Player player = BLACK;
    const Ptype capture_ptype = PTYPE_EMPTY;
    const bool promote = false;
    Move m(from, to, ptype, capture_ptype, promote, player);
    Move m_copy(m);
    m=m.newFrom(Position(8,7));
    CPPUNIT_ASSERT_EQUAL(Position(8,7), m.from());
    CPPUNIT_ASSERT_EQUAL(to, m.to());
    CPPUNIT_ASSERT_EQUAL(ptype, m.ptype());
    CPPUNIT_ASSERT_EQUAL(capture_ptype, m.capturePtype());
    CPPUNIT_ASSERT_EQUAL(promote, m.isPromote());
    CPPUNIT_ASSERT_EQUAL(player, m.player());

    m=m.newFrom(Position(7,7));
    CPPUNIT_ASSERT_EQUAL(m_copy, m);
  }
  void testCapturePtpeO()
  {
    const Move drop_b(Position(5,5), KNIGHT, BLACK);
    const Move drop_w(Position(5,5), KNIGHT, WHITE);
    CPPUNIT_ASSERT_EQUAL(PTYPEO_EMPTY, drop_w.capturePtypeOSafe());
    CPPUNIT_ASSERT_EQUAL(PTYPEO_EMPTY, drop_b.capturePtypeOSafe());
  }
  void testIgnoreUnpromote()
  {
    // pawn
    CPPUNIT_ASSERT(Move(Position(4,4),Position(4,3),PAWN,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(4,3),PPAWN,PTYPE_EMPTY,true,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),PAWN,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,4),PAWN,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,6),Position(4,7),PAWN,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,6),Position(4,7),PPAWN,PTYPE_EMPTY,true,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,7),PAWN,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,6),PAWN,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    // lance
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(4,3),LANCE,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,4),Position(4,2),LANCE,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(4,3),PLANCE,PTYPE_EMPTY,true,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),LANCE,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,4),LANCE,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,6),Position(4,7),LANCE,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(Move(Position(4,6),Position(4,8),LANCE,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,6),Position(4,7),PLANCE,PTYPE_EMPTY,true,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,7),LANCE,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,6),LANCE,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    // bishop
    CPPUNIT_ASSERT(Move(Position(4,4),Position(2,2),BISHOP,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,2),Position(8,6),BISHOP,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(2,2),PBISHOP,PTYPE_EMPTY,true,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,2),Position(8,6),PBISHOP,PTYPE_EMPTY,true,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),BISHOP,BLACK).ignoreUnpromote(BLACK));

    CPPUNIT_ASSERT(Move(Position(6,6),Position(8,8),BISHOP,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(Move(Position(6,8),Position(2,4),BISHOP,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,6),Position(8,8),PBISHOP,PTYPE_EMPTY,true,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,8),Position(2,4),PBISHOP,PTYPE_EMPTY,true,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,7),BISHOP,WHITE).ignoreUnpromote(WHITE));
    // ROOK
    CPPUNIT_ASSERT(Move(Position(4,4),Position(4,2),ROOK,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,2),Position(4,6),ROOK,PTYPE_EMPTY,false,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(4,2),PROOK,PTYPE_EMPTY,true,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,2),Position(4,6),PROOK,PTYPE_EMPTY,true,BLACK).ignoreUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),ROOK,BLACK).ignoreUnpromote(BLACK));

    CPPUNIT_ASSERT(Move(Position(6,6),Position(6,8),ROOK,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(Move(Position(6,8),Position(6,4),ROOK,PTYPE_EMPTY,false,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,6),Position(6,8),PROOK,PTYPE_EMPTY,true,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,8),Position(6,4),PROOK,PTYPE_EMPTY,true,WHITE).ignoreUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,7),ROOK,WHITE).ignoreUnpromote(WHITE));
  }
  void testHasIgnoredUnpromote()
  {
    // pawn
    CPPUNIT_ASSERT(Move(Position(4,4),Position(4,3),PPAWN,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,3),Position(4,2),PPAWN,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,2),Position(4,1),PPAWN,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),PAWN,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,4),PAWN,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote(BLACK));

    CPPUNIT_ASSERT(Move(Position(4,6),Position(4,7),PPAWN,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(Move(Position(4,7),Position(4,8),PPAWN,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,8),Position(4,9),PPAWN,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,7),PAWN,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,6),PAWN,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote(WHITE));
    // lance
    CPPUNIT_ASSERT(Move(Position(4,4),Position(4,2),PLANCE,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(4,3),PLANCE,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(4,1),PLANCE,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),LANCE,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,4),LANCE,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,6),Position(4,8),PLANCE,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,6),Position(4,7),PLANCE,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,6),Position(4,9),PLANCE,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,7),LANCE,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(4,5),Position(4,6),LANCE,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote(WHITE));
    // bishop
    CPPUNIT_ASSERT(Move(Position(4,4),Position(2,2),PBISHOP,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,2),Position(8,6),PBISHOP,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,4),Position(2,2),PBISHOP,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),BISHOP,BLACK).hasIgnoredUnpromote(BLACK));

    CPPUNIT_ASSERT(Move(Position(6,6),Position(8,8),PBISHOP,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(Move(Position(6,8),Position(2,4),PBISHOP,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,6),Position(8,8),PBISHOP,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,7),BISHOP,WHITE).hasIgnoredUnpromote(WHITE));
    // ROOK
    CPPUNIT_ASSERT(Move(Position(4,4),Position(4,2),PROOK,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(Move(Position(4,2),Position(4,6),PROOK,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,2),Position(4,6),PROOK,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote(BLACK));
    CPPUNIT_ASSERT(!Move(Position(4,3),ROOK,BLACK).hasIgnoredUnpromote(BLACK));

    CPPUNIT_ASSERT(Move(Position(6,6),Position(6,8),PROOK,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(Move(Position(6,8),Position(6,4),PROOK,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,8),Position(6,4),PROOK,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote(WHITE));
    CPPUNIT_ASSERT(!Move(Position(6,7),ROOK,WHITE).hasIgnoredUnpromote(WHITE));
  }
  void testRotate180()
  {
    const Move m76(Position(7,7), Position(7,6), PAWN, PTYPE_EMPTY, false, BLACK);
    const Move m34(Position(3,3), Position(3,4), PAWN, PTYPE_EMPTY, false, WHITE);
    CPPUNIT_ASSERT_EQUAL(m76.rotate180(), m34);

    const Move m41(Position(4,1), SILVER, BLACK);
    const Move m69(Position(6,9), SILVER, WHITE);
    CPPUNIT_ASSERT_EQUAL(m41.rotate180(), m69);

    CPPUNIT_ASSERT_EQUAL(Move::PASS(BLACK).rotate180(), Move::PASS(WHITE));
  }
};

CPPUNIT_TEST_SUITE_REGISTRATION(MoveTest);

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
