/* effectUtil.t.cc
 */
#include "osl/effect_util/effectUtil.h"
#include "osl/effect_util/effectUtil.tcc"
#include "osl/state/numEffectState.h"
#include "osl/record/csaString.h"
#include "osl/misc/fixedCapacityVector.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>

class EffectUtilTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(EffectUtilTest);
  CPPUNIT_TEST(testCheapest);
  CPPUNIT_TEST(testCheapestPtype);
  CPPUNIT_TEST(testSafeCaptureNotByKing);
  CPPUNIT_TEST_SUITE_END();
public:
  void testCheapest();
  void testCheapestPtype();
  void testSafeCaptureNotByKing();
};

CPPUNIT_TEST_SUITE_REGISTRATION(EffectUtilTest);

using namespace osl;

struct PositionVector : public FixedCapacityVector<Position,81>
{
  
};

std::ostream& operator<<(std::ostream& os, const PositionVector& vec)
{
  for (size_t i=0; i<vec.size(); ++i)
  {
    os << vec[i] << "\n";
  }
  return os;
}

struct StorePosition
{
  PositionVector& out;
  StorePosition(PositionVector& o) : out(o)
  {
  }
  void operator()(Position pos)
  {
    out.push_back(pos);
  }
};




void EffectUtilTest::testCheapest()
{
  const char *endGameState =
  "P1+NY * +TO *  *  * -OU-KE-KY\n"
  "P2 *  *  *  *  * -GI-KI *  *\n"
  "P3 * +RY *  * +UM * -KI-FU-FU\n"
  "P4 *  * +FU-FU *  *  *  *  *\n"
  "P5 *  * -KE * +FU *  * +FU *\n"
  "P6-KE *  * +FU+GI-FU *  * +FU\n"
  "P7 *  * -UM *  *  *  *  *  *\n"
  "P8 *  *  *  *  *  *  *  *  * \n"
  "P9 * +OU * -GI *  *  *  * -NG\n"
  "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU00FU\n"
  "P-00KI00KY00FU\n"
  "P-00AL\n"
  "+\n";
  const NumEffectState state(CsaString(endGameState).getInitialState());
  CPPUNIT_ASSERT_EQUAL(state.getPieceOnBoard(Position(8,3)),
		       state.findCheapThreat
		       (BLACK, Position(9,3)));
  CPPUNIT_ASSERT_EQUAL(state.getPieceOnBoard(Position(9,6)),
		       state.findCheapThreat
		       (WHITE, Position(8,8)));
  CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(),
		       state.findCheapThreat
		       (BLACK, Position(2,8)));
  CPPUNIT_ASSERT_EQUAL(state.getPieceOnBoard(Position(3,2)),
		       state.findCheapThreat
		       (WHITE, Position(2,2)));
}

void EffectUtilTest::testCheapestPtype()
{
  const char *endGameState =
  "P1+NY * +TO *  *  * -OU-KE-KY\n"
  "P2 *  *  *  *  * -GI-KI *  *\n"
  "P3 * +RY *  * +UM * -KI-FU-FU\n"
  "P4 *  * +FU-FU *  *  *  *  *\n"
  "P5 *  * -KE * +FU *  * +FU *\n"
  "P6-KE *  * +FU+GI-FU *  * +FU\n"
  "P7 *  * -UM *  *  *  *  *  *\n"
  "P8 *  *  *  *  *  *  *  *  * \n"
  "P9 * +OU * -GI *  *  *  * -NG\n"
  "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU00FU\n"
  "P-00KI00KY00FU\n"
  "P-00AL\n"
  "+\n";
  const NumEffectState state(CsaString(endGameState).getInitialState());
  CPPUNIT_ASSERT_EQUAL(PROOK, state.findCheapThreat(BLACK, Position(9,3)).ptype());
  CPPUNIT_ASSERT_EQUAL(KNIGHT, state.findCheapThreat(WHITE, Position(8,8)).ptype());
  CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, state.findCheapThreat(BLACK, Position(2,8)).ptype());
  CPPUNIT_ASSERT_EQUAL(GOLD, state.findCheapThreat(WHITE, Position(2,2)).ptype());
}


void EffectUtilTest::testSafeCaptureNotByKing()
{
  const char *state_false_string =
    "P1-OU-KI * +HI *  *  *  *  * \n"
    "P2+FU *  *  *  *  *  *  *  * \n"
    "P3 * +TO *  *  *  *  *  *  * \n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  *  *  *  *  *  *  * +OU\n"
    "P+00FU\n"
    "P-00AL\n"
    "-\n";
  const NumEffectState state_false(CsaString(state_false_string).getInitialState());
  const Position target = Position(9,2);
  CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), EffectUtil::safeCaptureNotByKing
		       (WHITE, state_false, target));
  
  const char *state_true_string =
    "P1-OU-KI * +KA *  *  *  *  * \n"
    "P2+FU *  *  *  *  *  *  *  * \n"
    "P3 * +TO *  *  *  *  *  *  * \n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  *  *  *  *  *  *  * +OU\n"
    "P+00FU\n"
    "P-00AL\n"
    "-\n";
  const NumEffectState state_true(CsaString(state_true_string).getInitialState());
  CPPUNIT_ASSERT_EQUAL(state_true.getPieceOnBoard(Position(8,1)), 
		       EffectUtil::safeCaptureNotByKing
		       (WHITE, state_true, target));
}


/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
