/* speculativeAllMoves.h
 */
#ifndef OSL_SPECULATIVEALLMOVES_H
#define OSL_SPECULATIVEALLMOVES_H

#include "osl/game_playing/computerPlayer.h"
#include "osl/game_playing/speculativeModel.h"
#include "osl/misc/lightMutex.h"
#include "osl/misc/fixedCapacityVector.h"
#include "osl/misc/milliSeconds.h"
#include <boost/thread/thread.hpp>
#include <boost/thread/condition.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>

namespace osl
{
  namespace misc
  {
    class RealTime;
  }
  namespace search
  {
    class TimeAssigned;
  }
  namespace game_playing
  {
    class SearchPlayer;
    /**
     * 1threadで全ての手を順番に投機的探索をする
     */
    class SpeculativeAllMoves : public SpeculativeModel
    {
    public:
      struct SearchAllMoves;
      struct ResultVector;
    private:
      boost::shared_ptr<SearchAllMoves> searcher;
      boost::scoped_ptr<boost::thread> thread;
      boost::scoped_ptr<ResultVector> results;
      boost::mutex mutex;
      int last_search_seconds;
      bool has_byoyomi;
      bool allowed;
      HashKey search_state;
    public:
      SpeculativeAllMoves();
      ~SpeculativeAllMoves();

      void startSpeculative(const boost::shared_ptr<GameState> state,
			    const SearchPlayer& main_player);
      void stopOtherThan(Move);
      void stopAll();

      void setMaxThreads(int new_max_threads)
      {
	boost::mutex::scoped_lock lk(mutex);
	allowed = (new_max_threads > 0);
      }

      const MoveWithComment waitResult(Move last_move, search::TimeAssigned wait_for,
				       SearchPlayer& main_player, int byoyomi);

      void selectBestMoveCleanUp();
      void clearResource();
      const HashKey searchState() const { return search_state; }
    private:
      class Runner;
    };

    class SpeculativeAllMoves::ResultVector
    {
      typedef FixedCapacityVector<std::pair<Move,MoveWithComment>,Move::MaxUniqMoves> vector_t;
      vector_t data;
      typedef LightMutex Mutex;
      mutable Mutex mutex;
    public:
      ResultVector();
      ~ResultVector();
      
      void add(Move prediction, const MoveWithComment& result);
      const MoveWithComment* find(Move prediction) const;
      void clear();
      void show(std::ostream&) const;
    };

    /**
     * 指手を生成し，結果をresultsにためる．
     * run を別threadで動かすことを想定しているが，逐次でもテスト可
     */
    class SpeculativeAllMoves::SearchAllMoves
    {
    public:
      enum Status { 
	INITIAL, RUNNING, PREDICTION1, PREDICTION2, SEARCH1, SEARCH2, FINISHED
      };
      class Generator;
      friend class Generator;
      friend class SpeculativeAllMoves;
    private:
      boost::shared_ptr<GameState> state;
      boost::shared_ptr<SearchPlayer> player;
      boost::scoped_ptr<Generator> generator;
      SpeculativeAllMoves::ResultVector& results;
      Move current_move;
      volatile Status status;
      int seconds;
      typedef boost::mutex Mutex;
      mutable Mutex mutex;
      boost::condition condition;
      /** true なら次の予想探索にはいらない */
      volatile bool stop_flag;
    public:
      explicit SearchAllMoves(SpeculativeAllMoves::ResultVector&);
      ~SearchAllMoves();

      void setUp(const GameState&, const SearchPlayer&, int standard_seconds,
		 bool has_byoyomi);

      void run();
      
      void stopNow();
      void stopOtherThan(Move);
      void waitRunning();
      bool isFinished() const { return status == FINISHED; }

      void setTimeAssign(const search::TimeAssigned&);
      const MilliSeconds startTime();
      const Move currentMove() const;

      SearchPlayer* currentPlayer() { return player.get(); }
    private:
      const MoveWithComment testMove(Move);
      class StatusLock;
    };
  } // game_playing
} // osl

#endif /* OSL_SPECULATIVEALLMOVES_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
