#include "osl/piece.h"
#include "osl/pieceTable.h"
#include "osl/container/pieceMask.h"
#include "osl/misc/slowBsf.h"
#include "osl/state/simpleState.h"

#include <iomanip>
#include <iostream>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class PieceTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PieceTest);
  CPPUNIT_TEST(testShow);
  CPPUNIT_TEST(testSize);
  CPPUNIT_TEST(testBsf);
  CPPUNIT_TEST(testAddPosition);
  CPPUNIT_TEST(testIsEmpty);
  CPPUNIT_TEST(testIsEdge);
  CPPUNIT_TEST(testIsPiece);
  CPPUNIT_TEST(testIsOnBoardBy);
  CPPUNIT_TEST(testPromote);
  CPPUNIT_TEST(testCanMoveOn);
  CPPUNIT_TEST(testPieceMask);
  CPPUNIT_TEST_SUITE_END();
 public:
  void testSize();
  void testShow();
  void testBsf();
  void testAddPosition();
  void testIsEmpty();
  void testIsEdge();
  void testIsPiece();
  void testIsOnBoardBy();
  void testPromote();
  void testCanMoveOn();
  void testPieceMask();
};

using namespace osl;
using namespace osl::misc;
extern bool isShortTest;

CPPUNIT_TEST_SUITE_REGISTRATION(PieceTest);

void PieceTest::testSize(){
  CPPUNIT_ASSERT_EQUAL(sizeof(int), sizeof(Piece));
}

void PieceTest::testShow(){
  if (! isShortTest)
  {
    for(int num=0;num<Piece::SIZE;num++)
      std::cerr << num << ":" << Piece_Table.getPtypeOf(num) 
		<< std::setbase(10) << std::endl;
  }
}
void PieceTest::testBsf(){
  for(unsigned int i=1;i<i+11111;i+=11111){
    CPPUNIT_ASSERT(BitOp::bsf(i)==slowbsf(i));
  }
}
void PieceTest::testAddPosition(){
  Piece p1=Piece(BLACK,PPAWN,0,Position(7,5));
  CPPUNIT_ASSERT(p1.position()==Position(7,5));
  p1+=Offset(1,2);
  CPPUNIT_ASSERT(p1==Piece(BLACK,PPAWN,0,Position(8,7)));
  p1.setPosition(Position(6,4));
  CPPUNIT_ASSERT(p1==Piece(BLACK,PPAWN,0,Position(6,4)));
}
void PieceTest::testIsEmpty(){
  CPPUNIT_ASSERT(! Piece(BLACK,PPAWN,0,Position(8,7)).isEmpty());
  CPPUNIT_ASSERT(! Piece(WHITE,KING,30,Position(6,4)).isEmpty());
  CPPUNIT_ASSERT(Piece::EMPTY().isEmpty());
  CPPUNIT_ASSERT(! Piece::EDGE().isEmpty());
}
void PieceTest::testIsEdge(){
  CPPUNIT_ASSERT(! Piece(BLACK,PPAWN,0,Position(8,7)).isEdge());
  CPPUNIT_ASSERT(! Piece(WHITE,KING,30,Position(6,4)).isEdge());
  CPPUNIT_ASSERT(! Piece::EMPTY().isEdge());
  CPPUNIT_ASSERT(Piece::EDGE().isEdge());
}
void PieceTest::testIsPiece()
{
  SimpleState state(HIRATE);
  for (int x=0; x<=10; ++x) {
    for (int y=0; y<=10; ++y) {
      const Piece p = state.getPieceAt(Position(x,y));
      const bool is_piece = p.isPiece();
      const bool is_piece_naive = (! p.isEdge()) && (! p.isEmpty());
      CPPUNIT_ASSERT_EQUAL(is_piece_naive, is_piece);
    }
  }
}

void PieceTest::testIsOnBoardBy(){
  for(int pi=0;pi<2;pi++){
    for(int ptype=PTYPE_PIECE_MIN;ptype<=PTYPE_MAX;ptype++){
      Player pl=indexToPlayer(pi);
      for(int num=0;num<40;num++){
	{
	  Piece p(pl,(Ptype)ptype,num,Position::STAND());
	  CPPUNIT_ASSERT(!p.isOnBoardByOwner<BLACK>());
	  CPPUNIT_ASSERT(!p.isOnBoardByOwner<WHITE>());
	  CPPUNIT_ASSERT(!p.isOnBoardByOwner(BLACK));
	  CPPUNIT_ASSERT(!p.isOnBoardByOwner(WHITE));
	}
	for(int x=1;x<=9;x++)
	  for(int y=1;y<=9;y++){
	    Piece p(pl,(Ptype)ptype,num,Position(x,y));
	    if(pl==BLACK){
	      CPPUNIT_ASSERT(p.isOnBoardByOwner<BLACK>());
	      CPPUNIT_ASSERT(!p.isOnBoardByOwner<WHITE>());
	      CPPUNIT_ASSERT(p.isOnBoardByOwner(BLACK));
	      CPPUNIT_ASSERT(!p.isOnBoardByOwner(WHITE));
	    }
	    else{
	      CPPUNIT_ASSERT(!p.isOnBoardByOwner<BLACK>());
	      CPPUNIT_ASSERT(p.isOnBoardByOwner<WHITE>());
	      CPPUNIT_ASSERT(!p.isOnBoardByOwner(BLACK));
	      CPPUNIT_ASSERT(p.isOnBoardByOwner(WHITE));
	    }
	  }
      }
    }
  }
  {
    CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner<BLACK>());
    CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner<WHITE>());
    CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner(BLACK));
    CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner(WHITE));
  }
  {
    CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner<BLACK>());
    CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner<WHITE>());
    CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner(BLACK));
    CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner(WHITE));
  }
  Piece p1=Piece(BLACK,PPAWN,0,Position(8,7));
  Piece p2=Piece(WHITE,KING,30,Position(6,4));
  Piece p3=Piece(BLACK,PAWN,0,Position::STAND());
  Piece p4=Piece(WHITE,ROOK,39,Position::STAND());
  CPPUNIT_ASSERT(p1.isOnBoardByOwner<BLACK>());
  CPPUNIT_ASSERT(!p1.isOnBoardByOwner<WHITE>());
  CPPUNIT_ASSERT(!p2.isOnBoardByOwner<BLACK>());
  CPPUNIT_ASSERT(p2.isOnBoardByOwner<WHITE>());
  CPPUNIT_ASSERT(!p3.isOnBoardByOwner<BLACK>());
  CPPUNIT_ASSERT(!p3.isOnBoardByOwner<WHITE>());
  CPPUNIT_ASSERT(!p4.isOnBoardByOwner<BLACK>());
  CPPUNIT_ASSERT(!p4.isOnBoardByOwner<WHITE>());
  CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner<BLACK>());
  CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner<WHITE>());
  CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner<BLACK>());
  CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner<WHITE>());
  CPPUNIT_ASSERT(p1.isOnBoardByOwner(BLACK));
  CPPUNIT_ASSERT(!p1.isOnBoardByOwner(WHITE));
  CPPUNIT_ASSERT(!p2.isOnBoardByOwner(BLACK));
  CPPUNIT_ASSERT(p2.isOnBoardByOwner(WHITE));
  CPPUNIT_ASSERT(!p3.isOnBoardByOwner(BLACK));
  CPPUNIT_ASSERT(!p3.isOnBoardByOwner(WHITE));
  CPPUNIT_ASSERT(!p4.isOnBoardByOwner(BLACK));
  CPPUNIT_ASSERT(!p4.isOnBoardByOwner(WHITE));
  CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner(BLACK));
  CPPUNIT_ASSERT(!Piece::EMPTY().isOnBoardByOwner(WHITE));
  CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner(BLACK));
  CPPUNIT_ASSERT(!Piece::EDGE().isOnBoardByOwner(WHITE));

  SimpleState state(HIRATE);
  for (int x=0; x<=10; ++x) {
    for (int y=0; y<=10; ++y) {
      const Piece p = state.getPieceAt(Position(x,y));
      const bool is_onboard_by_owner[2] = {
	p.isPiece() && p.owner() == BLACK,
	p.isPiece() && p.owner() == WHITE,
      };

      CPPUNIT_ASSERT_EQUAL(is_onboard_by_owner[0],
			   p.isOnBoardByOwner(BLACK));
      if(is_onboard_by_owner[1] &&
	 !p.isOnBoardByOwner(WHITE)){
	std::cerr << p << ",p.intValue()=" << p.intValue() << std::endl;
	Piece p1=Piece::EDGE();
	std::cerr << p1 << ",p1.intValue()=" << p1.intValue() << std::endl;
      }
      CPPUNIT_ASSERT_EQUAL(is_onboard_by_owner[1],
			   p.isOnBoardByOwner(WHITE));
      CPPUNIT_ASSERT_EQUAL(is_onboard_by_owner[0],
			   p.isOnBoardByOwner<BLACK>());
      CPPUNIT_ASSERT_EQUAL(is_onboard_by_owner[1],
			   p.isOnBoardByOwner<WHITE>());
    }
  }
}
void PieceTest::testPromote(){
  Piece p1=Piece(BLACK,PAWN,0,Position(8,7));
  Piece p2=p1.promote();
  CPPUNIT_ASSERT(p2==Piece(BLACK,PPAWN,0,Position(8,7)));
  Piece p3=p2.unpromote();
  CPPUNIT_ASSERT(p1==p3);
  Piece p4=p1.checkPromote(0); 
  CPPUNIT_ASSERT(p1==p4);
  Piece p5=p1.checkPromote(1); 
  CPPUNIT_ASSERT(p2==p5);
}

void PieceTest::testCanMoveOn(){
  for(int pi=0;pi<2;pi++){
    for(int ptype=PTYPE_PIECE_MIN;ptype<=PTYPE_MAX;ptype++)
      for(int x=1;x<=9;x++)
	for(int y=1;y<=9;y++)
	  for(int num=0;num<40;num++){
	    Player pl=indexToPlayer(pi);
	    Piece p(pl,(Ptype)ptype,num,Position(x,y));
	    if(pl==BLACK){
	      CPPUNIT_ASSERT(!p.canMoveOn<BLACK>());
	      CPPUNIT_ASSERT(p.canMoveOn<WHITE>());
	    }
	    else{
	      CPPUNIT_ASSERT(!p.canMoveOn<WHITE>());
	      CPPUNIT_ASSERT(p.canMoveOn<BLACK>() ||
			     (std::cerr << "p=" << p << std::endl,0)
			     );
	    }
	  }
  }
  {
    CPPUNIT_ASSERT(Piece::EMPTY().canMoveOn<BLACK>());
    CPPUNIT_ASSERT(Piece::EMPTY().canMoveOn<WHITE>());
  }
  {
    CPPUNIT_ASSERT(!Piece::EDGE().canMoveOn<BLACK>());
    CPPUNIT_ASSERT(!Piece::EDGE().canMoveOn<WHITE>());
  }
}

void PieceTest::testPieceMask(){
  PieceMask pm;
  // std::cerr << pm << std::endl;
  for(int i=0;i<40;i++)
    pm.set(i);
  for(int i=5;i<30;i++)
    pm.reset(i);
#if OSL_WORDSIZE == 64
  CPPUNIT_ASSERT_EQUAL(mask_t::makeDirect(0xffc000001fuLL),pm.getMask(0));
#elif OSL_WORDSIZE == 32
  std::cerr << std::hex << mask_t::makeDirect(0xc000001f) << "\n";
  CPPUNIT_ASSERT_EQUAL(mask_t::makeDirect(0xc000001f),pm.getMask(0));
  CPPUNIT_ASSERT_EQUAL(mask_t::makeDirect(0xff),pm.getMask(1));
#endif
  for(int i=0;i<5;i++)
    CPPUNIT_ASSERT(pm.test(i));
  for(int i=5;i<30;i++)
    CPPUNIT_ASSERT(!pm.test(i));
  for(int i=30;i<40;i++)
    CPPUNIT_ASSERT(pm.test(i));
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
