// $Id: cell_hex20.C 3874 2010-07-02 21:57:26Z roystgnr $

// The libMesh Finite Element Library.
// Copyright (C) 2002-2008 Benjamin S. Kirk, John W. Peterson, Roy H. Stogner
  
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
  
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
  
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


// C++ includes

// Local includes
#include "side.h"
#include "cell_hex20.h"
#include "edge_edge3.h"
#include "face_quad8.h"

namespace libMesh
{



// ------------------------------------------------------------
// Hex20 class static member initializations

const unsigned int Hex20::side_nodes_map[6][8] =
{
  {0, 3, 2, 1, 11, 10,  9,  8}, // Side 0
  {0, 1, 5, 4,  8, 13, 16, 12}, // Side 1
  {1, 2, 6, 5,  9, 14, 17, 13}, // Side 2
  {2, 3, 7, 6, 10, 15, 18, 14}, // Side 3
  {3, 0, 4, 7, 11, 12, 19, 15}, // Side 4
  {4, 5, 6, 7, 16, 17, 18, 19}  // Side 5
};

const unsigned int Hex20::edge_nodes_map[12][3] =
{
  {0, 1, 8},  // Side 0
  {1, 2, 9},  // Side 1
  {2, 3, 10}, // Side 2
  {0, 3, 11}, // Side 3
  {0, 4, 12}, // Side 4
  {1, 5, 13}, // Side 5
  {2, 6, 14}, // Side 6
  {3, 7, 15}, // Side 7
  {4, 5, 16}, // Side 8
  {5, 6, 17}, // Side 9
  {6, 7, 18}, // Side 10
  {4, 7, 19}  // Side 11
};



// ------------------------------------------------------------
// Hex20 class member functions

bool Hex20::is_vertex(const unsigned int i) const
{
  if (i < 8)
    return true;
  return false;
}

bool Hex20::is_edge(const unsigned int i) const
{
  if (i > 7)
    return true;
  return false;
}

bool Hex20::is_face(const unsigned int) const
{
  return false;
}

bool Hex20::is_node_on_side(const unsigned int n,
			    const unsigned int s) const
{
  libmesh_assert(s < n_sides());
  for (unsigned int i = 0; i != 8; ++i)
    if (side_nodes_map[s][i] == n)
      return true;
  return false;
}

bool Hex20::is_node_on_edge(const unsigned int n,
			    const unsigned int e) const
{
  libmesh_assert(e < n_edges());
  for (unsigned int i = 0; i != 3; ++i)
    if (edge_nodes_map[e][i] == n)
      return true;
  return false;
}



bool Hex20::has_affine_map() const
{
  // Make sure x-edge endpoints are affine
  Point v = this->point(1) - this->point(0);
  if (!v.relative_fuzzy_equals(this->point(2) - this->point(3)) ||
      !v.relative_fuzzy_equals(this->point(5) - this->point(4)) ||
      !v.relative_fuzzy_equals(this->point(6) - this->point(7)))
    return false;
  // Make sure x-edges are straight
  v /= 2;
  if (!v.relative_fuzzy_equals(this->point(8) - this->point(0)) ||
      !v.relative_fuzzy_equals(this->point(10) - this->point(3)) ||
      !v.relative_fuzzy_equals(this->point(16) - this->point(4)) ||
      !v.relative_fuzzy_equals(this->point(18) - this->point(7)))
    return false;
  // Make sure xz-faces are identical parallelograms
  v = this->point(4) - this->point(0);
  if (!v.relative_fuzzy_equals(this->point(7) - this->point(3)))
    return false;
  v /= 2;
  if (!v.relative_fuzzy_equals(this->point(12) - this->point(0)) ||
      !v.relative_fuzzy_equals(this->point(13) - this->point(1)) ||
      !v.relative_fuzzy_equals(this->point(14) - this->point(2)) ||
      !v.relative_fuzzy_equals(this->point(15) - this->point(3)))
    return false;
  // Make sure y-edges are straight
  v = (this->point(3) - this->point(0))/2;
  if (!v.relative_fuzzy_equals(this->point(11) - this->point(0)) ||
      !v.relative_fuzzy_equals(this->point(9) - this->point(1)) ||
      !v.relative_fuzzy_equals(this->point(17) - this->point(5)) ||
      !v.relative_fuzzy_equals(this->point(19) - this->point(4)))
    return false;
  // If all the above checks out, the map is affine
  return true;
}



AutoPtr<Elem> Hex20::build_side (const unsigned int i,
				 bool proxy ) const
{
  libmesh_assert (i < this->n_sides());

  if (proxy)
    {
      AutoPtr<Elem> ap(new Side<Quad8,Hex20>(this,i));
      return ap;
    }

  else
    {
      AutoPtr<Elem> face(new Quad8);

      // Think of a unit cube: (-1,1) x (-1,1)x (1,1)
      switch (i)
	{
	case 0:  // the face at z=0
	  {
	    face->set_node(0) = this->get_node(0);
	    face->set_node(1) = this->get_node(3);
	    face->set_node(2) = this->get_node(2);
	    face->set_node(3) = this->get_node(1);
	    face->set_node(4) = this->get_node(11);
	    face->set_node(5) = this->get_node(10);
	    face->set_node(6) = this->get_node(9);
	    face->set_node(7) = this->get_node(8);

	    return face;
	  }
	case 1:  // the face at y = 0
	  {
	    face->set_node(0) = this->get_node(0);
	    face->set_node(1) = this->get_node(1);
	    face->set_node(2) = this->get_node(5);
	    face->set_node(3) = this->get_node(4);
	    face->set_node(4) = this->get_node(8);
	    face->set_node(5) = this->get_node(13);
	    face->set_node(6) = this->get_node(16);
	    face->set_node(7) = this->get_node(12);

	    return face;
	  }
	case 2:  // the face at x=1
	  {
	    face->set_node(0) = this->get_node(1);
	    face->set_node(1) = this->get_node(2);
	    face->set_node(2) = this->get_node(6);
	    face->set_node(3) = this->get_node(5);
	    face->set_node(4) = this->get_node(9);
	    face->set_node(5) = this->get_node(14);
	    face->set_node(6) = this->get_node(17);
	    face->set_node(7) = this->get_node(13);

	    return face;
	  }
	case 3: // the face at y=1
	  {
	    face->set_node(0) = this->get_node(2);
	    face->set_node(1) = this->get_node(3);
	    face->set_node(2) = this->get_node(7);
	    face->set_node(3) = this->get_node(6);
	    face->set_node(4) = this->get_node(10);
	    face->set_node(5) = this->get_node(15);
	    face->set_node(6) = this->get_node(18);
	    face->set_node(7) = this->get_node(14);

	    return face;
	  }
	case 4: // the face at x=0
	  {
	    face->set_node(0) = this->get_node(3);
	    face->set_node(1) = this->get_node(0);
	    face->set_node(2) = this->get_node(4);
	    face->set_node(3) = this->get_node(7);
	    face->set_node(4) = this->get_node(11);
	    face->set_node(5) = this->get_node(12);
	    face->set_node(6) = this->get_node(19);
	    face->set_node(7) = this->get_node(15);

	    return face;
	  }
	case 5: // the face at z=1
	  {
	    face->set_node(0) = this->get_node(4);
	    face->set_node(1) = this->get_node(5);
	    face->set_node(2) = this->get_node(6);
	    face->set_node(3) = this->get_node(7);
	    face->set_node(4) = this->get_node(16);
	    face->set_node(5) = this->get_node(17);
	    face->set_node(6) = this->get_node(18);
	    face->set_node(7) = this->get_node(19);

	    return face;
	  }
	default:
	  {
	    libmesh_error();
	    return face;
	  }
	}
    }
  
  // We'll never get here.
  libmesh_error();
  AutoPtr<Elem> ap(NULL);  return ap;
}



AutoPtr<Elem> Hex20::build_edge (const unsigned int i) const
{
  libmesh_assert (i < this->n_edges());

  AutoPtr<Elem> ap(new SideEdge<Edge3,Hex20>(this,i));
  return ap;
}



void Hex20::connectivity(const unsigned int sc,
			 const IOPackage iop,
			 std::vector<unsigned int>& conn) const
{
  libmesh_assert (_nodes != NULL);
  libmesh_assert (sc < this->n_sub_elem());
  libmesh_assert (iop != INVALID_IO_PACKAGE);

  
  switch (iop)
    {
    case TECPLOT:
      {
	switch (sc)
	  {
	  case 0:
	    conn.resize(8);
	    conn[0] = this->node(0)+1;
	    conn[1] = this->node(1)+1;
	    conn[2] = this->node(2)+1;
	    conn[3] = this->node(3)+1;
	    conn[4] = this->node(4)+1;
	    conn[5] = this->node(5)+1;
	    conn[6] = this->node(6)+1;
	    conn[7] = this->node(7)+1;

	    return;
      
	  default:
	    libmesh_error();
	  }
      }

    case VTK:
      {
	switch (sc)
	  {
	  case 0:
	    conn.resize(20);
	    conn[0] = this->node(0);
	    conn[1] = this->node(1);
	    conn[2] = this->node(2);
	    conn[3] = this->node(3);
	    conn[4] = this->node(4);
	    conn[5] = this->node(5);
	    conn[6] = this->node(6);
	    conn[7] = this->node(7);
	    conn[8] = this->node(8);
	    conn[9] = this->node(9);
	    conn[10] = this->node(10);
	    conn[11] = this->node(11);
	    conn[12] = this->node(16);
	    conn[13] = this->node(17);
	    conn[14] = this->node(18);
	    conn[15] = this->node(19);
	    conn[16] = this->node(12);
	    conn[17] = this->node(13);
	    conn[18] = this->node(14);
	    conn[19] = this->node(15);
	    return;
	    
	  default:
	    libmesh_error();
	  }
      }
      
    default:
      libmesh_error();
    }

  libmesh_error();
}




unsigned short int Hex20::second_order_adjacent_vertex (const unsigned int n,
							const unsigned int v) const
{ 
  libmesh_assert (n >= this->n_vertices());
  libmesh_assert (n <  this->n_nodes());
  libmesh_assert (v < 2);
  /*
   * the _second_order_adjacent_vertices matrix is
   * stored in cell_hex.C, since this matrix is identical
   * for Hex20 and Hex27 (for the first 12 higher-order nodes)
   */
  return _second_order_adjacent_vertices[n-this->n_vertices()][v]; 
}



std::pair<unsigned short int, unsigned short int>
Hex20::second_order_child_vertex (const unsigned int n) const
{
  libmesh_assert (n >= this->n_vertices());
  libmesh_assert (n < this->n_nodes());
  /*
   * the _second_order_vertex_child_* vectors are
   * stored in cell_hex.C, since they are identical
   * for Hex20 and Hex27 (for the first 12 higher-order nodes)
   */
  return std::pair<unsigned short int, unsigned short int>
    (_second_order_vertex_child_number[n],
     _second_order_vertex_child_index[n]);
}




#ifdef LIBMESH_ENABLE_AMR

const float Hex20::_embedding_matrix[8][20][20] =
{
  // embedding matrix for child 0
  {
    //  0     1     2     3     4     5     6     7    8  9  10 11  12  13  14  15  16  17  18  19
    {    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 0
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 1
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 2
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 3
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 4
    {  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000 }, // 5
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 6
    {  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000 }, // 7
    {   0.375000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 8
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,   0.375000,   0.250000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 9
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.250000,   0.375000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 10
    {   0.375000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 11
    {   0.375000,    0.00000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 12
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000 }, // 13
    {  -0.281250,  -0.281250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.156250,  -0.156250,   0.375000,   0.375000,   0.375000,   0.375000,   0.187500,   0.187500,   0.187500,   0.187500,   0.125000,   0.125000,   0.125000,   0.125000 }, // 14
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.750000,   0.375000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000 }, // 15
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000,   0.250000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000 }, // 16
    {  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500 }, // 17
    {  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000 }, // 18
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.375000,   0.750000,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000 }  // 19
  },

  // embedding matrix for child 1
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 0
    {    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 1
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 2
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 3
    {  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000 }, // 4
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 5
    {    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000 }, // 6
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 7
    {  -0.125000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 8
    {    0.00000,   0.375000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 9
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.750000,   0.375000,   0.250000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 10
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,   0.375000,   0.250000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 11
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000 }, // 12
    {    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 13
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000 }, // 14
    {  -0.281250,  -0.281250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.156250,  -0.156250,   0.375000,   0.375000,   0.375000,   0.375000,   0.187500,   0.187500,   0.187500,   0.187500,   0.125000,   0.125000,   0.125000,   0.125000 }, // 15
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000,   0.750000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000 }, // 16
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000,   0.250000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000 }, // 17
    {  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000 }, // 18
    {  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500 }  // 19
  },

  // embedding matrix for child 2
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 0
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 1
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 2
    {    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 3
    {  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000 }, // 4
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 5
    {    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000 }, // 6
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 7
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.250000,   0.375000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 8
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.250000,   0.375000,   0.750000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 9
    {    0.00000,    0.00000,  -0.125000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 10
    {  -0.125000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 11
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.750000,   0.375000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000 }, // 12
    {  -0.281250,  -0.281250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.156250,  -0.156250,   0.375000,   0.375000,   0.375000,   0.375000,   0.187500,   0.187500,   0.187500,   0.187500,   0.125000,   0.125000,   0.125000,   0.125000 }, // 13
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.250000,    0.00000 }, // 14
    {    0.00000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 15
    {  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000 }, // 16
    {  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500 }, // 17
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000,   0.750000,    0.00000,    0.00000,   0.375000,    0.00000 }, // 18
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.375000,   0.250000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000 }  // 19
  },

  // embedding matrix for child 3
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 0
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 1
    {    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 2
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 3
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 4
    {    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000 }, // 5
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 6
    {    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000 }, // 7
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.750000,   0.375000,   0.250000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 8
    {    0.00000,  -0.125000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 9
    {    0.00000,    0.00000,   0.375000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 10
    {  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,   0.250000,   0.375000,   0.750000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 11
    {  -0.281250,  -0.281250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.156250,  -0.156250,   0.375000,   0.375000,   0.375000,   0.375000,   0.187500,   0.187500,   0.187500,   0.187500,   0.125000,   0.125000,   0.125000,   0.125000 }, // 12
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000 }, // 13
    {    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 14
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.250000,    0.00000 }, // 15
    {  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000 }, // 16
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000,   0.750000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000 }, // 17
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000,   0.250000,    0.00000,    0.00000,   0.375000,    0.00000 }, // 18
    {  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500 }  // 19
  },

  // embedding matrix for child 4
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 0
    {  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000 }, // 1
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 2
    {  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000 }, // 3
    {    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 4
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000 }, // 5
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000 }, // 6
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000 }, // 7
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000,   0.250000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000 }, // 8
    {  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500 }, // 9
    {  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000 }, // 10
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.375000,   0.750000,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000 }, // 11
    {  -0.125000,    0.00000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 12
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000 }, // 13
    {  -0.156250,  -0.156250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.281250,  -0.281250,   0.125000,   0.125000,   0.125000,   0.125000,   0.187500,   0.187500,   0.187500,   0.187500,   0.375000,   0.375000,   0.375000,   0.375000 }, // 14
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.250000,   0.375000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000 }, // 15
    {    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000 }, // 16
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,   0.375000,   0.250000,   0.375000 }, // 17
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.250000,   0.375000,   0.750000 }, // 18
    {    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000 }  // 19
  },

  // embedding matrix for child 5
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000 }, // 0
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 1
    {    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000 }, // 2
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 3
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000 }, // 4
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 5
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000 }, // 6
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000 }, // 7
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000,   0.750000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000 }, // 8
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000,   0.250000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000 }, // 9
    {  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000 }, // 10
    {  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500 }, // 11
    {  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000 }, // 12
    {    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 13
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000 }, // 14
    {  -0.156250,  -0.156250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.281250,  -0.281250,   0.125000,   0.125000,   0.125000,   0.125000,   0.187500,   0.187500,   0.187500,   0.187500,   0.375000,   0.375000,   0.375000,   0.375000 }, // 15
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.125000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000 }, // 16
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000 }, // 17
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.750000,   0.375000,   0.250000 }, // 18
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,   0.375000,   0.250000,   0.375000 }  // 19
  },

  // embedding matrix for child 6
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000 }, // 0
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 1
    {    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000 }, // 2
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 3
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000 }, // 4
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000 }, // 5
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000 }, // 6
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 7
    {  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,   0.187500,   0.125000,   0.187500,   0.375000,   0.375000,   0.125000,   0.125000,   0.375000,   0.187500,   0.125000,   0.187500,   0.375000 }, // 8
    {  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500 }, // 9
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000,   0.750000,    0.00000,    0.00000,   0.375000,    0.00000 }, // 10
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.375000,   0.250000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,   0.375000 }, // 11
    {  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,    0.00000,    0.00000,    0.00000,   0.250000,   0.375000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000 }, // 12
    {  -0.156250,  -0.156250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.281250,  -0.281250,   0.125000,   0.125000,   0.125000,   0.125000,   0.187500,   0.187500,   0.187500,   0.187500,   0.375000,   0.375000,   0.375000,   0.375000 }, // 13
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.750000,    0.00000 }, // 14
    {    0.00000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 15
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.250000,   0.375000,   0.750000 }, // 16
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.250000,   0.375000,   0.750000,   0.375000 }, // 17
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,  -0.125000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000 }, // 18
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000 }  // 19
  },

  // embedding matrix for child 7
  {
    //  0     1     2     3     4     5     6     7    8  9  11  12  13  14  15  16  17  18  19
    {  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000,   0.250000 }, // 0
    {    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000 }, // 1
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 2
    {    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,  -0.250000,  -0.250000,    0.00000,    0.00000,   0.500000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,    0.00000,    0.00000,   0.500000,    0.00000 }, // 3
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.250000,  -0.250000,  -0.250000,  -0.250000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.500000,   0.500000,   0.500000,   0.500000 }, // 4
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000 }, // 5
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 6
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    1.00000,    0.00000 }, // 7
    {  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000 }, // 8
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.250000,   0.750000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000 }, // 9
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,   0.750000,   0.250000,    0.00000,    0.00000,   0.375000,    0.00000 }, // 10
    {  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,   0.125000,   0.187500,   0.375000,   0.187500,   0.125000,   0.125000,   0.375000,   0.375000,   0.125000,   0.187500,   0.375000,   0.187500 }, // 11
    {  -0.156250,  -0.156250,  -0.156250,  -0.156250,  -0.281250,  -0.281250,  -0.281250,  -0.281250,   0.125000,   0.125000,   0.125000,   0.125000,   0.187500,   0.187500,   0.187500,   0.187500,   0.375000,   0.375000,   0.375000,   0.375000 }, // 12
    {    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000 }, // 13
    {    0.00000,    0.00000,  -0.125000,    0.00000,    0.00000,    0.00000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000 }, // 14
    {    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,  -0.187500,  -0.187500,    0.00000,    0.00000,   0.250000,    0.00000,    0.00000,    0.00000,   0.375000,   0.375000,    0.00000,    0.00000,   0.750000,    0.00000 }, // 15
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,   0.750000,   0.375000,   0.250000 }, // 16
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,  -0.125000,   0.375000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000,    0.00000 }, // 17
    {    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.375000,  -0.125000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.750000,    0.00000 }, // 18
    {    0.00000,    0.00000,    0.00000,    0.00000,  -0.187500,  -0.187500,  -0.187500,  -0.187500,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,    0.00000,   0.250000,   0.375000,   0.750000,   0.375000 }  // 19
  }
};

#endif

} // namespace libMesh
