/*******************************************************************************************************************************************
 cmenu.c
*******************************************************************************************************************************************/

#include "cmenu.h"
#include "cimage.h"
#include "ctoolbar.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// hotkey
//-----------------------------------------------------------------------------------------------------------------------------------------
THotKey::THotKey (const SInt16 inKey, const Bool inControl, const Bool inShift, const Bool inAlt)
	:Key	 (inKey),
	 Control (inControl),
	 Shift	 (inShift),
	 Alt	 (inAlt)
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CMenu);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenu::CMenu	  (CContainer *inOwner, const CWidgetListener *inListener)
      :CContainer (inOwner, inListener)
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenu::~CMenu ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CMenu::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CMenu::ChildMustBe () const
{
	return __metaclasses(CMenuItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// maximum children number handling
//-----------------------------------------------------------------------------------------------------------------------------------------
SInt16 CMenu::GetGtkChildrenNumberLeft () const
{
	return -1;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenu::PerformWidgetInitialize ()
{
	// show our job
	Show ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CMenuBarListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuBarListener::CMenuBarListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuBarListener::~CMenuBarListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnSelectionDone
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuBar::OnSelectionDone (GtkMenuShell *inGtkWidget, gpointer inData)
{
	// retreive the gtkol instance
	CMenuBar *inMenuBar = reinterpret_cast <CMenuBar *> (inData);

	// listener notification if any
	if (inMenuBar != NULL && inMenuBar -> GetListener() != NULL)
		static_cast <CMenuBarListener *> (inMenuBar -> GetListener()) -> OnSelectionDone (inMenuBar);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuBar);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuBar::CMenuBar (CContainer *inOwner, const CMenuBarListener *inListener)
	 :CMenu	   (inOwner, inListener)
{
	// widget instanciation process request
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuBar::~CMenuBar ()
{ 
	// widget deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuBar::PerformWidgetInstanciate ()
{
	return ::gtk_menu_bar_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuBar::PerformWidgetInitialize ()
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// signal connection
	if (inGtkWidget != NULL) ::g_signal_connect (G_OBJECT(inGtkWidget), "deactivate", G_CALLBACK(CMenuBar::OnSelectionDone), this);

	// generic call
	CMenu::PerformWidgetInitialize ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container add
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuBar::PerformContainerAdd (CWidget *inChild)
{
	// check the input pointer...
	if (inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// let's see if we got to handle an append or insert request, first get the child in owner index
	SInt16 inIndex = CContainer::GetGtkInOwnerIndex (this, inChild);

	// then, get the gktol widget handled children
	CWidgets inChildren (GetGtkChildren());

	// if an insert is to be handled (i.e. if it is not the only one child and if it has been found in the gtkol components hierarchy
	// and if it is not the last child, modify the gtk gui children sequence accordinaly)
	if ((inChildren.GetLength() > 1) && (inIndex >= 0) && (inIndex < (inChildren.GetLength()-1)))
		::gtk_menu_shell_insert (GTK_MENU_SHELL(GetGtkWidget()), inChild -> GetGtkWidget(), inIndex);

	// append requested
	else
		::gtk_menu_shell_append (GTK_MENU_SHELL(GetGtkWidget()), inChild -> GetGtkWidget());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container remove
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuBar::PerformContainerRemove (CWidget *inChild)
{
	// generic call
	CMenu::PerformContainerRemove (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CMenuBar::ListenerMustBe () const
{
	return __metaclass(CMenuBarListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CMenuItemListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemListener::CMenuItemListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemListener::~CMenuItemListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnSelect
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::OnSelect (GtkItem *inItem, gpointer inData)
{
	// retreive the gtkol instance
	CMenuItem *inMenuItem = reinterpret_cast <CMenuItem *> (inData);

	// send notification to the listener if any
	if (inMenuItem != NULL && inMenuItem -> GetListener() != NULL)
		static_cast <CMenuItemListener *> (inMenuItem -> GetListener()) -> OnSelect (inMenuItem, true);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnDeselect
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::OnDeselect (GtkItem *inItem, gpointer inData)
{
	// retreive the gtkol instance
	CMenuItem *inMenuItem = reinterpret_cast <CMenuItem *> (inData);

	// send notification to the listener if any
	if (inMenuItem != NULL && inMenuItem -> GetListener() != NULL)
		static_cast <CMenuItemListener *> (inMenuItem -> GetListener()) -> OnSelect (inMenuItem, false);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnActivate
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::OnActivate (GtkMenuItem *inItem, gpointer inData)
{
	// retreive the gtkol instance
	CMenuItem *inMenuItem = reinterpret_cast <CMenuItem *> (inData);

	// send notification to the listener if any
	if (inMenuItem != NULL && inMenuItem -> GetListener() != NULL)
		static_cast <CMenuItemListener *> (inMenuItem -> GetListener()) -> OnClick (inMenuItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuItem);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItem::CMenuItem (CMenu *inOwner, const CString &inCaption, const THotKey &inHotKey, const CMenuItemListener *inListener)
	  :CMenu     (inOwner, inListener),
	   m_SubMenu (NULL),
	   m_Caption (inCaption==CString()?"id"+CString(GetId()):inCaption),
	   m_HotKey  (inHotKey)
{
	// widget instanciation request
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItem::~CMenuItem ()
{

	// first of all, the menu item may be associated to a tool button for click event redirection, so get the application whole tool
	// button instances to see if this menu item is referenced...
	CComponents inToolButtons (CComponent::GetComponents (__metaclass(CToolButton)));

	// go through the tool button instances and remove this menu item reference if set
	for (size_t i=inToolButtons.GetLength(), j=0; i>0; i--, j++)
		if (static_cast <CToolButton *> (*inToolButtons[j]) -> GetMenuItem() == this)
			static_cast <CToolButton *> (*inToolButtons[j]) -> SetMenuItem (NULL);

	// widget deletion coherence request
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuItem::PerformWidgetInstanciate ()
{
	// the instanciated widget
	GtkWidget *outGtkWidget = NULL;
	
	// mnemonic ?
	if (m_Caption.Find ('_')) 
		outGtkWidget = ::gtk_menu_item_new_with_mnemonic (m_Caption.Get());
			
	// default instanciation
	else
		outGtkWidget = ::gtk_menu_item_new_with_label (m_Caption.Get());

	// ok
	return outGtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::PerformWidgetInitialize ()
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check...
	if (inGtkWidget == NULL) return;

	// static listeners connections
	::g_signal_connect (G_OBJECT(inGtkWidget), "select",   	 G_CALLBACK(CMenuItem::OnSelect),   this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "deselect", 	 G_CALLBACK(CMenuItem::OnDeselect), this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "activate", 	 G_CALLBACK(CMenuItem::OnActivate), this);

	// hotkey ?
	if (m_HotKey.Key != 0x0000 && CForm::GetAccelGroup() != NULL)
	{
		SInt32 ModifierKey = 0L;
		if (m_HotKey.Control) ModifierKey |= GDK_CONTROL_MASK;
		if (m_HotKey.Shift  ) ModifierKey |= GDK_SHIFT_MASK;
		if (m_HotKey.Alt    ) ModifierKey |= GDK_MOD1_MASK;
		::gtk_widget_add_accelerator (inGtkWidget, "activate", CForm::GetAccelGroup(), m_HotKey.Key, (GdkModifierType)ModifierKey,
					      GTK_ACCEL_VISIBLE);
	}

	// generic call
	CMenu::PerformWidgetInitialize ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container child add
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::PerformContainerAdd (CWidget *inChild)
{
	// pointer in check
	if (inChild == NULL) return;

	// child gtk widget
	GtkWidget *inChildGtkWidget = inChild -> GetGtkWidget();

	// pointer check
	if (inChildGtkWidget == NULL) return;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// if the sub menu of this menu item is not defined yet
	if (m_SubMenu == NULL)
	{
		// instanciate the submenu item
		m_SubMenu = ::gtk_menu_new ();

		// submenu association
		::gtk_menu_item_set_submenu (GTK_MENU_ITEM(inGtkWidget), m_SubMenu);
		::gtk_menu_set_accel_group  (GTK_MENU(m_SubMenu), CForm::GetAccelGroup());
	}

	// the PerformContainerAdd request may be called inderictly by the CWidget::SetOwner, i.e. the specified inChild instance may not
	// be the last one because the gtk widget may not have been created while the gtkol associated widget were instanciated; so we have
	// get the in owner index of the specified child to see if we have to insert or append the widget and to respect the gtkol component
	// hierarchy representation
	SInt16 inIndex = CComponent::GetInOwnerIndex (this, inChild, __metaclass(CMenuItem));

	// get the children we are supposed to handle
	CComponents inChildren (GetChildren (__metaclass(CMenuItem)));

	// see if we have to insert or append the menu item i.e. if it is the last one of our children
	if ((inChildren.GetLength() > 1) && (inIndex >= 0) && (inIndex < inChildren.GetLength()-1))
		::gtk_menu_shell_insert (GTK_MENU_SHELL(m_SubMenu), inChildGtkWidget, inIndex);
	else
		::gtk_menu_shell_append (GTK_MENU_SHELL(m_SubMenu), inChildGtkWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container child remove
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::PerformContainerRemove (CWidget *inChild)
{
	// check the inChild pointer
	if (inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// check the submenu pointer
	if (m_SubMenu != NULL) ::gtk_container_remove (GTK_CONTAINER(m_SubMenu), inChild -> GetGtkWidget());

	// get the children items if any
	CComponents inChildren (GetChildren(__metaclass(CMenuItem)));

	// the submenu is automatically deleted if empty, do not point on an empty address
	if (inChildren.GetLength() == 1 && inChild == *inChildren[0]) m_SubMenu = NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CMenuItem::ListenerMustBe () const
{
	return __metaclass(CMenuItemListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CMenuItem::OwnerMustBe () const
{
	return __metaclasses(CMenu);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// bounds
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::SetBounds (const TBounds &inBounds)
{
	// no gui bounds affectation at this level, let the menu item manage itself as this function may be called by the CControl/CWidget
	// serialization process, so just keep a local copy of the bounds and do not redirect the request to the gtk api i.e. bypass
	// the CWidget gui handling level
	CControl::SetBounds (inBounds);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// menu item pixbuf representation
//-----------------------------------------------------------------------------------------------------------------------------------------
CPixbuf * CMenuItem::GetControlPixbuf () const
{
	// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	// !!! no menu item pixbuf representation, TODO !!!
	// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	return NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::SetCaption (const CString &inCaption)
{
	// local copy of the caption
	m_Caption = inCaption;

	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();
	
	// pointer check
	if (inGtkWidget == NULL) return;

	// get the menu item associated label
	GtkWidget *inGtkLabel = ::gtk_bin_get_child (GTK_BIN(inGtkWidget));

	// pointer check
	if (inGtkLabel == NULL) return;

	// mnemonic use ?
	if (inCaption.Find ('_')) 
		::gtk_label_set_use_underline (GTK_LABEL(inGtkLabel), true); 
	else 
		::gtk_label_set_use_underline (GTK_LABEL(inGtkLabel), false); 

	// caption affectation
	::gtk_label_set_label (GTK_LABEL(inGtkLabel), m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption access
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CMenuItem::GetCaption () const
{
	// the caption may have been modified with gtk api directly, so retreive the gtk widget to get its caption rather than reading
	// directly the local copy of the caption that is only used in the instanciation process; get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();
	
	// pointer check
	if (inGtkWidget == NULL) return m_Caption;
	
	// get the menu item associated label
	GtkWidget *inGtkLabel = ::gtk_bin_get_child (GTK_BIN(inGtkWidget));

	// pointer check
	if (inGtkLabel == NULL) return m_Caption;

	// get the label text
	const_cast <CString &> (m_Caption) = CString(::gtk_label_get_label (GTK_LABEL(inGtkLabel)));

	// ok
	return m_Caption;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// hotkey affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::SetHotKey (const THotKey &inHotKey)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// pointers check 
	if (inGtkWidget == NULL || CForm::GetAccelGroup() == NULL) return;

	// modifiers keys
	SInt32 ModifierKey = 0L;

	// old one removal if any
	if (m_HotKey.Key != 0x0000)
	{
		if (m_HotKey.Control) ModifierKey |= GDK_CONTROL_MASK;
		if (m_HotKey.Shift  ) ModifierKey |= GDK_SHIFT_MASK;
		if (m_HotKey.Alt    ) ModifierKey |= GDK_MOD1_MASK;
		::gtk_widget_remove_accelerator (inGtkWidget, CForm::GetAccelGroup(), m_HotKey.Key, (GdkModifierType)ModifierKey);
	}
	
	// local copy of the hotkey value
	m_HotKey = inHotKey;

	// new one affectation if any
	if (m_HotKey.Key != 0x0000)
	{
		ModifierKey = 0L;
		if (m_HotKey.Control) ModifierKey |= GDK_CONTROL_MASK;
		if (m_HotKey.Shift  ) ModifierKey |= GDK_SHIFT_MASK;
		if (m_HotKey.Alt    ) ModifierKey |= GDK_MOD1_MASK;
		::gtk_widget_add_accelerator (inGtkWidget, "activate", CForm::GetAccelGroup(), m_HotKey.Key, (GdkModifierType)ModifierKey,
					      GTK_ACCEL_VISIBLE);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// hotkey access
//-----------------------------------------------------------------------------------------------------------------------------------------
THotKey CMenuItem::GetHotKey () const
{
	// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	// ok, we would have to get the actual hotkey from the gtk api because those values may be modified outside the gtkol api, think
	// about that, TODO...
	// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	return m_HotKey;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnClick event sender
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::Click ()
{
	if (GetGtkWidget() != NULL) ::gtk_menu_item_activate (GTK_MENU_ITEM(GetGtkWidget()));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItem::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CMenu::Serialize (ioXMLElementNode, inMode);

	// serialization reuest analyse
	switch (inMode)
	{
		// xml dump	
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_MENUITEM_ELEMENT);

			// get the menu item hot key
			THotKey inHotKey (GetHotKey());
			CString outCombinaison (inHotKey.Control ? (inHotKey.Alt || inHotKey.Shift) ? "ctrl+" : "ctrl" : NULL); 
			outCombinaison += inHotKey.Alt   ? inHotKey.Shift ? "alt+" : "alt"  : "";
			outCombinaison += inHotKey.Shift ? "shift" : "";
 
			// add the menu item attributes
			newXMLElement -> AddAttribute (XML_MENUITEM_ATTR_CAPTION, GetCaption());			
			if (inHotKey.Key != 0x0000)
			{
				newXMLElement -> AddAttribute (XML_MENUITEM_ATTR_HOTKEY_KEY, CString((SInt32)inHotKey.Key));
				newXMLElement -> AddAttribute (XML_MENUITEM_ATTR_HOTKEY_COMBINAISON, outCombinaison);
			}

			// modify the in/out pointer so that the potential derived serialization process continues under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the xml node this instance is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_MENUITEM_ELEMENT);

			// check we got an expected cmenuitem node
			if (inXMLNode == NULL)
				throw new CException (CString("CMenuItem::Serialize, specified xml node is not a \"") + 
							      XML_MENUITEM_ELEMENT + CString("\" element one."), __exception(XMLPARSE));
			// modify the in/out xml node
			ioXMLElementNode = inXMLNode;

			// set the menu item caption from the appropriate xml attribute
			SetCaption (::xml_node_get_attribute (inXMLNode, XML_MENUITEM_ATTR_CAPTION).GetValue());

			// get the menu item hotkey value if any
			THotKey inHotKey (::xml_node_get_attribute (inXMLNode, XML_MENUITEM_ATTR_HOTKEY_KEY).GetValue().ToULong());

			// get the nemu item hot key combinaison if any, '\0' otherwise
			CString inCombinaison (::xml_node_get_attribute (inXMLNode, XML_MENUITEM_ATTR_HOTKEY_COMBINAISON).GetValue());

			// combinaison analyse if any...
			if (inCombinaison.Find (CString("ctrl" ))) inHotKey.Control = true;
			if (inCombinaison.Find (CString("shift"))) inHotKey.Shift   = true;
			if (inCombinaison.Find (CString("alt"  ))) inHotKey.Alt	    = true;

			// set the associated hotkey if defined (check in the function itself)
			SetHotKey (inHotKey);
		}
		break;
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuItemSeparator);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemSeparator::CMenuItemSeparator (CMenu *inOwner)
		   :CMenuItem	       (NULL, CString(), _HKNone_, NULL)
{
	// gtk widget instanciation
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemSeparator::~CMenuItemSeparator ()
{
	// widget deletion coherence
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuItemSeparator::PerformWidgetInstanciate ()
{
	return ::gtk_separator_menu_item_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemSeparator::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// do not consider the CMenuItem serialization part...
	CMenu::Serialize (ioXMLElementNode, inMode);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuItemCheck);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemCheck::CMenuItemCheck (CMenu *inOwner, const CString &inCaption, const THotKey &inHotKey, const CMenuItemListener *inListener)
	       :CMenuItem       (NULL, inCaption, inHotKey, inListener)
{
	// gtk widget instanciation process request
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemCheck::~CMenuItemCheck ()
{
	// gtk widget deletion coherence request
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuItemCheck::PerformWidgetInstanciate ()
{
	// out gtk widget
	GtkWidget *outGtkWidget = NULL;
	
	// mnemonic ?
	if (m_Caption.Find ('_')) 
		outGtkWidget = ::gtk_check_menu_item_new_with_mnemonic (m_Caption.Get());
	
	// instanciation par défaut
	else
		outGtkWidget = ::gtk_check_menu_item_new_with_label (m_Caption.Get());

	// ok
	return outGtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// check
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemCheck::Check (const bool inCheck)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// affectation de la coche sans émission de signal
	::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CMenuItem::OnActivate), this);
	::gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(inGtkWidget), inCheck);
	::g_signal_connect (G_OBJECT(inGtkWidget), "activate", G_CALLBACK(CMenuItem::OnActivate), this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// is checked ?
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CMenuItemCheck::IsChecked () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok ?
	return inGtkWidget != NULL ? ::gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemCheck::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CMenuItem::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_MENUITEMCHECK_ELEMENT);

			// add our attribute
			newXMLElement -> AddAttribute (XML_MENUITEMCHECK_ATTR_CHECKED, CString(IsChecked()?"true":"false"));

			// modify the in/out pointer so that the overwritten serialization process continues inder the newly created node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the xml node this instance is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_MENUITEMCHECK_ELEMENT);

			// check we got an expected cmenuitem node
			if (inXMLNode == NULL)
				throw new CException (CString("CMenuItemCheck::Serialize, specified xml node is not a \"") + 
							      XML_MENUITEMCHECK_ELEMENT + CString("\" element one."),
						      __exception(XMLPARSE));

			// modify the in/out xml node
			ioXMLElementNode = inXMLNode;

			// set the check propertie of this instance from the xml attribute
			Check (::xml_node_get_attribute (inXMLNode, XML_MENUITEMCHECK_ATTR_CHECKED).GetValue().ToBool());
		}
		break;
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuItemRadio);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemRadio::CMenuItemRadio (CMenu *inOwner, const CString &inCaption, const THotKey &inHotKey, const CMenuItemListener *inListener)
	       :CMenuItemCheck (NULL, inCaption, inHotKey, inListener)
{
	// gtk widget instanciation process request
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemRadio::~CMenuItemRadio ()
{
	// gtk widget deletion coherence
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// specific gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuItemRadio::PerformWidgetInstanciate ()
{
	// out gtk widget
	GtkWidget *outGtkWidget = NULL;

	// the potential gtk radio group this radio menu should be added to
	GSList *inMenuItemRadioGroup = NULL;

	// get our siblings, be sure to consider only CMenuItem instances...
	CComponents inMenuItems (GetSiblings (__metaclass(CMenuItem)));

	// go through the siblings
	for (size_t i=inMenuItems.GetLength(), j=0; i>0; i--, j++)
	{
		// is it our gtkol instance ?
		if ((*inMenuItems[j]) == this) break;

		// get the gtk group of the menu item if it is radio one
		if ((*inMenuItems[j]) -> ClassIs (__metaclass(CMenuItemRadio)))
			inMenuItemRadioGroup = ::gtk_radio_menu_item_get_group 
				(GTK_RADIO_MENU_ITEM (static_cast <CMenuItemRadio *> (*inMenuItems[j]) -> GetGtkWidget()));

		// if the menu item is a logical menu items separator, consider a new menu item radio group and do not use the previous one
		if ((*inMenuItems[j]) -> ClassIs (__metaclass(CMenuItemSeparator))) inMenuItemRadioGroup = NULL;
	}

	// mnemonic specification ?
	if (m_Caption.Find ('_')) 
		outGtkWidget = ::gtk_radio_menu_item_new_with_mnemonic (inMenuItemRadioGroup, m_Caption.Get());
	
	// default instanciation
	else
		outGtkWidget = ::gtk_radio_menu_item_new_with_label (inMenuItemRadioGroup, m_Caption.Get());

	// ok
	return outGtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// radio check
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemRadio::Check (const bool inCheck)
{
	// our gtk widget 
	GtkWidget *inGtkWidget = GetGtkWidget();
	
	// be paranoïd one more time...
	if (inGtkWidget == NULL) return;

	// get our sibling
	CComponents inChildren (GetSiblings (__metaclass(CMenuItemRadio)));
	
	// remove all of the signals connections, do not consider groups...
	for (size_t i=inChildren.GetLength(); i>0; i--)
		::g_signal_handlers_disconnect_by_func 
			(G_OBJECT(static_cast <CMenuItemRadio *> (*inChildren[i-1]) -> GetGtkWidget()), 
			 	(void*)G_CALLBACK(CMenuItem::OnActivate), *inChildren[i-1]);
		
	// check on our gtk widget
	::gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(inGtkWidget), inCheck);
	
	// all signals connection
	for (size_t i=inChildren.GetLength(); i>0; i--)
		::g_signal_connect 
			(G_OBJECT(static_cast <CMenuItemRadio *> (*inChildren[i-1]) -> GetGtkWidget()), "activate", 
			 	G_CALLBACK(CMenuItem::OnActivate), *inChildren[i-1]);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuItemImage);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemImage::CMenuItemImage (CMenu *inOwner, const CString &inCaption, const THotKey &inHotKey, const CMenuItemListener *inListener)
	       :CMenuItem      (NULL, inCaption, inHotKey, inListener),
		m_StockId      ()
{
	// launch the gtkol affectation / gtk instanciation process
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemImage::CMenuItemImage (CMenu *inOwner, const char *inStockId, const CMenuItemListener *inListener)
	       :CMenuItem      (NULL, CString(), _HKNone_, inListener),
		m_StockId      (inStockId)
{
	// launch the gtkol affectation / gtk instanciation process
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuItemImage::~CMenuItemImage ()
{
	// gtk widget deletion coherence
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// menu item instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuItemImage::PerformWidgetInstanciate ()
{
	// the instanciated widget
	GtkWidget *outGtkWidget = NULL;
	
	// mnemonic ?
	if (m_Caption.Find ('_')) 
		outGtkWidget = ::gtk_image_menu_item_new_with_mnemonic (m_Caption.Get());
		
	// stock id specified
	else if (m_StockId != CString())
		outGtkWidget = ::gtk_image_menu_item_new_from_stock (m_StockId.Get(), CForm::GetAccelGroup());

	// default instanciation
	else
		outGtkWidget = ::gtk_image_menu_item_new_with_label (m_Caption.Get());

	// ok
	return outGtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// child addon
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemImage::PerformContainerAdd (CWidget *inChild)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inChild == NULL || inChild -> GetGtkWidget() == NULL || inGtkWidget == NULL) return;

	// instance check, is it an image ?
	if (inChild -> ClassIs (__metaclass(CImage)))

		// gtk property affectation
		::gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM(inGtkWidget), inChild -> GetGtkWidget());

	// it is not an image...
	else
		// generic call
		CMenuItem::PerformContainerAdd (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// child removal
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemImage::PerformContainerRemove (CWidget *inChild)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inChild == NULL || inGtkWidget == NULL) return;

	// instance check, is it an image ?
	if (inChild -> ClassIs (__metaclass(CImage)))

		// gtk property affectation
		::gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM(inGtkWidget), NULL);

	// it is not an image...
	else
		// generic call
		CMenuItem::PerformContainerRemove (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children types
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CMenuItemImage::ChildMustBe () const
{
	return __metaclasses(CMenuItem) + __metaclass(CImage);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuItemImage::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// generic call first
			CMenuItem::Serialize (ioXMLElementNode, inMode);

			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_MENUITEMIMAGE_ELEMENT);

			// append the stock id attribute if any
			if (m_StockId != CString()) newXMLElement -> AddAttribute (XML_MENUITEMIMAGE_ATTR_STOCK_ID, m_StockId);

			// change the current xml node for potential overloaded serializations
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// we first have to find the menu item image node because the stock id must be filled in before launching the
			// gtk instanciation process
                        CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_MENUITEMIMAGE_ELEMENT);

                        // check we got an expected node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CComboBox::Serialize, specified xml node is not a \"") +
                                                              XML_MENUITEMIMAGE_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// retreive the stock id specification if any
			m_StockId = ::xml_node_get_attribute (inXMLNode, XML_MENUITEMIMAGE_ATTR_STOCK_ID).GetValue();

			// we got the potential stock id specification, so launch the generic process
			CMenuItem::Serialize (ioXMLElementNode, inMode);

			// change the current xml node so that overloaded definitions would continue under this definition
			ioXMLElementNode = inXMLNode;
		}
		break;
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CMenuPopupListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuPopupListener::CMenuPopupListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuPopupListener::~CMenuPopupListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnSelectionDone
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuPopup::OnSelectionDone (GtkMenuShell *inGtkWidget, gpointer inData)
{
	// retreive the gtkol instance
	CMenuPopup *inMenuPopup = reinterpret_cast <CMenuPopup *> (inData);

	// pointer check
	if (inMenuPopup == NULL) return;

	// listener notification if any
	if (inMenuPopup -> GetListener() != NULL)
		static_cast <CMenuPopupListener *> (inMenuPopup -> GetListener()) -> OnSelectionDone (inMenuPopup);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CMenuPopup);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuPopup::CMenuPopup (CContainer *inOwner, const CMenuPopupListener *inListener)
	   :CMenu      (inOwner, inListener)
{
	// the menu popup container does not care about serializing d&d capabilites either handling the serialization shown attribute, so 
	// change the generic attributes to avoid those handling
	CControl::m_SerializeDnD   = false;
	CWidget ::m_SerializeShown = false;

	// gtk widget instanciation process request (no container process handling, all signal connections); the menu popup instance is 
	// not added on the gui container interface, it is just handled by the gtkol widget engine itself avoiding gtk errors when trying
	// to add the popup context as a container to a gtk bin that might have already an assigned child
	CWidget::m_PerformContainerProcess = false; if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CMenuPopup::~CMenuPopup ()
{
	// gtk widget coherence deletion requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CMenuPopup::PerformWidgetInstanciate ()
{
	// widget instanciation
	return ::gtk_menu_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuPopup::PerformWidgetInitialize ()
{
	// signal connection
	if (GetGtkWidget() != NULL) 
		::g_signal_connect (G_OBJECT(GetGtkWidget()), "deactivate", G_CALLBACK(CMenuPopup::OnSelectionDone), this);

	// generic call
	CMenu::PerformWidgetInitialize ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container child add
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuPopup::PerformContainerAdd (CWidget *inChild)
{
	// pointer check
	if (inChild == NULL || inChild -> GetGtkWidget() == NULL || GetGtkWidget() == NULL) return;

	// let's see if we got to handle an append or insert request, first get the child in owner index
	SInt16 inIndex = CContainer::GetGtkInOwnerIndex (this, inChild);

	// then, get the gktol widget handled children
	CWidgets inChildren (GetGtkChildren());

	// if an insert is to be handled (i.e. if it is not the only one child and if it has been found in the gtkol components hierarchy
	// and if it is not the last child, modify the gtk gui children sequence accordinaly)
	if ((inChildren.GetLength() > 1) && (inIndex >= 0) && (inIndex < (inChildren.GetLength()-1)))
		::gtk_menu_shell_insert (GTK_MENU_SHELL(GetGtkWidget()), inChild -> GetGtkWidget(), inIndex);

	// append requested
	else
		::gtk_menu_shell_append (GTK_MENU_SHELL(GetGtkWidget()), inChild -> GetGtkWidget());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// container child remove
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuPopup::PerformContainerRemove (CWidget *inChild)
{
	// generic call...
	CMenu::PerformContainerRemove (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CMenuPopup::ListenerMustBe () const
{
	return __metaclass(CMenuPopupListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// bounds affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuPopup::SetBounds (const TBounds &inBounds)
{
	// no gui handling
	CControl::SetBounds (inBounds);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// popup sender
//-----------------------------------------------------------------------------------------------------------------------------------------
CControl * CMenuPopup::GetPopupSender () const
{
	return m_Sender;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// popup point
//-----------------------------------------------------------------------------------------------------------------------------------------
TPoint CMenuPopup::GetPopupRelativePoint () const
{
	return m_RelativePoint;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// popup
//-----------------------------------------------------------------------------------------------------------------------------------------
void CMenuPopup::Popup ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL) return;

	// do popup
	::gtk_menu_popup (GTK_MENU(inGtkWidget), NULL, NULL, NULL, NULL, 3, ::gtk_get_current_event_time());
}

