/******************************************************************************************************************************************
 ciconview.h

 CClass						CClass
 |--CSerialized					|--CSerialized
    |--CMetaModule				   |--CMetaModule
       |--CObject				      |--CObjectListener
          |--CComponent				         |
             |--CControl				 |
                |--CWidget				 |--CWidgetListener
                |  |--CContainer			    |
		|     |--CLayout			    |
                |     	 |--CIconView			    |--CIconViewListener
                |--CIconViewItem

 CItemFieldValuePixbuf + CItemFieldValueString FIXED model display into an icon view. As the item definition derives the CControl abstract,
 items are draggable components. CIconView derives the CLayout so there would be an intelligent d&d index calculation when performing
 such requests.

 => The field value attribute of the CItemFieldValue derived classes are the only one handled, any other attributes are not <=
******************************************************************************************************************************************/

#ifndef __CICONVIEW_H__
#define __CICONVIEW_H__

#include "clayout.h"
#include "citemfieldvalue.h"

// what's defined here
class CIconView;				// the icon view widget
class CIconViewListener;			// the icon view listener api
class CIconViewItem;				// the icon view item 
class CIconViewFieldValueAPIListener;		// the icon view item field values listener (internally used)

//-----------------------------------------------------------------------------------------------------------------------------------------
// define an icon view items buffer list
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef NServices::TBuffer <CIconViewItem *> CIconViewItems;

//-----------------------------------------------------------------------------------------------------------------------------------------
// ciconviewitem xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ciconviewitem>
//   <fields>
//     ...
//   </fields>
// </ciconviewitem>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_ICONVIEWITEM_ELEMENT		("ciconviewitem");
static CString XML_ICONVIEWITEM_FIELDS_ELEMENT	("fields");

//-----------------------------------------------------------------------------------------------------------------------------------------
// ciconview xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ciconview selection="single|multiple|browse|none" selected="[UInt32[,UInt32]]" orientation="horizontal|vertical" editable="true|false" 
//	      columns="SInt16" item-width="SInt16" spacing="UInt16" row-spacing="UInt16" col-spacing="UInt16" margin="UInt16">
//   ...
// </ciconview>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_ICONVIEW_ELEMENT		("ciconview");
static CString XML_ICONVIEW_ATTR_SELECTION_MODE	("selection");
static CString XML_ICONVIEW_ATTR_SELECTED	("selected");
static CString XML_ICONVIEW_ATTR_ORIENTATION	("orientation");
static CString XML_ICONVIEW_ATTR_EDITABLE	("editable");
static CString XML_ICONVIEW_ATTR_COLUMNS	("columns");
static CString XML_ICONVIEW_ATTR_ITEM_WIDTH	("item-width");
static CString XML_ICONVIEW_ATTR_SPACING	("spacing");
static CString XML_ICONVIEW_ATTR_ROW_SPACING	("row-spacing");
static CString XML_ICONVIEW_ATTR_COL_SPACING	("col-spacing");
static CString XML_ICONVIEW_ATTR_MARGIN		("margin");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CIconViewItem class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CIconViewItem : public CControl
{
	// instanciation section
	public :

		CIconViewItem			(CIconView *inOwner=NULL, const CItemFieldValues &inFieldValues=CItemFieldValues(),
						 const CObjectListener *inListener=NULL) THROWABLE;
		CIconViewItem			(CIconView *inOwner, const CItemFieldValuePixbuf *, const CItemFieldValueString *,
						 const CObjectListener *inListener=NULL);
		CIconViewItem			(CIconView *inOwner, const CPixbuf *, const CString &, 
						 const CObjectListener *inListener=NULL);
		virtual ~CIconViewItem		();

	// gtkol icon view item / gtk tree view item correspondance
	public :
	
		// get the gtk tree iter associated to the gtkol icon item instance
		GtkTreeIter *			GetGtkTreeIter			() const;

		// get the gtkol icon view item of specified gtk tree iter if any
		static CIconViewItem *		GetIconViewItem			(const GtkTreeIter *inGtkTreeIter);

	// CComponent redefinition
	public :

		// expected owner type : CIconView
		virtual CMetaClasses		OwnerMustBe			() const;

		// owner affectation, must be derived from CIconView
		virtual bool			SetOwner			(CComponent *inOwner, const SInt16 inIndex=-1);

	// CControl redefinition
	public :

		// retreive the icon view item bounds
		virtual TBounds			GetBounds			() const;

		// get the icon view item pixbuf representation, used for friendly d&d representations
		virtual CPixbuf *		GetControlPixbuf		() const;

	// icon view item field values access
	public :

		// item field values access
		virtual bool			SetItemFieldValues		(const CItemFieldValues &inItemFieldValues);
		CItemFieldValues		GetItemFieldValues		() const;

		// 0 indexed field value direct access, no check on range so be carefull, index may be 0 or 1
		CItemFieldValue &		operator []			(const size_t) const;

		// field value explicit access
		CItemFieldValuePixbuf &		GetItemFieldValuePixbuf		() const;
		CItemFieldValueString &		GetItemFieldValueString		() const;

	// icon view item specifics
	public :

		// icon view item selection
		virtual void			Select				(const bool inSelect=true);
		virtual bool			IsSelected			() const;

		// activate the icon view item
		virtual void			Activate			();

	// CSerialized redefinition
	public :

		// icon view item xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

	// protected attributes
	protected :

		// gtk tree iter handled attribute
		GtkTreeIter			m_GtkTreeIter;

		// icon view item field values
		CItemFieldValues		m_ItemFieldValues;

		// some friends of us...
		friend class			CIconView;

		// associated metaclass
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('icvi', CIconViewItem, CControl);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CIconViewListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CIconViewListener : public CWidgetListener
{
	// instanciation section
	public :

		CIconViewListener		();
		virtual ~CIconViewListener	() =0;

	// specific listener api
	public :

		// called when the specified icon view item of the inSender icon view has been activated i.e. double clicked
		virtual void			OnItemActivated			(CObject *inSender, CObject *inItem)			{ }

		// called when the selection of the specified inSender icon view has changed
		virtual void			OnItemSelectionChanged		(CObject *inSender)					{ }
	
		// called when the specified string field value of the inSender's icon view item is about to be modified after beeing
		// edited - may be called only when the icon item is an editable one -
		virtual void			OnItemEdited			(CObject *inSender, CObject *inItem, CString &ioNewString,
										 Bool &ioDoChange)					{ }

		// associated metaclass
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_icw', CIconViewListener, CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CIconView class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CIconView : public CLayout
{
	// instanciation section
	public :

		CIconView			(CContainer *inOwner=NULL, const CIconViewListener *inListener=NULL);
		virtual ~CIconView		();

	// protected specific gtk widget requests handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// CObject redefinition
	protected :

		// listener affectation, must be derived from CIconViewListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// CComponent redefinition
	public :

		// expected owner types : CContainer, expected children types : CIconViewItem
		virtual CMetaClasses		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

	// CControl / CWidget redefinition
	public :

		// get the icon view item owned at the specified relative point
		virtual CControl *		GetControlAtPoint		(const TPoint &inRelativePoint) const;

	// CLayout redefinition
	public :

		// the layout may be analysed in dnd processes as an indirect drop site target, this function returns the insertion
		// index of a given potential new child inCandidate from layout's relative coordinates specifications
		virtual SInt16			GetDropIndexAtPoint		(const TPoint &inRelativePoint, 
										 const CControl *inCandidate) const;

	// icon view specifics
	public :

		// editable state of the icon items
		virtual void			SetEditable			(const bool inEditable);
		virtual bool			GetEditable			() const;

		// icon view selection mode access
		virtual void			SetSelectionMode		(const GtkSelectionMode inSelectionMode);
		GtkSelectionMode		GetSelectionMode		() const;

		// current icon view selection, returns the list of the currently selected items if any
		CIconViewItems			GetSelection			() const;

		// determines whether the labels are drawn beside the icons instead of below
		virtual void			SetOrientation			(const GtkOrientation inGtkOrientation);
		GtkOrientation			GetOrientation			() const;

		// icon view maximum columns number, if -1, the icon view will automatically determine a suitable number 
		virtual void			SetColumns			(const SInt16 inNumber);
		SInt16				GetColumns			() const;

		// the width to use for each item. if -1, the icon view will automatically determine a suitable item size
		virtual void			SetItemWidth			(const SInt16 inItemWidth);
		SInt16				GetItemWidth			() const;

		// the spacing property specifies the space which is inserted between the cells (the icon and the text) of an item
		virtual void			SetSpacing			(const UInt16 inSpacing);
		UInt16				GetSpacing			() const;

		// the space which is inserted between the rows of the icon view
		virtual void			SetRowSpacing			(const UInt16 inRowSpacing);
		UInt16				GetRowSpacing			() const;

		// the space which is inserted between the columns of the icon view
		virtual void			SetColSpacing			(const UInt16 inColSpacing);
		UInt16				GetColSpacing			() const;
		
		// the extra space which is inserted at the top, bottom, left and right of the icon view
		virtual void			SetMargin			(const UInt16 inMargin);
		UInt16				GetMargin			() const;

	// CSerialized redefinition
	public :

		// tree view xml serialization handling
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;
	// protected static listening section
	protected :

		static void			OnItemActivated			(GtkIconView *, GtkTreePath *, gpointer);
		static void			OnItemSelectionChanged		(GtkIconView *, gpointer);
		static void			OnItemEdited			(GtkCellRendererText *, const gchar *, const gchar *,
										 gpointer);

		bool				m_Editable;
		GtkCellRenderer	*		m_GtkCellRendererText;

		// associated metaclass
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('icvw', CIconView, CLayout);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CIconViewItemFieldValueAPIListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CIconViewItemFieldValueAPIListener : public CItemFieldValueAPIListener
{
	// public handling
	public :

		// called when the developper explicitely requests some modification on the item field value api
		virtual void			OnStateChange			(CObject *);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('_ipi', CIconViewItemFieldValueAPIListener, CItemFieldValueAPIListener);

#endif
