/*--------------------------------------------------------------------------+
$Id: OptionsTest.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.options;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import junit.framework.TestCase;
import edu.tum.cs.commons.options.Options.ValueConversionException;

/**
 * JUnit test for <code>Options</code> class.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * 
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 5980F83C5BFD443EB00DEFAB4F21C6E7
 */
public class OptionsTest extends TestCase {

	/** Test enumeration */
	private enum Colors {
		/***/
		RED, /***/
		GREEN, /***/
		BLUE, /***/
		BROWN_RED
	}

	/** Test content. */
	private static final String[] testContent = {
			"# Input file for JUnit test", "", "# Option without value",
			"param0", "", "# used by testGetValue()", "param1    = test1", "",
			"# used by testGetValue() and testGetValues()",
			"paramList = test1 test2  \"test   3\" test4", "",
			"# used by testHasBooleanValue()", "bool1 = true", "bool2 = false",
			"bool3 = notBool", "", "# used by testHasIntValue()", "int1 = 33",
			"int2 = -13", "int3 = notInt", "enum1 = RED", "enum2= red",
			"enum3=noColor", "enum4 = brown-red", "enum5=brown_red" };

	/** Test file. Created by {@link #createTestFile()}. */
	private File inputFile;

	/** Options under test. */
	private Options options;

	/**
	 * Set up test fixture.
	 */
	@Override
	public void setUp() throws IOException {
		options = new Options();
		createTestFile();
		options.init(inputFile.getAbsolutePath());
	}

	/**
	 * Delete temporary file.
	 */
	@Override
	public void tearDown() {

		inputFile.delete();
	}

	/**
	 * Create a temporary input file.
	 * 
	 * @throws IOException
	 * 
	 */
	private void createTestFile() throws IOException {
		inputFile = new File("options.test.tmp");

		BufferedWriter out = new BufferedWriter(new FileWriter(inputFile));
		for (String element : testContent) {
			out.write(element);
			out.newLine();
		}

		out.close();
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testHasOption() {
		boolean value = options.hasOption("bool1");
		assertTrue(value);

		value = options.hasOption("param0");
		assertTrue(value);

		value = options.hasOption("not_present");
		assertFalse(value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testHasValue() {
		boolean value = options.hasValue("bool1");
		assertTrue(value);

		value = options.hasValue("param0");
		assertFalse(value);

		value = options.hasValue("not_present");
		assertFalse(value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testCountValues() {
		int count = options.countValues("param0");
		assertEquals(0, count);

		count = options.countValues("param1");
		assertEquals(1, count);

		count = options.countValues("paramList");
		assertEquals(4, count);

		count = options.countValues("not_present");
		assertEquals(Options.OPTION_NOT_PRESENT, count);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testGetValue() {
		String value = options.getValue("param1");
		String expected = "test1";
		assertEquals(expected, value);

		value = options.getValue("not_present", "test1");
		assertEquals(expected, value);

		value = options.getValue("paramList");
		expected = "test1 test2  \"test   3\" test4";
		assertEquals(expected, value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testGetValues() {
		String[] values = options.getValues("paramList");
		assertEquals("test1", values[0]);
		assertEquals("test2", values[1]);
		assertEquals("test   3", values[2]);
		assertEquals("test4", values[3]);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testHasBooleanValue() {
		boolean value = options.hasBooleanValue("bool1");
		assertTrue(value);

		value = options.hasBooleanValue("bool2");
		assertTrue(value);

		value = options.hasBooleanValue("bool3");
		assertFalse(value);

		value = options.hasBooleanValue("not_present");
		assertFalse(value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testGetBooleanValue() throws ValueConversionException {
		boolean value = options.getBooleanValue("bool1");
		assertTrue(value);

		value = options.getBooleanValue("bool2");
		assertFalse(value);

		try {
			value = options.getBooleanValue("bool3");
			fail();
		} catch (Options.ValueConversionException ex) {
			// Nothing.
		}

		value = options.getBooleanValue("bool3", false);
		assertFalse(value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testHasIntValue() {
		boolean value = options.hasIntValue("int1");
		assertTrue(value);

		value = options.hasIntValue("int2");
		assertTrue(value);

		value = options.hasIntValue("int3");
		assertFalse(value);

		value = options.hasIntValue("not_present");
		assertFalse(value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 * 
	 * @exception ValueConversionException
	 */
	public void testGetIntValue() throws ValueConversionException {
		int value = options.getIntValue("int1");
		assertEquals(33, value);

		value = options.getIntValue("int2");
		assertEquals(-13, value);

		try {
			value = options.getIntValue("int3");
			fail();
		} catch (Options.ValueConversionException ex) {
			// This is expected.
		}

		value = options.getIntValue("bool3", 55);
		assertEquals(55, value);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 * 
	 * @exception ValueConversionException
	 */
	public void testGetEnumValue() throws ValueConversionException {

		Colors actual = options.getEnumValue("enum1", Colors.class);

		assertEquals(Colors.RED, actual);

		actual = options.getEnumValue("enum2", Colors.class);
		assertEquals(Colors.RED, actual);

		try {
			actual = options.getEnumValue("enum3", Colors.class);
			fail();
		} catch (Options.ValueConversionException ex) {
			// This is expected.
		}

		actual = options.getEnumValue("enum4", Colors.class);
		assertEquals(Colors.BROWN_RED, actual);

		actual = options.getEnumValue("enum5", Colors.class);
		assertEquals(Colors.BROWN_RED, actual);

		actual = options.getEnumValue("enum3", Colors.GREEN, Colors.class);
		assertEquals(Colors.GREEN, actual);

		actual = options.getEnumValue("enumX", Colors.GREEN, Colors.class);
		assertEquals(Colors.GREEN, actual);
	}

	/**
	 * Test corresponding method of class <code>Options</code>.
	 */
	public void testSetOption() {
		boolean overriden = options.setOption("option5", "value5");
		String actual = options.getValue("option5");
		assertEquals("value5", actual);
		assertFalse(overriden);
		overriden = options.setOption("option5", "value5");
		assertTrue(overriden);
	}
}