/*--------------------------------------------------------------------------+
$Id: TwoDimHashMapTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import junit.framework.TestCase;

/**
 * JUnit test for <code>Options</code> class.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * 
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: 82FFACFEC5C4C6E2F9113C29B31F284D
 */
public class TwoDimHashMapTest extends TestCase {

	/** Map under test. */
	private TwoDimHashMap<String, String, String> map;

	/** Create fixture. */
	@Override
	protected void setUp() {
		map = new TwoDimHashMap<String, String, String>();
		init(map);
	}

	/** Initialize map. */
	private void init(TwoDimHashMap<String, String, String> map) {
		// Project A
		// +- Dan -> Testing
		// +- Flo -> Documentation
		// Project B
		// +- Flo -> Design
		// +- Dan -> QA
		// +- Markus -> CM

		map.putValue("A", "Dan", "Testing");
		map.putValue("A", "Flo", "Documentation");
		map.putValue("B", "Flo", "Design");
		map.putValue("B", "Dan", "QA");
		map.putValue("B", "Markus", "CM");
	}

	/** Test {@link TwoDimHashMap#putAll(TwoDimHashMap)}. */
	public void testPutAll() {
		TwoDimHashMap<String, String, String> newMap = new TwoDimHashMap<String, String, String>();
		init(newMap);

		map.clear();
		assertTrue(map.isEmpty());

		map.putAll(newMap);
		testGetValues();
		testGetFirstKeys();
		testGetSecondKeys();
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.putValue(K1, K2, I)'
	 */
	public void testPutValue() {
		// non-existent first
		map.putValue("C", "Flo", "Test");
		String result = map.getValue("C", "Flo");
		assertEquals("Test", result);

		// non-existent second
		map.putValue("A", "Peter", "Test");
		result = map.getValue("A", "Peter");
		assertEquals("Test", result);

		// overwrite existent value
		map.putValue("A", "Dan", "QA");
		result = map.getValue("A", "Dan");
		assertEquals("QA", result);

	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getValue(K1, K2)'
	 */
	public void testGetValue() {

		assertEquals("Testing", map.getValue("A", "Dan"));
		assertEquals("Documentation", map.getValue("A", "Flo"));
		assertEquals("Design", map.getValue("B", "Flo"));
		assertEquals("QA", map.getValue("B", "Dan"));
		assertEquals("CM", map.getValue("B", "Markus"));
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getValuesByFirstKey(Object)'
	 */
	public void testGetValuesByFirstKey() {
		Collection<String> values = map.getValuesByFirstKey("A");
		assertTrue(values.contains("Testing"));
		assertTrue(values.contains("Documentation"));
		assertEquals(2, values.size());

		values = map.getValuesByFirstKey("B");
		assertTrue(values.contains("Design"));
		assertTrue(values.contains("QA"));
		assertTrue(values.contains("CM"));
		assertEquals(3, values.size());
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getFirstKeys()'
	 */
	public void testGetFirstKeys() {
		Set<String> keys = map.getFirstKeys();
		assertTrue(keys.contains("A"));
		assertTrue(keys.contains("B"));
		assertEquals(2, keys.size());
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getSecondKeys(Object)'
	 */
	public void testGetSecondKeys() {
		Set<String> keys = map.getSecondKeys("A");
		assertTrue(keys.contains("Dan"));
		assertTrue(keys.contains("Flo"));
		assertEquals(2, keys.size());

		keys = map.getSecondKeys("B");
		assertTrue(keys.contains("Dan"));
		assertTrue(keys.contains("Flo"));
		assertTrue(keys.contains("Markus"));
		assertEquals(3, keys.size());
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getValuesBySecondKey(Object)'
	 */
	public void testGetValuesBySecondKey() {
		List<String> values = map.getValuesBySecondKey("Flo");
		assertTrue(values.contains("Design"));
		assertTrue(values.contains("Documentation"));
		assertEquals(2, values.size());

		values = map.getValuesBySecondKey("Dan");
		assertTrue(values.contains("Testing"));
		assertTrue(values.contains("QA"));
		assertEquals(2, values.size());

		values = map.getValuesBySecondKey("Markus");
		assertTrue(values.contains("CM"));
		assertEquals(1, values.size());
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getValues()'
	 */
	public void testGetValues() {
		List<String> values = map.getValues();
		assertEquals(5, values.size());
		assertTrue(values.contains("Testing"));
		assertTrue(values.contains("Documentation"));
		assertTrue(values.contains("Design"));
		assertTrue(values.contains("QA"));
		assertTrue(values.contains("CM"));
	}

	/**
	 * Test method for 'edu.tum.cs.commons.collections.TwoDimHashMap.getSize()'
	 */
	public void testGetSize() {
		assertEquals(5, map.getSize());
		map.putValue("testX", "testY", "testZ");
		assertEquals(6, map.getSize());
		map.remove("testX", "testY");
		assertEquals(5, map.getSize());
		map.remove("A", "Flo");
		assertEquals(4, map.getSize());
	}

	/**
	 * Test method for 'edu.tum.cs.commons.collections.TwoDimHashMap.isEmpty()'
	 */
	public void testIsEmpty() {
		assertFalse(map.isEmpty());
		map.clear();
		assertTrue(map.isEmpty());

		setUp();

		assertFalse(map.isEmpty());
		map.remove("A", "Flo");
		map.remove("A", "Dan");
		map.remove("B", "Flo");
		map.remove("B", "Dan");
		map.remove("B", "Markus");
		assertTrue(map.isEmpty());
	}

	/**
	 * Test method for
	 * 'edu.tum.cs.commons.collections.TwoDimHashMap.getFirstSize(K1)'
	 */
	public void testGetFirstSize() {
		assertEquals(2, map.getSecondSize("A"));
		assertEquals(3, map.getSecondSize("B"));
		map.putValue("A", "TestX", "TestY");
		assertEquals(3, map.getSecondSize("A"));
	}

	/**
	 * Test method for 'edu.tum.cs.commons.collections.TwoDimHashMap.clear()'
	 */
	public void testClear() {
		testIsEmpty();
	}

	/**
	 * Test method for 'edu.tum.cs.commons.collections.TwoDimHashMap.remove(K1,
	 * K2)'
	 */
	public void testRemoveSecondLevelKey() {
		map.remove("A", "Flo");
		map.remove("A", "Dan");

		assertEquals(3, map.getSize());

		assertNull(map.getValue("A", "Flo"));
		assertNull(map.getValue("A", "Dan"));
		assertNull(map.getValuesByFirstKey("A"));

		map.remove("B", "Flo");
		map.remove("B", "Dan");
		map.remove("B", "Markus");

		assertTrue(map.isEmpty());
	}

	/**
	 * Test method for 'edu.tum.cs.commons.collections.TwoDimHashMap.remove(K1)'
	 */
	public void testRemoveFirstLevelKey() {
		map.remove("A");

		assertEquals(3, map.getSize());

		assertNull(map.getValue("A", "Flo"));
		assertNull(map.getValue("A", "Dan"));
		assertNull(map.getValuesByFirstKey("A"));

		map.remove("B");

		assertTrue(map.isEmpty());
	}

}