/*--------------------------------------------------------------------------+
$Id: MathUtils.java 29788 2010-08-19 08:46:02Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.math;

import java.util.Collection;

/**
 * Collection of math utility methods.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 29788 $
 * @levd.rating GREEN Hash: AC3B3CC253965F47E5F474B132F7F59B
 */
public class MathUtils {

	/**
	 * Sum values.
	 * 
	 * @see SumAggregator
	 * @see EAggregationStrategy#SUM
	 */
	public static double sum(Collection<? extends Number> collection) {
		return aggregate(collection, EAggregationStrategy.SUM);
	}

	/**
	 * Find maximum.
	 * 
	 * @see MaxAggregator
	 * @see EAggregationStrategy#MAX
	 */
	public static double max(Collection<? extends Number> collection) {
		return aggregate(collection, EAggregationStrategy.MAX);
	}

	/**
	 * Find minimum.
	 * 
	 * @see MinAggregator
	 * @see EAggregationStrategy#MIN
	 */
	public static double min(Collection<? extends Number> collection) {
		return aggregate(collection, EAggregationStrategy.MIN);
	}

	/**
	 * Find mean.
	 * 
	 * @return {@link Double#NaN} for empty input collection
	 * 
	 * @see MeanAggregator
	 * @see EAggregationStrategy#MEAN
	 */
	public static double mean(Collection<? extends Number> collection) {
		return aggregate(collection, EAggregationStrategy.MEAN);
	}

	/**
	 * Find median.
	 * 
	 * @return {@link Double#NaN} for empty input collection
	 * 
	 * @see MedianAggregator
	 * @see EAggregationStrategy#MEDIAN
	 */
	public static double median(Collection<? extends Number> collection) {
		return aggregate(collection, EAggregationStrategy.MEDIAN);
	}

	/**
	 * Aggregate collections of values with a given aggregation strategy.
	 * 
	 * @return certain aggregation strategies may return {@link Double#NaN} for
	 *         empty input collections
	 */
	public static double aggregate(Collection<? extends Number> values,
			EAggregationStrategy aggregation) {
		return aggregation.getAggregator().aggregate(values);
	}

	/**
	 * Computes the factorial of n. Errors are not handled. If n is negative, 1
	 * will be returned. If n to too large, wrong results will be produced due
	 * to numerical overflow.
	 */
	public static long factorial(int n) {
		long result = 1;
		for (int i = 2; i <= n; ++i) {
			result *= i;
		}
		return result;
	}

	/** Checks if the provided number is neither infinite nor NaN. */
	public static boolean isNormal(double number) {
		return !Double.isInfinite(number) && !Double.isNaN(number);
	}
}