#include "claw/graph.hpp"
#include "claw/graph_algorithm.hpp"
#include <iostream>
#include <fstream>

/*---------------------------------------------------------------------------*/
/**
 * \brief Un sommet de notre graph.
 * \author Julien Jorge
 */
class sommet
{
public:
  int val;
  bool operator<(const sommet& s) const { return val < s.val; }
  bool operator==(const sommet& s) const { return val == s.val; }

  friend std::ostream& operator<<(std::ostream& os, const sommet& s)
  {
    return os << s.val;
  }
}; // class sommet

/*---------------------------------------------------------------------------*/
/**
 * \brief Un arc de notre graph.
 * \author Julien Jorge
 */
class arc
{
public:
  char symbole;
  int  poids;

  bool operator<(const arc& a) const { return poids < a.poids; }
  bool operator==(const arc& a) const
  {
    return (symbole == a.symbole) && (poids == a.poids);
  }

  friend std::ostream& operator<<(std::ostream& os, const arc& a)
  {
    return os << "[" << a.symbole << ", " << a.poids << "]";
  }
}; // class arc

/*---------------------------------------------------------------------------*/
class pl_scan_events : public claw::scan_events< claw::graph<sommet, arc> >
{
public:
  pl_scan_events( const sommet& s )
  {
    distances[s] = 0;
  }

  void visit_edge( const sommet& source, const sommet& dest )
  {
    parents[dest] = source;
    distances[dest] = distances[source] + 1;
  }

  std::map<sommet, unsigned int> distances;
  std::map<sommet, sommet> parents;
}; // class pl_scan_events

/*---------------------------------------------------------------------------*/
class pp_scan_events : public claw::scan_events< claw::graph<sommet, arc> >
{
public:
  pp_scan_events()
  {
    m_date = 0;
  }

  void start(const sommet& s)
  {
    ++m_date;
    dates[s].first = m_date;
  }

  void visit_edge( const sommet& source, const sommet& dest )
  {
    parents[dest] = source;
  }

  void end(const sommet& s)
  {
    ++m_date;
    dates[s].second = m_date;
  }

  std::map<sommet, std::pair<unsigned int, unsigned int> > dates;
  std::map<sommet, sommet> parents;
  
private:
  unsigned int m_date;
}; // class pp_scan_events

/*---------------------------------------------------------------------------*/
/**
 * \brief Affichage d'un graphe
 */
template<class S, class A, class Comp>
std::ostream& operator<<( std::ostream& os, const claw::graph<S, A, Comp>& g)
{
  typename claw::graph<S, A, Comp>::vertex_iterator it_s;
  typename claw::graph<S, A, Comp>::edge_iterator it_e;

  // sommets
  os << "S = {";
  for (it_s = g.vertex_begin(); it_s != g.vertex_end(); ++it_s)
    {
      os << " " << *it_s;
    }
  os << " }" << std::endl;

  // arcs
  os << "A = {" << std::endl;
  for (it_e = g.edge_begin(); it_e != g.edge_end(); ++it_e)
    os << " [" << it_e->label() << ", " << it_e->source() << ", " 
       << it_e->target() << "]" << std::endl;
  os << " }" << std::endl;
          
  return os;
} // operator<<





/*---------------------------------------------------------------------------*/
void test_pl(const claw::graph<sommet, arc>& g)
{
  sommet s;
  std::cout << "Distances, calcul depuis le sommet : " << std::endl;
  std::cin >> s.val;
  pl_scan_events events(s);
  claw::breadth_scan<claw::graph<sommet, arc>, pl_scan_events> scan( g,
                                                                     s,
                                                                     events );

  scan();

  std::map<sommet, unsigned int>::const_iterator it_d;
  for (it_d = events.distances.begin(); it_d!=events.distances.end(); ++it_d)
    std::cout << it_d->first.val << "  " << it_d->second << std::endl;

  std::map<sommet, sommet>::const_iterator it_s;
  for (it_s = events.parents.begin(); it_s!=events.parents.end(); ++it_s)
    std::cout << it_s->second.val << " parent de " 
              << it_s->first.val << std::endl;
}


/*---------------------------------------------------------------------------*/
void test_pp(const claw::graph<sommet, arc>& g)
{
  pp_scan_events events;
  claw::depth_scan<claw::graph<sommet, arc>, pp_scan_events> scan( g, events );

  scan();

  std::map<sommet, std::pair<unsigned int,unsigned int> >::const_iterator it_d;
  for (it_d = events.dates.begin(); it_d!=events.dates.end(); ++it_d)
    std::cout << it_d->first.val << " : " << it_d->second.first << ","
              << it_d->second.second << std::endl;

  std::map<sommet, sommet>::const_iterator it_s;
  for (it_s = events.parents.begin(); it_s!=events.parents.end(); ++it_s)
    std::cout << it_s->second.val << " parent de " 
              << it_s->first.val << std::endl;
}


/*---------------------------------------------------------------------------*/
void tri_topologique(const claw::graph<sommet, arc>& g)
{
  claw::topological_sort< claw::graph<sommet, arc> > sort;
  claw::topological_sort< claw::graph<sommet, arc> >::const_iterator it;

  sort(g);

  for (it = sort.begin(); it!=sort.end(); ++it)
    std::cout << *it << std::endl;
} // tri_topologique()

/*---------------------------------------------------------------------------*/
bool test_graphs( const std::string& filename )
{
  claw::graph<sommet, arc> g;
  sommet s1, s2;
  arc a;
  std::ifstream f(filename.c_str());
  
  if (f)
    {
      while (f >> s1.val)
        {
          f >> s2.val;

          if (s2.val > 0)
            {
              f >> a.symbole >> a.poids;
              g.add_edge(s1, s2, a);
            }
          else
            {
              std::cout << s1 << " : isol" << std::endl;
              g.add_vertex(s1);
            }
        }

      f.close();

      std::cout << "        -------------------------------------- affichage" 
                << std::endl;
      std::cout << g << std::endl;

      std::cout << "        ------------------- parcours en largeur" 
                << std::endl;

      test_pl( g );

      std::cout << "        ------------------- parcours en profondeur" 
                << std::endl;

      test_pp( g );

      std::cout << "        ------------------- tri topologique" 
                << std::endl;

      tri_topologique( g );

      return true;
    }
  else
    return false;
} // test_graphs()

/*---------------------------------------------------------------------------*/
int main(int argc, char* argv[])
{
  if (argc != 2)
    {
      std::cout << argv[0] << " test_file" << std::endl;
      return 1;
    }
  else
    {
      try
        {
          if ( !test_graphs(argv[1]) )
            std::cout << "Problme. Le fichier " << argv[1] << " existe-t-il ?"
                      << std::endl;
        }
      catch( claw::graph_exception e )
        {
          std::cerr << e.what() << std::endl;
        }
          
      return 0;
    }
}

