/*
   Copyright (C) 2003 Commonwealth Scientific and Industrial Research
   Organisation (CSIRO) Australia

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

   - Neither the name of CSIRO Australia nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
   PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE ORGANISATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


#include <config.h>

#include <time.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>

/*#define DEBUG*/
/*#define DEBUG_VERBOSE*/
/*#define DEBUG_FILE "/tmp/anx_import_ogg.out"*/

#define MIN(a,b) ((a)<(b)?(a):(b))

/* Tolerance for floating point rounding errors: used for portable
 * floating point comparisons in filter().
 */
#define TOLERANCE 0.000000000001

#include <oggz/oggz.h>

#include <annodex/anx_import.h>
#include <annodex/annodex.h>

#define INT8_AT(x) (*(unsigned char *)(x))
#define INT32_LE_AT(x) _le_32((*(ogg_int32_t *)(x)))
#define INT32_BE_AT(x) _be_32((*(ogg_int32_t *)(x)))
#define INT64_LE_AT(x) _le_64((*(ogg_int64_t *)(x)))

/* RFC3534 */
#define OGG_CONTENT_TYPE "application/ogg"

#define ANX_CONTENT_TYPE "application/x-annodex"

#define CMML_CONTENT_TYPE "text/x-cmml"
#define VORBIS_CONTENT_TYPE "audio/x-vorbis"
#define SPEEX_CONTENT_TYPE "audio/x-speex"
#define THEORA_CONTENT_TYPE "video/x-theora"
#define XVID_CONTENT_TYPE "video/x-xvid"

/** well-defined sizeof(AnxData) binary portion for Annodex v2 */
#define ANX_DATA_SIZE 28

#undef  USE_THEORA_PRE_ALPHA_3_FORMAT

#define SUBSECONDS 1000.0

typedef enum {
  STATE_HEADERS,
  STATE_GRANULEINFO,
  STATE_FILTER,
  STATE_DATA
} state_t;

typedef enum {
  NEED_SEEK_PENDING = 0,
  NEED_SEEK = 1,
  NEED_SEEK_DONE = 2
} need_seek_t;

typedef struct _AnxOggPacket AnxOggPacket;
typedef struct _AnxOggTrack AnxOggTrack;
typedef struct _AnxOggData AnxOggData;

struct _AnxOggPacket {
  long length;
  unsigned char * data;
  long granulepos;
  AnxSourceTrack * source_track;
  double current_time;
  int eos;
};

struct _AnxOggTrack {
  AnxSourceTrack source_track;
  int anxv2_ignore_packet;
  int need_keygranule;
  ogg_int64_t keygranule;
  double keygranule_time;
  int filter_got_keygranule;
};

struct _AnxOggData {
  OGGZ * oggz;

#ifdef DEBUG
  FILE * df; /* debug filehandle */
#endif

  AnxSource * anx_source;

  state_t state;

  char * id;
  int ignore_media;
  int got_non_bos;

  long skeleton_serialno;
  int got_skeleton_eos;

  long nr_headers_remaining;
  long headers_unread;

  int use_granule_seek;
  double min_granule_seek;

  need_seek_t need_seek;
  int got_end;

  /* Maintain a list of AnxOggTracks */
  OggzTable * logicals;

  AnxList * media_packets;
  long current_offset; /* offset into current packet */

  AnxImportCMML import_cmml;
  void * import_user_data;

  long cmml_serialno; /* Serialno of CMML track, or -1 if not present */
  int cmml_granuleshift;
  int cmml_need_keygranule;
};

static AnxImporter anxogg_importer;

static ogg_uint32_t
_le_32 (ogg_uint32_t i)
{
   ogg_uint32_t ret=i;
#ifdef WORDS_BIGENDIAN
   ret =  (i>>24);
   ret += (i>>8) & 0x0000ff00;
   ret += (i<<8) & 0x00ff0000;
   ret += (i<<24);
#endif
   return ret;
}

static ogg_uint32_t
_be_32 (ogg_uint32_t i)
{
   ogg_uint32_t ret=i;
#ifndef WORDS_BIGENDIAN
   ret =  (i>>24);
   ret += (i>>8) & 0x0000ff00;
   ret += (i<<8) & 0x00ff0000;
   ret += (i<<24);
#endif
   return ret;
}

static  ogg_int64_t
_le_64 (ogg_int64_t l)
{
  ogg_int64_t ret=l;
  unsigned char *ucptr = (unsigned char *)&ret;
#ifdef WORDS_BIGENDIAN
  unsigned char temp;

  temp = ucptr [0] ;
  ucptr [0] = ucptr [7] ;
  ucptr [7] = temp ;

  temp = ucptr [1] ;
  ucptr [1] = ucptr [6] ;
  ucptr [6] = temp ;

  temp = ucptr [2] ;
  ucptr [2] = ucptr [5] ;
  ucptr [5] = temp ;

  temp = ucptr [3] ;
  ucptr [3] = ucptr [4] ;
  ucptr [4] = temp ;

#endif
  return (*(ogg_int64_t *)ucptr);
}

static ogg_int64_t
gp_to_granule (OGGZ * oggz, long serialno, ogg_int64_t granulepos)
{
  int granuleshift;
  ogg_int64_t iframe, pframe;

  granuleshift = oggz_get_granuleshift (oggz, serialno);

  iframe = granulepos >> granuleshift;
  pframe = granulepos - (iframe << granuleshift);

  return (iframe + pframe);

}

static double
gp_to_time (OGGZ * oggz, long serialno, ogg_int64_t granulepos)
{
  ogg_int64_t gr_n, gr_d;
  ogg_int64_t granule;

  if (oggz_get_granulerate (oggz, serialno, &gr_n, &gr_d) != 0) return -1.0;

  granule = gp_to_granule (oggz, serialno, granulepos);

  return (double)(granule * gr_d) / ((double)gr_n * SUBSECONDS);
}

static char *
anxogg_strdup (const char * s)
{
  char * ret;
  if (!s) return NULL;
  ret = anx_malloc (strlen (s) + 1);
  return strcpy (ret, s);
}

static double
anxogg_seek_update (AnxSource * source)
{
  AnxOggData * aod = (AnxOggData *)source->custom_data;
  double seek_offset;
  ogg_int64_t units, units_at;
  double offset;

  if (aod->use_granule_seek) {
    seek_offset = aod->min_granule_seek;
  } else {
    seek_offset = source->start_time;
  }
  seek_offset -= 1.0;
  if (seek_offset < 0.0) seek_offset = 0.0;

  units = (ogg_int64_t)(SUBSECONDS * seek_offset);
  units_at = oggz_seek_units (aod->oggz, units, SEEK_SET);

  if (units_at == -1) {
#ifdef DEBUG
    fprintf (aod->df, "anxogg_seek_update: oggz_seek_units FAIL\n");
#endif
    return -1.0;
  }

  offset = ((double)units_at) / SUBSECONDS;

  aod->need_seek = NEED_SEEK_DONE;

#ifdef DEBUG
  fprintf (aod->df,
	   "anxogg: seek_update to %lld (wanted %lld (%f))\n", units_at,
	  units, seek_offset);
#endif

  if (source->end_time != -1.0 && offset >= source->end_time) {
#ifdef DEBUG
    fprintf (aod->df,
	     "anxoggg: seek update > end_time %f\n", source->end_time);
#endif
    aod->got_end = 1;
  }

  return offset;
}

static int
read_packet_headers (OGGZ * oggz, ogg_packet * op, long serialno,
		     void * user_data)
{
  unsigned char * header = op->packet;
  AnxOggData * aod = (AnxOggData *)user_data;
  AnxOggTrack * aot = NULL;
  AnxSource * m = aod->anx_source;
  AnxSourceTrack * track = NULL;
  int need_insert = 1;
  int got_headers = 0;

  aot = (AnxOggTrack *) oggz_table_lookup (aod->logicals, serialno);
  if (aot != NULL) {
    need_insert = 0;
  }

  if (op->b_o_s) {
    if (!strncmp ((char *)header, "CMML", 5)) {
      /* Don't insert this into the media */
      need_insert = 0;
	
      aod->cmml_serialno = serialno;

      /* Check the CMML header is of correct length */
      if (op->bytes > 28)
	aod->cmml_granuleshift = (int)header[28];
      else
	aod->cmml_granuleshift = 0;

      if (aod->cmml_granuleshift == 0)
	aod->cmml_need_keygranule = 0;
      else
	aod->cmml_need_keygranule = 1;
    } else {
      if (aot == NULL) {
	aot = (AnxOggTrack *) anx_malloc (sizeof (AnxOggTrack));
	memset (aot, 0, sizeof(AnxOggTrack));
	need_insert = 1;
      }
      
      track = &(aot->source_track);
      track->eos = 0;

      if (!strncmp ((char *)&header[1], "vorbis", 6)) {
	track->content_type = anxogg_strdup (VORBIS_CONTENT_TYPE);
	track->granule_rate_n = (anx_int64_t) INT32_LE_AT(&header[12]);
	track->granule_rate_d = 1;
	track->nr_header_packets = 3;
	track->basegranule = 0;
	track->preroll = 2;
	track->granuleshift = 0;
      } else if (!strncmp ((char *)&op->packet[0], "Speex   ", 8)) {
	track->content_type = anxogg_strdup (SPEEX_CONTENT_TYPE);
	track->granule_rate_n = (ogg_int64_t) INT32_LE_AT(&header[36]);
	track->granule_rate_d = 1;
	track->nr_header_packets = 2 + (ogg_int64_t) INT32_LE_AT(&header[68]);
	track->basegranule = 0;
	track->preroll = 3;
	track->granuleshift = 0;
      } else if (!strncmp ((char *)&op->packet[1], "theora", 6)) {
	char keyframe_granule_shift = 0;
	track->content_type = anxogg_strdup (THEORA_CONTENT_TYPE);
#if USE_THEORA_PRE_ALPHA_3_FORMAT
	/* old header format, used by Theora alpha 2 and earlier */
	keyframe_granule_shift =  (header[36] & 0xf8) >> 3;
#else
	keyframe_granule_shift =  (header[40] & 0x03) /* 00000011 */ << 3;
	keyframe_granule_shift |= (header[41] & 0xe0) /* 11100000 */ >> 5;
#endif

#if USE_ANNODEX2_GRANULERATE
	track->granule_rate_n = INT32_BE_AT(&header[22]) *
	  (1 << keyframe_granule_shift) / 1000000;
	track->granule_rate_d = INT32_BE_AT(&header[26]) / 1000000;
#else /* anx v3 */
	track->granule_rate_n = INT32_BE_AT(&header[22]);
	track->granule_rate_d = INT32_BE_AT(&header[26]);
#endif
	track->nr_header_packets = 3;
	track->basegranule = 0;
	track->preroll = 0;
	track->granuleshift = keyframe_granule_shift;
      } else if (!strncmp ((char *)&op->packet[0], "fishead", 8)) {
	need_insert = 0;
	aod->skeleton_serialno = serialno;
      } else if (!strncmp ((char *)&op->packet[0], "AnxData", 7)) {
	AnxParams * params;
	long n;
	char * buf;

	/* Mark this track as anxv2, and don't copy the AnxData packet
	 * into the bitstream */
	aot->anxv2_ignore_packet = 1;

	/* Set default values, these will be overwritten if the track
	 * actually contains a content type we can parse */

	/* Ensure params is NUL terminated */
	n = op->bytes - ANX_DATA_SIZE;
	buf = anx_malloc (n+1);
	memcpy (buf, (char *)header + ANX_DATA_SIZE, n);
	buf[n] = '\0';

	params = anx_params_new_parse (buf, ANX_PARAMS_HEADERS);
	track->content_type =
	  anxogg_strdup (anx_params_get (params, "Content-Type"));
	track->id =
	  anxogg_strdup (anx_params_get (params, "ID"));
	anx_params_free (params);

	anx_free (buf);

	track->granule_rate_n = INT64_LE_AT(&header[8]);
	track->granule_rate_d = INT64_LE_AT(&header[16]);
	track->nr_header_packets = INT32_LE_AT(&header[24]);
	track->preroll = 0;
	track->granuleshift = 0;

	if (!strcmp (track->content_type, CMML_CONTENT_TYPE)) {
	  need_insert = 0;
	  aod->cmml_serialno = serialno;
	  aod->cmml_granuleshift = 0;
	  aod->cmml_need_keygranule = 0;
	}

      } else if (!strncmp ((char *)&op->packet[1], "video", 5) && 
		 !strncmp ((char *)&op->packet[9], "XVID", 4)) {
	track->content_type = anxogg_strdup (XVID_CONTENT_TYPE);
	track->granule_rate_n = (ogg_int64_t) 25;
	track->granule_rate_d = 1;
	track->nr_header_packets = 2;
	track->basegranule = 0;
	track->preroll = 0;
	track->granuleshift = 0;
      } else {
	anx_free (aot);
	return OGGZ_STOP_ERR;
      }

      aod->nr_headers_remaining += track->nr_header_packets;
      aod->headers_unread += track->nr_header_packets;

      if (track->granuleshift > 0) {
	aod->use_granule_seek = 1;
	aot->need_keygranule = 1;
      }

      if (need_insert) {
	oggz_table_insert (aod->logicals, serialno, aot);

#ifdef DEBUG
	fprintf (aod->df,
		 "anxogg::read_packet_headers: Added track for (%010ld): %ld/%ld, +%d headers\n",
		 serialno,
		 (long)track->granule_rate_n, (long)track->granule_rate_d,
		 (int)track->nr_header_packets);
#endif
      }

      /* XXX: fix this for theora */
      if (track->granule_rate_n != 0 && track->granule_rate_d != 0) {
	if (m->start_time != 0.0) {
	  track->start_granule =
	    m->start_time * track->granule_rate_n /
	    track->granule_rate_d;
	} else {
	  track->start_granule = 0;
	}

	if (m->end_time != -1.0) {
	  track->end_granule =
	    m->end_time * track->granule_rate_n / track->granule_rate_d;
#ifdef DEBUG
	  fprintf (aod->df, "anxogg::read_packet_headers: end_granule %lld\n",
		  track->end_granule);
#endif
	}
      }

      if (need_insert) {
#ifdef DEBUG
	fprintf (aod->df,
		 "anxogg::read_packet_headers: adding track for %s\n", track->content_type);
#endif
	m->tracks = anx_list_append (m->tracks, track);
      } else {
        anx_free (aot);
        aot = NULL;
      }
    }
  } else {
    if (op->e_o_s && serialno == aod->skeleton_serialno) {
      aod->got_skeleton_eos = 1;
    }

    aod->got_non_bos = 1;
  }

  if (aot && aot->anxv2_ignore_packet) {
    aot->anxv2_ignore_packet = 0;
    op->b_o_s = 1;
  }

  if (aod->nr_headers_remaining > 0) {
    aod->nr_headers_remaining--;
  }

  if (aod->skeleton_serialno == -1L) {
    if (aod->got_non_bos && aod->nr_headers_remaining == 0) got_headers = 1;
  } else {
    if (aod->got_skeleton_eos) got_headers = 1;
  }
  
  if (got_headers) {
    if (m->start_time != 0.0) {
      if (oggz_seek_units (oggz, 0, SEEK_CUR) >= 0) {
	oggz_set_data_start (oggz, oggz_tell (oggz));
	if (aod->use_granule_seek) aod->state = STATE_GRANULEINFO;
      }
    } else {
      aod->state = STATE_DATA;
    }
    return OGGZ_STOP_OK;
  }

  return OGGZ_CONTINUE;
}

static int
filter (AnxOggData * aod, AnxOggTrack * aot, long serialno, ogg_packet * op)
{
  AnxSourceTrack * track = &(aot->source_track);
  double timestamp;

  /* TRUE if we're not in state FILTER */
  if (aod->state != STATE_FILTER) return 1;

  /* TRUE if we haven't done the seek yet */
  if (aod->need_seek == NEED_SEEK_PENDING) return 1;

  timestamp = gp_to_time (aod->oggz, serialno, op->granulepos);

  /* TRUE if we're beyond the start time */
  if (timestamp-TOLERANCE >= aod->anx_source->start_time) {
    aod->state = STATE_DATA;
    return 1;
  }

  /* from here, we know we're filtering */

  /* FALSE if we're not using granule_seek at all */
  if (!aod->use_granule_seek) return 0;

  /* FALSE if this track doesn't use granuleshift */
  if (track->granuleshift == 0) return 0;

  /* TRUE if we're beyond the keygranule for this track */
  if (aot->filter_got_keygranule) return 1;

  /* TRUE if we're after the keygranule time */
  if (op->granulepos != -1 && timestamp+TOLERANCE >= aot->keygranule_time) {
    aot->filter_got_keygranule = 1;
    return 1;
  }

  /* XXX: TRUE if a theora from a keyframe, even if no granulepos */
  if (!strcmp (track->content_type, THEORA_CONTENT_TYPE) &&
      op->bytes > 0 && !(op->packet[0] & 0x40)) {
#ifdef DEBUG
    fprintf (aod->df,
	     "anxogg::FILTER got Theora keyframe\n");
#endif

    aot->filter_got_keygranule = 1;
    return 1;
  }

  /* FALSE otherwise */
  return 0;

#if 0
  /* FALSE if this track uses keygranules, and we're before its time */
  if (granulepos != -1 && timestamp+TOLERANCE < aot->keygranule_time) return 0;

  /* TRUE otherwise */
  return 1;
#endif
}

static int
read_packet_data (OGGZ * oggz, ogg_packet * op, long serialno,
		  void * user_data)
{
  AnxOggData * aod = (AnxOggData *)user_data;
  AnxOggTrack * aot = NULL;
  AnxOggPacket * aop;
  AnxSource * m = aod->anx_source;
  AnxSourceTrack * track = NULL;

#ifdef DEBUG
  fprintf (aod->df, "anxogg::read_packet_data IN: %010ld, granulepos %llx\n",
	  serialno, op->granulepos);
#endif

  if (!(aod->cmml_serialno != -1 && serialno == aod->cmml_serialno)) {
    aot = (AnxOggTrack *) oggz_table_lookup (aod->logicals, serialno);
    if (aot == NULL) {
#ifdef DEBUG
      fprintf (aod->df,
	       "anxogg::read_packet_data: %010ld not found\n", serialno);
#endif
      return OGGZ_STOP_OK;
    }
  }

  if (op->b_o_s) {
    if (!strncmp ((char *)&op->packet[0], "AnxData", 7)) {
      /* Mark this track as anxv2, and don't copy the AnxData packet
       * into the bitstream */
      if (aot) aot->anxv2_ignore_packet = 1;
    }
  } else {
    if (aod->cmml_serialno != -1 && serialno == aod->cmml_serialno) {
      double at_time;

#ifdef DEBUG
      fprintf (aod->df, "aod->cmml_serialno %010ld, op->granulepos %lld\n",
	      aod->cmml_serialno, op->granulepos);
#endif

      if (op->granulepos == -1) {
	return OGGZ_STOP_ERR;
      }
      at_time = gp_to_time (aod->oggz, serialno, op->granulepos);
      
#ifdef DEBUG
      {
	char * header = (char *)op->packet;
	fprintf (aod->df,
		 "anxogg::read_packet_data: got CMML <%c%c%c%c> at %f\n",
		header[1], header[2], header[3], header[4], at_time);
      }
#endif

      /* If this content doesn't use cmml_granuleshift, and this clip is
       * before the start_time, drop it */
      if (aod->cmml_granuleshift == 0 && op->bytes > 6 &&
	  !strncmp ((char *)op->packet, "<clip", 5) &&
	  at_time < aod->anx_source->start_time-TOLERANCE) {
#ifdef DEBUG
	fprintf (aod->df,
		 "anxogg::read_packet_data: CMML clip too early, dropping\n");
#endif
	return OGGZ_STOP_OK;
      }

      if (aod->import_cmml)
	aod->import_cmml ((char *)op->packet, op->bytes, at_time,
			  aod->import_user_data);
      return OGGZ_CONTINUE;
    } else {
      if (aot->anxv2_ignore_packet) {
	aot->anxv2_ignore_packet = 0;
	op->b_o_s = 1;
      } else {
	aod->got_non_bos = 1;
      }
    }
  }

  track = &(aot->source_track);

  if (!track) {
#ifdef DEBUG
    fprintf (aod->df, "anxogg::read_packet: track NULL\n");
#endif
    return OGGZ_STOP_OK;
  }

  if (aod->nr_headers_remaining == 0 && !aod->got_end && 
      (m->end_time != -1.0) && (op->granulepos != -1)) {
    ogg_int64_t granule = gp_to_granule (oggz, serialno, op->granulepos);
    if (granule >= track->end_granule) {
#ifdef DEBUG
      fprintf (aod->df, "anxogg::read_packet_data: got_end\n");
#endif
      aod->got_end = 1;
    }
  }

#ifdef DEBUG
  fprintf (aod->df, "anxogg::read_packet: Filter?\n");
#endif

  if (!aod->ignore_media && !aod->got_end && !aot->anxv2_ignore_packet &&
      (aod->state != STATE_FILTER ||
       filter (aod, aot, serialno, op))) {
#ifdef DEBUG
    fprintf (aod->df, "anxogg::read_packet_data: no, copy out\n");
#endif
    aop = anx_malloc (sizeof (AnxOggPacket));
    aop->length = op->bytes;
    aop->data = anx_malloc (op->bytes);
    aop->granulepos = op->granulepos;
    aop->current_time = ((double)oggz_tell_units (oggz)) / SUBSECONDS;
    aop->source_track = track;
    aop->eos = op->e_o_s;

#if 0
    if (aod->nr_headers_remaining == 0 && aop->granulepos != -1) {
      aop->granulepos -= track->start_granule;
    }
#endif

    memcpy (aop->data, op->packet, op->bytes);
    
    aod->media_packets = anx_list_append (aod->media_packets, aop);

    if (aod->nr_headers_remaining > 0) {
      aod->nr_headers_remaining--;

#ifdef DEBUG
      fprintf (aod->df,
	       "anxogg::read_packet_data: nr_headers_remaining = %d, start_time = %f\n",
	      aod->nr_headers_remaining, m->start_time);
#endif
    }

    if (aod->need_seek == NEED_SEEK_PENDING &&
	aod->nr_headers_remaining == 0 &&
	m->start_time != 0.0) {
      if (oggz_seek_units (oggz, 0, SEEK_CUR) >= 0) {
#ifdef DEBUG
	fprintf (aod->df, "anxogg::read_packet_data NEED seek\n");
#endif
	aod->need_seek = NEED_SEEK;
	return OGGZ_CONTINUE;
      }
    }
  }

  return OGGZ_STOP_OK;
}

static int
granuleinfo_update_state (AnxOggData * aod)
{
  AnxOggTrack * aot;
  int i, n;

#ifdef DEBUG
  fprintf (aod->df, "anxogg::granuleinfo_update_state cmml %010ld %s\n",
	   aod->cmml_serialno,
	   aod->cmml_need_keygranule ? "needs keygranule" : "no keygranule");
#endif

  if (aod->cmml_serialno != -1 && aod->cmml_need_keygranule) return 0;

  n = oggz_table_size (aod->logicals);

#ifdef DEBUG
  fprintf (aod->df, "anxogg::granuleinfo_update_state %d logicals\n", n);
#endif

  for (i = 0; i < n; i++) {
    aot = (AnxOggTrack *)oggz_table_nth (aod->logicals, i, NULL);
    if (aot->need_keygranule) {
#ifdef DEBUG
      fprintf (aod->df, "anxogg::granuleinfo_update_state logical %d needs keygranule\n", i);
#endif
      return 0;
    }
  }

  aod->state = STATE_FILTER;

  return 0;
}

static int
read_page_granuleinfo (OGGZ * oggz, const ogg_page * og, long serialno,
		       void * user_data)
{
  AnxOggData * aod = (AnxOggData *)user_data;
  AnxOggTrack * aot = NULL;
  AnxSourceTrack * track = NULL;
  ogg_int64_t granulepos, iframe, pframe;
  ogg_int64_t cmml_keygranule;
  double offset, start_time;

#ifdef DEBUG
  fprintf (aod->df, "read_page_granuleinfo: track %010ld\n", serialno);
#endif

  granulepos = ogg_page_granulepos ((ogg_page *)og);

  if (aod->cmml_serialno != -1 && serialno == aod->cmml_serialno) {
    if (!aod->cmml_need_keygranule) return OGGZ_STOP_OK;

    aod->cmml_need_keygranule = 0;

    /* If this content does use cmml_granuleshift, and this clip is right at
     * the start time, there's no need to include the clip previous to this.
     */
    start_time = aod->anx_source->start_time;
    offset = gp_to_time (aod->oggz, serialno, granulepos);
    if (offset-TOLERANCE <= start_time) {
      return OGGZ_STOP_OK;
    }

    iframe = granulepos >> aod->cmml_granuleshift;
    cmml_keygranule = iframe << aod->cmml_granuleshift;

    offset = gp_to_time (aod->oggz, serialno, cmml_keygranule);
    if (aod->min_granule_seek == 0.0 || offset < aod->min_granule_seek)
      aod->min_granule_seek = offset;    
  } else {

    aot = (AnxOggTrack *) oggz_table_lookup (aod->logicals, serialno);
    if (aot == NULL) {
      /* If this track is not in the table, ignore it. */
      return OGGZ_STOP_OK;
    }

    track = &(aot->source_track);
    
    /* Slurp in granuleinfo */
    if (aot->need_keygranule && granulepos != -1) {
      iframe = granulepos >> track->granuleshift;
      pframe = granulepos - (iframe << track->granuleshift);

#if 1
      /* XXX: vaguely similar reasoning to CMML above */
      start_time = aod->anx_source->start_time;
      offset = gp_to_time (aod->oggz, serialno, granulepos);
      if (offset-TOLERANCE <= start_time) {
	return OGGZ_STOP_OK;
      }
      /* End XXX: */
#endif


      granulepos = (iframe + pframe);
      aot->keygranule = iframe << track->granuleshift;
      aot->need_keygranule = 0;
      
      offset = gp_to_time (aod->oggz, serialno, aot->keygranule);
      aot->keygranule_time = offset;
      if (aod->min_granule_seek == 0.0 || offset < aod->min_granule_seek)
	aod->min_granule_seek = offset;
      
#ifdef DEBUG
      fprintf (aod->df,
	       "read_page_granuleinfo: ^^^ has keygranule %lld (%lld|0) (%f seconds) (granuleshift %d)\n",
	      aot->keygranule, iframe, offset, track->granuleshift);
#endif
    }
  }

  /* Update state */
  granuleinfo_update_state (aod);

  return OGGZ_STOP_OK;
}

static int
anxogg_setup (AnxOggData * aod)
{
  long n;
  double start_time = 0.0, end_time = -1.0;
  off_t start_offset = 0, end_offset = -1;
  ogg_int64_t units, units_at;

  /* Slurp headers */
  oggz_set_read_callback (aod->oggz, -1, read_packet_headers, aod);
  while (aod->state == STATE_HEADERS &&
	 (n = oggz_read (aod->oggz, 1024)) != 0);

#ifdef DEBUG
  fprintf (aod->df,
	   "anxogg: slurped headers:\n"
	  "\toggz_read returned %ld\n"
	  "\taod->got_non_bos == %d\n"
	  "\taod->got_skeleton_eos == %d\n"
	  "\taod->nr_headers_remaining = %d\n",
	  n, aod->got_non_bos, aod->got_skeleton_eos,
	  aod->nr_headers_remaining);
#endif

  /* Find bitrate info */
  start_time = aod->anx_source->start_time;
  end_time = aod->anx_source->end_time;

  if (end_time == -1.0) {
    end_offset = oggz_seek (aod->oggz, 0, SEEK_END);

    /* Find the actual end time */
    units_at = oggz_seek_units (aod->oggz, 0, SEEK_END);
    if (units_at == -1) {
#ifdef DEBUG
      fprintf (aod->df, "anxogg_setup: oggz_seek_units end FAIL\n");
#endif
      return -1;
    }
    end_time = ((double)units_at) / 1000.0;
  } else {
    units = (ogg_int64_t)(SUBSECONDS * end_time);
    units_at = oggz_seek_units (aod->oggz, units, SEEK_SET);
    if (units_at == -1) {
#ifdef DEBUG
      fprintf (aod->df, "anxogg_setup: oggz_seek_units end FAIL\n");
#endif
      return -1;
    }
    end_offset = oggz_tell (aod->oggz);
  }

  units = (ogg_int64_t)(SUBSECONDS * start_time);
  units_at = oggz_seek_units (aod->oggz, units, SEEK_SET);
  if (units_at == -1) {
#ifdef DEBUG
    fprintf (aod->df, "anxogg_setup: oggz_seek_units start FAIL\n");
#endif
      return -1;
  }
  start_offset = oggz_tell (aod->oggz);

  aod->anx_source->byte_length = end_offset - start_offset;
  aod->anx_source->duration = end_time - start_time;

#ifdef DEBUG
  fprintf (aod->df,
	   "anxogg_setup: times (%f - %f) = %f\n", start_time, end_time,
	  end_time - start_time);
  fprintf (aod->df, "anxogg_setup: got byte range [%ld - %ld] = %ld bytes\n",
	  start_offset, end_offset, end_offset - start_offset);
#endif

  /* Find granule_seek info */
  if (aod->use_granule_seek && start_time > 0.0) {
    oggz_set_read_callback (aod->oggz, -1, NULL, NULL);
    oggz_set_read_page (aod->oggz, -1, read_page_granuleinfo, aod);
    while (aod->state == STATE_GRANULEINFO &&
	   (n = oggz_read (aod->oggz, 1024)) != 0);
    oggz_set_read_page (aod->oggz, -1, NULL, NULL);
  }

  /* Reset to beginning */
  oggz_seek (aod->oggz, 0, SEEK_SET);
  aod->nr_headers_remaining = aod->headers_unread;

  return 0;
}

static AnxOggData *
aod_new (void)
{
  AnxOggData * aod;

  aod = (AnxOggData *) anx_malloc (sizeof (AnxOggData));

#ifdef DEBUG
#ifdef DEBUG_FILE
  aod->df = fopen (DEBUG_FILE, "w");
#else
  aod->df = stdout;
#endif
#endif

  aod->oggz = NULL;
  aod->anx_source = NULL;
  aod->state = STATE_HEADERS;

  aod->id = NULL;;
  aod->ignore_media = 0;
  aod->got_non_bos = 0;

  aod->skeleton_serialno = -1L;
  aod->got_skeleton_eos = 0;

  aod->nr_headers_remaining = 0;
  aod->headers_unread = 0;

  aod->use_granule_seek = 0;
  aod->min_granule_seek = 0;

  aod->need_seek = NEED_SEEK_PENDING;
  aod->got_end = 0;
  aod->logicals = oggz_table_new ();

  aod->media_packets = NULL;
  aod->current_offset = 0;

  aod->import_cmml = NULL;
  aod->import_user_data = NULL;

  aod->cmml_serialno = -1;
  aod->cmml_granuleshift = 0;
  aod->cmml_need_keygranule = 0;

  return aod;
}

static AnxSource *
anxogg_open (const char * path, const char * id, int ignore_media,
	     double start_time, double end_time,
	     AnxImportCallbacks * import_callbacks)
{
  AnxSource * m;
  AnxOggData * aod;
  OGGZ * oggz;
  long n;

  if ((oggz = oggz_open ((char *)path, OGGZ_READ | OGGZ_AUTO)) == NULL)
    return NULL;

  m = (AnxSource *) anx_malloc (sizeof (AnxSource));
  if (m == NULL) return NULL;

  m->importer = &anxogg_importer;

  m->eos = 0;
  m->start_time = start_time;
  m->end_time = end_time;

  m->tracks = NULL;
  m->current_track = NULL;
  m->current_time = start_time;
  m->written_secondaries = 0;

  aod = aod_new ();

  aod->oggz = oggz;
  aod->anx_source = m;
  aod->id = (char *)id;
  aod->ignore_media = ignore_media;
  aod->import_cmml = import_callbacks->import_cmml;
  aod->import_user_data = import_callbacks->import_user_data;

#ifdef DEBUG
  fprintf (aod->df, "anxogg: open %s t=%f\n", path, start_time);
#endif

  anxogg_setup (aod);

  /* Set normal data reading callback */
  oggz_set_read_callback (oggz, -1, read_packet_data, aod);
  
  if (ignore_media) {
    /* Slurp CMML */
    while ((n = oggz_read (oggz, 1024)) != 0) {
    }
  }

  m->custom_data = aod;

  return m;
}

static long
anxogg_read_update (AnxSource * media)
{
  AnxOggData * aod = (AnxOggData *)media->custom_data;

 do_read:

  while ((aod->media_packets == NULL) && (oggz_read (aod->oggz, 1024)) != 0);

  if (aod->need_seek == NEED_SEEK && aod->nr_headers_remaining == 0) {
    anxogg_seek_update (media);
    goto do_read;
  }

  return 0;
}

static AnxOggPacket *
anxogg_packet_free (AnxOggPacket * aop)
{
  anx_free (aop->data);
  anx_free (aop);
  return NULL;
}

static long
anxogg_read (AnxSource * media, char * buf, long n, long bound)
{
  AnxOggData * aod = (AnxOggData *)media->custom_data;
  AnxOggPacket * aop;
  AnxList * head;
  long bytes_to_read;

  if (aod->ignore_media) return -1;

  anxogg_read_update (media);

  head = aod->media_packets;
  if (head == NULL) {
    media->eos = 1;
    return 0;
  }

  aop = (AnxOggPacket *)head->data;
  bytes_to_read = MIN (n, aop->length - aod->current_offset);

  memcpy (buf, &aop->data[aod->current_offset], bytes_to_read);

  aod->current_offset += bytes_to_read;

  if (aod->headers_unread > 0) aod->headers_unread--;
  if (aod->headers_unread == 0) media->written_secondaries = 1;

  media->current_track = aop->source_track;

#ifdef DEBUG_VERBOSE
  fprintf (aod->df, "anxogg: reading from stream %s\n",
	  media->current_track->content_type);
#endif

  media->current_track->current_granule = aop->granulepos;
  media->current_track->eos = aop->eos;

  /* If that's finished this media packet, advance to the next one */
  if (aod->current_offset >= aop->length) {
    aod->media_packets =
      anx_list_remove (aod->media_packets, head);

    aop = anxogg_packet_free (aop);
    anx_free (head);

    aod->current_offset = 0;

    anxogg_read_update (media);

    if (aod->media_packets != NULL)
      aop = (AnxOggPacket *)aod->media_packets->data;
  }

  /* Set current time to the current time of the next packet that would
   * be served */
  if (aop && aop->current_time != -1)
    media->current_time = aop->current_time;   

  return bytes_to_read;
}

static long
anxogg_sizeof_next_read (AnxSource * media, long bound)
{
  AnxOggData * aod = (AnxOggData *)media->custom_data;
  AnxOggPacket * aop;
  long bytes_to_read;

  if (aod->ignore_media) return -1;

  while ((aod->media_packets == NULL) && (oggz_read (aod->oggz, 1024)) != 0);

  if (aod->media_packets == NULL) {
    media->eos = 1;
    return 0;
  }

  aop = (AnxOggPacket *)aod->media_packets->data;
  bytes_to_read = aop->length - aod->current_offset;

  return bytes_to_read;
}

static int
anxogg_close (AnxSource * source)
{
  AnxOggData * aod = (AnxOggData *)source->custom_data;
  AnxOggTrack * aot;
  AnxSourceTrack * track;
  int i, n;

  oggz_close (aod->oggz);

  anx_list_free_with (aod->media_packets, (AnxFreeFunc)anxogg_packet_free);

  n = oggz_table_size (aod->logicals);
  for (i = 0; i < n; i++) {
    aot = (AnxOggTrack *)oggz_table_nth (aod->logicals, i, NULL);
    if (aot) {
      track = &(aot->source_track);
      anx_free (track->id);
      anx_free (track->content_type);
      free (aot);
    }
  }

  oggz_table_delete (aod->logicals);

#ifdef DEBUG
#ifdef DEBUG_FILE
  fclose (aod->df);
#endif
#endif

  anx_free (aod);

  anx_list_free (source->tracks);
  anx_free (source);

  return 0;
}

static struct _AnxImporter anx_ogg_importer = {
  (AnxImporterOpenFunc)anxogg_open,
  (AnxImporterOpenFDFunc)NULL,
  (AnxImporterCloseFunc)anxogg_close,
  (AnxImporterReadFunc)anxogg_read,
  (AnxImporterSizeofNextReadFunc)anxogg_sizeof_next_read,
  OGG_CONTENT_TYPE
};

static struct _AnxImporter anx_anx_importer = {
  (AnxImporterOpenFunc)anxogg_open,
  (AnxImporterOpenFDFunc)NULL,
  (AnxImporterCloseFunc)anxogg_close,
  (AnxImporterReadFunc)anxogg_read,
  (AnxImporterSizeofNextReadFunc)anxogg_sizeof_next_read,
  ANX_CONTENT_TYPE
};

static struct _AnxImporter anx_vorbis_importer = {
  (AnxImporterOpenFunc)anxogg_open,
  (AnxImporterOpenFDFunc)NULL,
  (AnxImporterCloseFunc)anxogg_close,
  (AnxImporterReadFunc)anxogg_read,
  (AnxImporterSizeofNextReadFunc)anxogg_sizeof_next_read,
  VORBIS_CONTENT_TYPE
};

static struct _AnxImporter anx_speex_importer = {
  (AnxImporterOpenFunc)anxogg_open,
  (AnxImporterOpenFDFunc)NULL,
  (AnxImporterCloseFunc)anxogg_close,
  (AnxImporterReadFunc)anxogg_read,
  (AnxImporterSizeofNextReadFunc)anxogg_sizeof_next_read,
  SPEEX_CONTENT_TYPE
};

static struct _AnxImporter anx_theora_importer = {
  (AnxImporterOpenFunc)anxogg_open,
  (AnxImporterOpenFDFunc)NULL,
  (AnxImporterCloseFunc)anxogg_close,
  (AnxImporterReadFunc)anxogg_read,
  (AnxImporterSizeofNextReadFunc)anxogg_sizeof_next_read,
  THEORA_CONTENT_TYPE
};

static struct _AnxImporter anx_xvid_importer = {
  (AnxImporterOpenFunc)anxogg_open,
  (AnxImporterOpenFDFunc)NULL,
  (AnxImporterCloseFunc)anxogg_close,
  (AnxImporterReadFunc)anxogg_read,
  (AnxImporterSizeofNextReadFunc)anxogg_sizeof_next_read,
  XVID_CONTENT_TYPE
};

AnxImporter *
anx_importer_init (int i)
{
  switch (i) {
  case 0:
    return &anx_ogg_importer;
  case 1:
    return &anx_anx_importer;
  case 2:
    return &anx_vorbis_importer;
  case 3:
    return &anx_speex_importer;
  case 4:
    return &anx_theora_importer;
  case 5:
    return &anx_xvid_importer;
  default:
    return NULL;
  }
}

