/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     lagrange_4_2d.c                                                */
/*                                                                          */
/* description:  piecewise quartic Lagrange elements in 2d                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS4_2D  15

static const REAL bary4_2d[N_BAS4_2D][N_LAMBDA] =
  {{    1.0,     0.0,     0.0, 0.0},
   {    0.0,     1.0,     0.0, 0.0},
   {    0.0,     0.0,     1.0, 0.0},
   {    0.0, 3.0/4.0, 1.0/4.0, 0.0},
   {    0.0,     0.5,     0.5, 0.0},
   {    0.0, 1.0/4.0, 3.0/4.0, 0.0},
   {1.0/4.0,     0.0, 3.0/4.0, 0.0},
   {    0.5,     0.0,     0.5, 0.0},
   {3.0/4.0,     0.0, 1.0/4.0, 0.0},
   {3.0/4.0, 1.0/4.0,     0.0, 0.0},
   {    0.5,     0.5,     0.0, 0.0},
   {1.0/4.0, 3.0/4.0,     0.0, 0.0},
   {1.0/2.0, 1.0/4.0, 1.0/4.0, 0.0},
   {1.0/4.0, 1.0/2.0, 1.0/4.0, 0.0},
   {1.0/4.0, 1.0/4.0, 1.0/2.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4v0_2d(const REAL l[N_LAMBDA])
{
  return((((32.0*l[0] - 48.0)*l[0] + 22.0)*l[0] - 3.0)*l[0]/3.0);
}

static const REAL *grd_phi4v0_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = ((128.0*l[0] - 144.0)*l[0] + 44.0)*l[0]/3.0 - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4v0_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = (128.0*l[0] - 96.0)*l[0] + 44.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4v1_2d(const REAL l[N_LAMBDA])
{
  return((((32.0*l[1] - 48.0)*l[1] + 22.0)*l[1] - 3.0)*l[1]/3.0);
}

static const REAL *grd_phi4v1_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = ((128.0*l[1] - 144.0)*l[1] + 44.0)*l[1]/3.0 - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4v1_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = (128.0*l[1] - 96.0)*l[1] + 44.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4v2_2d(const REAL l[N_LAMBDA])
{
  return((((32.0*l[2] - 48.0)*l[2] + 22.0)*l[2] - 3.0)*l[2]/3.0);
}

static const REAL *grd_phi4v2_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[2] = ((128.0*l[2] - 144.0)*l[2] + 44.0)*l[2]/3.0 - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4v2_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = (128.0*l[2] - 96.0)*l[2] + 44.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e00_2d(const REAL l[N_LAMBDA])
{
  return(((128.0*l[1] - 96.0)*l[1] + 16.0)*l[1]*l[2]/3.0);
}

static const REAL *grd_phi4e00_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = ((128*l[1] - 64.0)*l[1] + 16.0/3.0)*l[2];
  grd[2] = ((128*l[1] - 96.0)*l[1] + 16.0)*l[1]/3.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e00_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = (256.0*l[1] - 64.0)*l[2];
  D2[1][2] = D2[2][1] = (128.0*l[1] - 64.0)*l[1] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e10_2d(const REAL l[N_LAMBDA])
{
  return(((128.0*l[2] - 96.0)*l[2] + 16.0)*l[2]*l[0]/3.0);
}

static const REAL *grd_phi4e10_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = ((128*l[2] - 96.0)*l[2] + 16.0)*l[2]/3.0;
  grd[2] = ((128*l[2] - 64.0)*l[2] + 16.0/3.0)*l[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e10_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = (256.0*l[2] - 64.0)*l[0];
  D2[2][0] = D2[0][2] = (128.0*l[2] - 64.0)*l[2] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e20_2d(const REAL l[N_LAMBDA])
{
  return(((128.0*l[0] - 96.0)*l[0] + 16.0)*l[0]*l[1]/3.0);
}

static const REAL *grd_phi4e20_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = ((128*l[0] - 64.0)*l[0] + 16.0/3.0)*l[1];
  grd[1] = ((128*l[0] - 96.0)*l[0] + 16.0)*l[0]/3.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e20_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = (256.0*l[0] - 64.0)*l[1];
  D2[0][1] = D2[1][0] = (128.0*l[0] - 64.0)*l[0] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e01_2d(const REAL l[N_LAMBDA])
{
  return((4.0*l[1] - 1.0)*l[1]*(4.0*l[2] - 1.0)*l[2]*4.0);
}

static const REAL *grd_phi4e01_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = 4.0*(8.0*l[1] - 1.0)*l[2]*(4.0*l[2] - 1.0);
  grd[2] = 4.0*l[1]*(4.0*l[1] - 1.0)*(8.0*l[2] - 1.0);
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e01_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 32.0*l[2]*(4.0*l[2] - 1.0);
  D2[1][2] = D2[2][1] = 4.0*(8.0*l[1] - 1.0)*(8.0*l[2] - 1.0);
  D2[2][2] = 32.0*l[1]*(4.0*l[1] - 1.0);
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e11_2d(const REAL l[N_LAMBDA])
{
  return((4.0*l[2] - 1.0)*l[2]*(4.0*l[0] - 1.0)*l[0]*4.0);
}

static const REAL *grd_phi4e11_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = 4.0*l[2]*(4.0*l[2] - 1.0)*(8.0*l[0] - 1.0);
  grd[2] = 4.0*(8.0*l[2] - 1.0)*l[0]*(4.0*l[0] - 1.0);
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e11_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = 32.0*l[0]*(4.0*l[0] - 1.0);
  D2[2][0] = D2[0][2] = 4.0*(8.0*l[2] - 1.0)*(8.0*l[0] - 1.0);
  D2[0][0] = 32.0*l[2]*(4.0*l[2] - 1.0);
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e21_2d(const REAL l[N_LAMBDA])
{
  return((4.0*l[0] - 1.0)*l[0]*(4.0*l[1] - 1.0)*l[1]*4.0);
}

static const REAL *grd_phi4e21_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA];
  grd[0] = 4.0*(8.0*l[0] - 1.0)*l[1]*(4.0*l[1] - 1.0);
  grd[1] = 4.0*l[0]*(4.0*l[0] - 1.0)*(8.0*l[1] - 1.0);
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e21_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 32.0*l[1]*(4.0*l[1] - 1.0);
  D2[0][1] = D2[1][0] = 4.0*(8.0*l[0] - 1.0)*(8.0*l[1] - 1.0);
  D2[1][1] = 32.0*l[0]*(4.0*l[0] - 1.0);
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 2 at edge 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e02_2d(const REAL l[N_LAMBDA])
{
  return(((128.0*l[2] - 96.0)*l[2] + 16.0)*l[2]*l[1]/3.0);
}

static const REAL *grd_phi4e02_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = ((128*l[2] - 96.0)*l[2] + 16.0)*l[2]/3.0;
  grd[2] = ((128*l[2] - 64.0)*l[2] + 16.0/3.0)*l[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e02_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = (256.0*l[2] - 64.0)*l[1];
  D2[1][2] = D2[2][1] = (128.0*l[2] - 64.0)*l[2] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}


/*--------------------------------------------------------------------------*/
/*  basisfunction 2 at edge 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e12_2d(const REAL l[N_LAMBDA])
{
  return(((128.0*l[0] - 96.0)*l[0] + 16.0)*l[0]*l[2]/3.0);
}

static const REAL *grd_phi4e12_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = ((128*l[0] - 64.0)*l[0] + 16.0/3.0)*l[2];
  grd[2] = ((128*l[0] - 96.0)*l[0] + 16.0)*l[0]/3.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e12_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = (256.0*l[0] - 64.0)*l[2];
  D2[2][0] = D2[0][2] = (128.0*l[0] - 64.0)*l[0] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 2 at edge 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4e22_2d(const REAL l[N_LAMBDA])
{
  return(((128.0*l[1] - 96.0)*l[1] + 16.0)*l[1]*l[0]/3.0);
}

static const REAL *grd_phi4e22_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = ((128*l[1] - 96.0)*l[1] + 16.0)*l[1]/3.0;
  grd[1] = ((128*l[1] - 64.0)*l[1] + 16.0/3.0)*l[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4e22_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = (256.0*l[1] - 64.0)*l[0];
  D2[0][1] = D2[1][0] = (128.0*l[1] - 64.0)*l[1] + 16.0/3.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at center                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4c0_2d(const REAL l[N_LAMBDA])
{
  return((4.0*l[0] - 1.0)*l[0]*l[1]*l[2]*32.0);
}

static const REAL *grd_phi4c0_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = 32.0*(8.0*l[0] - 1.0)*l[1]*l[2];
  grd[1] = 32.0*(4.0*l[0] - 1.0)*l[0]*l[2];
  grd[2] = 32.0*(4.0*l[0] - 1.0)*l[0]*l[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4c0_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 256.0*l[1]*l[2];
  D2[0][1] = D2[1][0] = 32.0*(8.0*l[0] - 1.0)*l[2];
  D2[0][2] = D2[2][0] = 32.0*(8.0*l[0] - 1.0)*l[1];
  D2[1][2] = D2[2][1] = 32.0*(4.0*l[0] - 1.0)*l[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at center                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4c1_2d(const REAL l[N_LAMBDA])
{
  return(l[0]*(4.0*l[1] - 1.0)*l[1]*l[2]*32.0);
}

static const REAL *grd_phi4c1_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = 32.0*(4.0*l[1] - 1.0)*l[1]*l[2];
  grd[1] = 32.0*(8.0*l[1] - 1.0)*l[0]*l[2];
  grd[2] = 32.0*(4.0*l[1] - 1.0)*l[0]*l[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4c1_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][1] = D2[1][0] = 32.0*(8.0*l[1] - 1.0)*l[2];
  D2[0][2] = D2[2][0] = 32.0*(4.0*l[1] - 1.0)*l[1];
  D2[1][1] = 256.0*l[0]*l[2];
  D2[1][2] = D2[2][1] = 32.0*(8.0*l[1] - 1.0)*l[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 2 at center                                               */
/*--------------------------------------------------------------------------*/

static REAL phi4c2_2d(const REAL l[N_LAMBDA])
{
  return(l[0]*l[1]*(4.0*l[2] - 1.0)*l[2]*32.0);
}

static const REAL *grd_phi4c2_2d(const REAL l[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = 32.0*(4.0*l[2] - 1.0)*l[1]*l[2];
  grd[1] = 32.0*(4.0*l[2] - 1.0)*l[0]*l[2];
  grd[2] = 32.0*(8.0*l[2] - 1.0)*l[0]*l[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi4c2_2d(const REAL l[N_LAMBDA]))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][1] = D2[1][0] = 32.0*(4.0*l[2] - 1.0)*l[2];
  D2[0][2] = D2[2][0] = 32.0*(8.0*l[2] - 1.0)*l[1];
  D2[1][2] = D2[2][1] = 32.0*(8.0*l[2] - 1.0)*l[0];
  D2[2][2] = 256.0*l[0]*l[1];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

static const DOF *get_dof_indices4_2d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  index_vec[N_BAS4_2D];
  DOF         *rvec = idof ? idof : index_vec;
  int         i, j, n0;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = dof[i][n0];
  n0 = admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = dof[N_VERTICES_2D+i][n0];
      rvec[j++] = dof[N_VERTICES_2D+i][n0+1];
      rvec[j++] = dof[N_VERTICES_2D+i][n0+2];
    }
    else
    {
      rvec[j++] = dof[N_VERTICES_2D+i][n0+2];
      rvec[j++] = dof[N_VERTICES_2D+i][n0+1];
      rvec[j++] = dof[N_VERTICES_2D+i][n0];
    }
  }
  n0 = admin->n0_dof[CENTER];
  for (i = 0; i < 3; i++)
    rvec[12+i] = dof[6][n0+i];

  return((const DOF *) rvec);
}


static const S_CHAR *get_bound4_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("get_bound4_2d");
  static S_CHAR  bound_vec[N_BAS4_2D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i, j;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = el_info->vertex_bound[i];

  for (j = N_VERTICES_2D, i = 0; i < N_EDGES_2D; j += 3, i++)
    rvec[j] = rvec[j+1] = rvec[j+2] = el_info->edge_bound[i];
  
  for (i = 0; i < 3; i++)
    rvec[12+i] = INTERIOR;

  return((const S_CHAR *) rvec);
}

static const int *get_int_vec4_2d(const EL *el, const DOF_INT_VEC *vec,
				  int *ivec)
{
  FUNCNAME("get_int_vec4_2d");
  int            i, j, n0;
  static int     local_vec[N_BAS4_2D];
  int            *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  for (i = 0; i < 3; i++)
    rvec[12+i] = v[dof[6][n0+i]];

  return((const int *) rvec);
}

static const REAL *get_real_vec4_2d(const EL *el, const DOF_REAL_VEC *vec,
				    REAL *Rvec)
{
  FUNCNAME("get_real_vec4_2d");
  int            i, j, n0;
  static REAL    local_vec[N_BAS4_2D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  for (i = 0; i < 3; i++)
    rvec[12+i] = v[dof[6][n0+i]];

  return((const REAL *) rvec);
}

static const REAL_D *get_real_d_vec4_2d(const EL *el, 
					const DOF_REAL_D_VEC *vec,
					REAL_D *Rdvec)
{
  FUNCNAME("get_real_vec4_2d");
  int            i, j, k, n0;
  static REAL_D  local_vec[N_BAS4_2D];
  REAL_D         *v = nil, *rvec = Rdvec ? Rdvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	rvec[j][k] = v[dof[N_VERTICES_2D+i][n0]][k];
	rvec[j+1][k] = v[dof[N_VERTICES_2D+i][n0+1]][k];
	rvec[j+2][k] = v[dof[N_VERTICES_2D+i][n0+2]][k];
      }
      j += 3;
    }
    else
    {
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	rvec[j][k] = v[dof[N_VERTICES_2D+i][n0+2]][k];
	rvec[j+1][k] = v[dof[N_VERTICES_2D+i][n0+1]][k];
	rvec[j+2][k] = v[dof[N_VERTICES_2D+i][n0]][k];
      }
      j += 3;
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  for (i = 0; i < 3; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[12+i][k] = v[dof[6][n0+i]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *get_uchar_vec4_2d(const EL *el, const DOF_UCHAR_VEC *vec,
				       U_CHAR *uvec)
{
  FUNCNAME("get_uchar_vec4_2d");
  int            i, j, n0;
  static U_CHAR  local_vec[N_BAS4_2D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  for (i = 0; i < 3; i++)
    rvec[12+i] = v[dof[6][n0+i]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *get_schar_vec4_2d(const EL *el, const DOF_SCHAR_VEC *vec,
				       S_CHAR *svec)
{
  FUNCNAME("get_schar_vec4_2d");
  int            i, j, n0;
  static S_CHAR  local_vec[N_BAS4_2D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+2]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  for (i = 0; i < 3; i++)
    rvec[12+i] = v[dof[6][n0+i]];


  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,4,2,15)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,4,2,15)
  
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

static void real_refine_inter4_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_refine_inter4_2d");
  EL        *el;
  REAL      *v = nil;
  DOF       pdof[N_BAS4_2D];
  const DOF *cdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[cdof[2]] = v[pdof[10]];
  v[cdof[3]] = (0.2734375*v[pdof[0]] - 0.0390625*v[pdof[1]]
		+ 1.09375*v[pdof[9]] - 0.546875*v[pdof[10]]
		+ 0.21875*v[pdof[11]]);
  v[cdof[4]] = v[pdof[9]];
  v[cdof[5]] = (-0.0390625*v[pdof[0]] + 0.0234375*v[pdof[1]]
		+ 0.46875*v[pdof[9]] + 0.703125*v[pdof[10]]
		- 0.15625*v[pdof[11]]);
  v[cdof[6]] = (0.0234375*(v[pdof[0]] + v[pdof[1]]) 
		+ 0.0625*(-v[pdof[3]] - v[pdof[8]]) 
		+ 0.09375*(-v[pdof[9]] - v[pdof[11]]) + 0.140625*v[pdof[10]]
		+ 0.5625*(v[pdof[12]] + v[pdof[13]]));
  v[cdof[7]] = v[pdof[14]];
  v[cdof[8]] = (0.0390625*(-v[pdof[0]] - v[pdof[1]])
		+ 0.1875*(v[pdof[3]] + v[pdof[8]]-v[pdof[12]]-v[pdof[13]])
		+ 0.375*(-v[pdof[4]] - v[pdof[7]]) 
		+ 0.5*(v[pdof[5]] + v[pdof[6]])
		+ 0.03125*(v[pdof[9]] + v[pdof[11]])
		+ 0.015625*v[pdof[10]] + 0.75*v[pdof[14]]);
  v[cdof[12]] = (0.0234375*v[pdof[0]] - 0.0390625*v[pdof[1]]
		 + 0.125*(v[pdof[3]] - v[pdof[4]] - v[pdof[8]])
		 + 0.375*(v[pdof[7]] + v[pdof[12]] - v[pdof[13]])
		 - 0.03125*v[pdof[9]] - 0.046875*v[pdof[10]]
		 + 0.09375*v[pdof[11]] + 0.75*v[pdof[14]]);
  v[cdof[13]] = (0.0390625*(-v[pdof[0]] - v[pdof[1]]) + 0.0625*v[pdof[3]]
		 + 0.3125*(v[pdof[8]] - v[pdof[13]])
		 + 0.15625*(v[pdof[9]] + v[pdof[11]])
		 - 0.234375*v[pdof[10]] + 0.9375*v[pdof[12]]);
  v[cdof[14]] = v[pdof[12]];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cdof = get_dof_indices(el->child[1], admin, nil);

  v[cdof[6]] = (0.0234375*v[pdof[0]] - 0.0390625*v[pdof[1]]
		- 0.15625*v[pdof[9]] + 0.703125*v[pdof[10]]
		+ 0.46875*v[pdof[11]]);
  v[cdof[7]] = v[pdof[11]];
  v[cdof[8]] = (-0.0390625*v[pdof[0]] + 0.2734375*v[pdof[1]]
		+ 0.21875*v[pdof[9]] - 0.546875*v[pdof[10]]
		+ 1.09375*v[pdof[11]]);
  v[cdof[12]] = (0.0390625*(-v[pdof[0]] - v[pdof[1]])
		 + 0.3125*(v[pdof[3]] - v[pdof[12]]) + 0.0625*v[pdof[8]]
		 + 0.15625*(v[pdof[9]] + v[pdof[11]])
		 - 0.234375*v[pdof[10]] + 0.9375*v[pdof[13]]);
  v[cdof[13]] = (-0.0390625*v[pdof[0]] + 0.0234375*v[pdof[1]]
		 + 0.125*(-v[pdof[3]] - v[pdof[7]] + v[pdof[8]])
		 + 0.375*(v[pdof[4]] - v[pdof[12]] + v[pdof[13]])
		 + 0.09375*v[pdof[9]] - 0.046875*v[pdof[10]]
		 - 0.03125*v[pdof[11]] + 0.75*v[pdof[14]]);
  v[cdof[14]] = v[pdof[13]];

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;

  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[0]                                          */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[cdof[6]] = (0.0234375*(v[pdof[0]] + v[pdof[1]])
		+ 0.0625*(-v[pdof[3]] - v[pdof[8]])
		+ 0.09375*(-v[pdof[9]] - v[pdof[11]])
		+ 0.140625*v[pdof[10]] + 0.5625*(v[pdof[12]] + v[pdof[13]]));
  v[cdof[7]] = v[pdof[14]];
  v[cdof[8]] = (0.0390625*(-v[pdof[0]] - v[pdof[1]])
		+ 0.1875*(v[pdof[3]]+v[pdof[8]]-v[pdof[12]]-v[pdof[13]])
		+ 0.375*(-v[pdof[4]] - v[pdof[7]])
		+ 0.5*(v[pdof[5]] + v[pdof[6]])
		+ 0.03125*(v[pdof[9]] + v[pdof[11]])
		+ 0.015625*v[pdof[10]] + 0.75*v[pdof[14]]);
  v[cdof[12]] = (0.0234375*v[pdof[0]] - 0.0390625*v[pdof[1]]
		 + 0.125*(v[pdof[3]] - v[pdof[4]] - v[pdof[8]])
		 + 0.375*(v[pdof[7]] + v[pdof[12]] - v[pdof[13]])
		 - 0.03125*v[pdof[9]] - 0.046875*v[pdof[10]]
		 + 0.09375*v[pdof[11]] + 0.75*v[pdof[14]]);
  v[cdof[13]] = (0.0390625*(-v[pdof[0]] - v[pdof[1]]) + 0.0625*v[pdof[3]]
		 + 0.3125*(v[pdof[8]] - v[pdof[13]])
		 + 0.15625*(v[pdof[9]] + v[pdof[11]])
		 - 0.234375*v[pdof[10]] + 0.9375*v[pdof[12]]);
  v[cdof[14]] = v[pdof[12]];

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[1]                                          */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  v[cdof[12]] = (0.0390625*(-v[pdof[0]] - v[pdof[1]])
		 + 0.3125*(v[pdof[3]] - v[pdof[12]])
		 + 0.0625*v[pdof[8]] + 0.15625*(v[pdof[9]] + v[pdof[11]])
		 - 0.234375*v[pdof[10]] + 0.9375*v[pdof[13]]);
  v[cdof[13]] = (-0.0390625*v[pdof[0]] + 0.0234375*v[pdof[1]]
		 + 0.125*(-v[pdof[3]] - v[pdof[7]] + v[pdof[8]])
		 + 0.375*(v[pdof[4]] - v[pdof[12]] + v[pdof[13]])
		 + 0.09375*v[pdof[9]] - 0.046875*v[pdof[10]]
		 - 0.03125*v[pdof[11]] + 0.75*v[pdof[14]]);
  v[cdof[14]] = v[pdof[13]];

  return;
}


static void real_coarse_inter4_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_inter4_2d");
  EL        *el;
  REAL      *v = nil;
  DOF       pdof[N_BAS4_2D];
  const DOF *cdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[pdof[9]] = v[cdof[4]];
  v[pdof[10]] = v[cdof[2]];
  v[pdof[12]] = v[cdof[14]];
  v[pdof[14]] = v[cdof[7]];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cdof = get_dof_indices(el->child[1], admin, nil);

  v[pdof[11]] = v[cdof[7]];
  v[pdof[13]] = v[cdof[14]];

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;

  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[0]                                          */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[pdof[12]] = v[cdof[14]];
  v[pdof[14]] = v[cdof[7]];

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[1]                                          */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  v[pdof[13]] = v[cdof[14]];

  return;
}

static void real_coarse_restr4_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_restr4_2d");
  EL        *el;
  REAL      *v = nil;
  DOF       pdof[N_BAS4_2D];
  const DOF *cdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[pdof[0]] += (0.2734375*v[cdof[3]]
		 + 0.0390625*(-v[cdof[5]] - v[cdof[8]] - v[cdof[13]])
		 + 0.0234375*(v[cdof[6]] + v[cdof[12]]));
  v[pdof[1]] += (0.0390625*(-v[cdof[3]] - v[cdof[8]]
			    - v[cdof[12]] - v[cdof[13]])
		 + 0.0234375*(v[cdof[5]] + v[cdof[6]]));
  v[pdof[3]] += (0.0625*(-v[cdof[6]] + v[cdof[13]]) + 0.1875*v[cdof[8]]
		 + 0.125*v[cdof[12]]);
  v[pdof[4]] += (-0.375*v[cdof[8]] - 0.125*v[cdof[12]]);
  v[pdof[5]] += 0.5*v[cdof[8]];
  v[pdof[6]] += 0.5*v[cdof[8]];
  v[pdof[7]] += 0.375*(-v[cdof[8]] + v[cdof[12]]);
  v[pdof[8]] += (-0.0625*v[cdof[6]] + 0.1875*v[cdof[8]] - 0.125*v[cdof[12]]
		 + 0.3125*v[cdof[13]]);
  v[pdof[9]] = (v[cdof[4]] + 1.09375*v[cdof[3]] + 0.46875*v[cdof[5]]
		- 0.09375*v[cdof[6]] + 0.15625*v[cdof[13]]
		+ 0.03125*(v[cdof[8]] - v[cdof[12]]));
  v[pdof[10]] = (v[cdof[2]] - 0.546875*v[cdof[3]] + 0.703125*v[cdof[5]]
		 + 0.140625*v[cdof[6]] + 0.015625*v[cdof[8]]
		 - 0.046875*v[cdof[12]] - 0.234375*v[cdof[13]]);
  v[pdof[11]] = (0.21875*v[cdof[3]] + 0.15625*(-v[cdof[5]] + v[cdof[13]])
		 + 0.09375*(-v[cdof[6]] + v[cdof[12]]) + 0.03125*v[cdof[8]]);
  v[pdof[12]] = (v[cdof[14]] + 0.5625*v[cdof[6]] - 0.1875*v[cdof[8]]
		 + 0.375*v[cdof[12]] + 0.9375*v[cdof[13]]);
  v[pdof[13]] = (0.5625*v[cdof[6]] - 0.1875*v[cdof[8]] - 0.375*v[cdof[12]]
		 - 0.3125*v[cdof[13]]);
  v[pdof[14]] = (v[cdof[7]] + 0.75*(v[cdof[8]] + v[cdof[12]]));

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cdof = get_dof_indices(el->child[1], admin, nil);

  v[pdof[0]] += (0.0234375*v[cdof[6]] 
		 + 0.0390625*(-v[cdof[8]] - v[cdof[12]] - v[cdof[13]]));
  v[pdof[1]] += (0.0390625*(-v[cdof[6]] - v[cdof[12]])
		 + 0.2734375*v[cdof[8]] + 0.0234375*v[cdof[13]]);
  v[pdof[3]] += 0.3125*v[cdof[12]] - 0.125*v[cdof[13]];
  v[pdof[4]] += 0.375*v[cdof[13]];
  v[pdof[7]] += -0.125*v[cdof[13]];
  v[pdof[8]] += 0.0625*v[cdof[12]] + 0.125*v[cdof[13]];
  v[pdof[9]] += (0.15625*(-v[cdof[6]] + v[cdof[12]]) + 0.21875*v[cdof[8]]
		 + 0.09375*v[cdof[13]]);
  v[pdof[10]] += (0.703125*v[cdof[6]] - 0.546875*v[cdof[8]]
		  - 0.234375*v[cdof[12]] - 0.046875*v[cdof[13]]);
  v[pdof[11]] += (v[cdof[7]] + 0.46875*v[cdof[6]] + 1.09375*v[cdof[8]]
		  + 0.15625*v[cdof[12]] - 0.03125*v[cdof[13]]);
  v[pdof[12]] += -0.3125*v[cdof[12]] - 0.375*v[cdof[13]];
  v[pdof[13]] += v[cdof[14]] + 0.9375*v[cdof[12]] + 0.375*v[cdof[13]];
  v[pdof[14]] += 0.75*v[cdof[13]];

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;

  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[0]                                          */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[pdof[0]] += (0.0234375*(v[cdof[6]] + v[cdof[12]])
		 + 0.0390625*(-v[cdof[8]] - v[cdof[13]]));
  v[pdof[1]] += (0.0234375*v[cdof[6]] 
		 + 0.0390625*(-v[cdof[8]] - v[cdof[12]] - v[cdof[13]]));
  v[pdof[3]] += (0.0625*(-v[cdof[6]] + v[cdof[13]]) + 0.1875*v[cdof[8]]
		 + 0.12500000*v[cdof[12]]);
  v[pdof[4]] += -0.375*v[cdof[8]] - 0.125*v[cdof[12]];
  v[pdof[5]] += 0.5*v[cdof[8]];
  v[pdof[6]] += 0.5*v[cdof[8]];
  v[pdof[7]] += 0.375*(-v[cdof[8]] + v[cdof[12]]);
  v[pdof[8]] += (-0.0625*v[cdof[6]] + 0.1875*v[cdof[8]] - 0.125*v[cdof[12]]
		 + 0.3125*v[cdof[13]]);
  v[pdof[9]] += (-0.09375*v[cdof[6]] + 0.03125*(v[cdof[8]] - v[cdof[12]])
		 + 0.15625*v[cdof[13]]);
  v[pdof[10]] += (0.140625*v[cdof[6]] + 0.015625*v[cdof[8]]
		  - 0.046875*v[cdof[12]] - 0.234375*v[cdof[13]]);
  v[pdof[11]] += (0.09375*(-v[cdof[6]] + v[cdof[12]])
		  + 0.03125*v[cdof[8]] + 0.15625*v[cdof[13]]);
  v[pdof[12]] = (v[cdof[14]] + 0.5625*v[cdof[6]] - 0.1875*v[cdof[8]]
		 + 0.375*v[cdof[12]] + 0.9375*v[cdof[13]]);
  v[pdof[13]] = (0.5625*v[cdof[6]] - 0.1875*v[cdof[8]] - 0.375*v[cdof[12]]
		 - 0.3125*v[cdof[13]]);
  v[pdof[14]] = v[cdof[7]] + 0.75*(v[cdof[8]] + v[cdof[12]]);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[1]                                          */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  v[pdof[0]] += 0.0390625*(-v[cdof[12]] - v[cdof[13]]);
  v[pdof[1]] += -0.0390625*v[cdof[12]] + 0.0234375*v[cdof[13]];
  v[pdof[3]] += 0.3125*v[cdof[12]] - 0.125*v[cdof[13]];
  v[pdof[4]] += 0.375*v[cdof[13]];
  v[pdof[7]] += -0.125*v[cdof[13]];
  v[pdof[8]] += 0.0625*v[cdof[12]] + 0.125*v[cdof[13]];
  v[pdof[9]] += 0.15625*v[cdof[12]] + 0.09375*v[cdof[13]];
  v[pdof[10]] += -0.234375*v[cdof[12]] - 0.046875*v[cdof[13]];
  v[pdof[11]] += 0.15625*v[cdof[12]] - 0.03125*v[cdof[13]];
  v[pdof[12]] += -0.3125*v[cdof[12]] - 0.375*v[cdof[13]];
  v[pdof[13]] += v[cdof[14]] + 0.9375*v[cdof[12]] + 0.375*v[cdof[13]];
  v[pdof[14]] += 0.75*v[cdof[13]];

  return;
}


static void real_d_refine_inter4_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_refine_inter4_2d");
  EL        *el;
  REAL_D    *v = nil;
  int       k;
  DOF       pd[N_BAS4_2D];
  const DOF       *cd;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;


  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[cd[2]][k] = v[pd[10]][k];
    v[cd[3]][k] = (0.2734375*v[pd[0]][k] - 0.0390625*v[pd[1]][k]
		   + 1.09375*v[pd[9]][k] - 0.546875*v[pd[10]][k]
		   + 0.21875*v[pd[11]][k]);
    v[cd[4]][k] = v[pd[9]][k];
    v[cd[5]][k] = (-0.0390625*v[pd[0]][k] + 0.0234375*v[pd[1]][k]
		   + 0.46875*v[pd[9]][k] + 0.703125*v[pd[10]][k]
		   - 0.15625*v[pd[11]][k]);
    v[cd[6]][k] = (0.0234375*(v[pd[0]][k] + v[pd[1]][k]) 
		   + 0.0625*(-v[pd[3]][k] - v[pd[8]][k]) 
		   + 0.09375*(-v[pd[9]][k] - v[pd[11]][k])
		   + 0.140625*v[pd[10]][k]
		   + 0.5625*(v[pd[12]][k] + v[pd[13]][k]));
    v[cd[7]][k] = v[pd[14]][k];
    v[cd[8]][k] = (0.0390625*(-v[pd[0]][k] - v[pd[1]][k])
		   + 0.1875*(v[pd[3]][k] + v[pd[8]][k]
			     -v[pd[12]][k]-v[pd[13]][k])
		   + 0.375*(-v[pd[4]][k] - v[pd[7]][k]) 
		   + 0.5*(v[pd[5]][k] + v[pd[6]][k])
		   + 0.03125*(v[pd[9]][k] + v[pd[11]][k])
		   + 0.015625*v[pd[10]][k] + 0.75*v[pd[14]][k]);
    v[cd[12]][k] = (0.0234375*v[pd[0]][k] - 0.0390625*v[pd[1]][k]
		    + 0.125*(v[pd[3]][k] - v[pd[4]][k] - v[pd[8]][k])
		    + 0.375*(v[pd[7]][k] + v[pd[12]][k]-v[pd[13]][k])
		    - 0.03125*v[pd[9]][k] - 0.046875*v[pd[10]][k]
		    + 0.09375*v[pd[11]][k] + 0.75*v[pd[14]][k]);
    v[cd[13]][k] = (0.0390625*(-v[pd[0]][k] - v[pd[1]][k])
		    + 0.0625*v[pd[3]][k] 
		    + 0.3125*(v[pd[8]][k] - v[pd[13]][k])
		    + 0.15625*(v[pd[9]][k] + v[pd[11]][k])
		    - 0.234375*v[pd[10]][k] + 0.9375*v[pd[12]][k]);
    v[cd[14]][k] = v[pd[12]][k];
  }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cd = get_dof_indices(el->child[1], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[cd[6]][k] = (0.0234375*v[pd[0]][k] - 0.0390625*v[pd[1]][k]
		   - 0.15625*v[pd[9]][k] + 0.703125*v[pd[10]][k]
		   + 0.46875*v[pd[11]][k]);
    v[cd[7]][k] = v[pd[11]][k];
    v[cd[8]][k] = (-0.0390625*v[pd[0]][k] + 0.2734375*v[pd[1]][k]
		   + 0.21875*v[pd[9]][k] - 0.546875*v[pd[10]][k]
		   + 1.09375*v[pd[11]][k]);
    v[cd[12]][k] = (0.0390625*(-v[pd[0]][k] - v[pd[1]][k])
		    + 0.3125*(v[pd[3]][k] - v[pd[12]][k])
		    + 0.0625*v[pd[8]][k]
		    + 0.15625*(v[pd[9]][k] + v[pd[11]][k])
		    - 0.234375*v[pd[10]][k] + 0.9375*v[pd[13]][k]);
    v[cd[13]][k] = (-0.0390625*v[pd[0]][k] + 0.0234375*v[pd[1]][k]
		    + 0.125*(-v[pd[3]][k] - v[pd[7]][k] + v[pd[8]][k])
		    + 0.375*(v[pd[4]][k] - v[pd[12]][k] + v[pd[13]][k])
		    + 0.09375*v[pd[9]][k] - 0.046875*v[pd[10]][k]
		    - 0.03125*v[pd[11]][k] + 0.75*v[pd[14]][k]);
    v[cd[14]][k] = v[pd[13]][k];
  }

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[0]                                          */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[cd[6]][k] = (0.0234375*(v[pd[0]][k] + v[pd[1]][k])
		   + 0.0625*(-v[pd[3]][k] - v[pd[8]][k])
		   + 0.09375*(-v[pd[9]][k] - v[pd[11]][k])
		   + 0.140625*v[pd[10]][k]
		   + 0.5625*(v[pd[12]][k] + v[pd[13]][k]));
    v[cd[7]][k] = v[pd[14]][k];
    v[cd[8]][k] = (0.0390625*(-v[pd[0]][k] - v[pd[1]][k])
		   + 0.1875*(v[pd[3]][k] + v[pd[8]][k]
			     - v[pd[12]][k]-v[pd[13]][k])
		   + 0.375*(-v[pd[4]][k] - v[pd[7]][k])
		   + 0.5*(v[pd[5]][k] + v[pd[6]][k])
		   + 0.03125*(v[pd[9]][k] + v[pd[11]][k])
		   + 0.015625*v[pd[10]][k] + 0.75*v[pd[14]][k]);
    v[cd[12]][k] = (0.0234375*v[pd[0]][k] - 0.0390625*v[pd[1]][k]
		    + 0.125*(v[pd[3]][k] - v[pd[4]][k] - v[pd[8]][k])
		    + 0.375*(v[pd[7]][k] + v[pd[12]][k] - v[pd[13]][k])
		    - 0.03125*v[pd[9]][k] - 0.046875*v[pd[10]][k]
		    + 0.09375*v[pd[11]][k] + 0.75*v[pd[14]][k]);
    v[cd[13]][k] = (0.0390625*(-v[pd[0]][k] - v[pd[1]][k])
		    + 0.0625*v[pd[3]][k] + 0.3125*(v[pd[8]][k] - v[pd[13]][k])
		    + 0.15625*(v[pd[9]][k] + v[pd[11]][k])
		    - 0.234375*v[pd[10]][k] + 0.9375*v[pd[12]][k]);
    v[cd[14]][k] = v[pd[12]][k];
  }
/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[1]                                          */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[cd[12]][k] = (0.0390625*(-v[pd[0]][k] - v[pd[1]][k])
		    + 0.3125*(v[pd[3]][k] - v[pd[12]][k])
		    + 0.0625*v[pd[8]][k]
		    + 0.15625*(v[pd[9]][k] + v[pd[11]][k])
		    - 0.234375*v[pd[10]][k] + 0.9375*v[pd[13]][k]);
    v[cd[13]][k] = (-0.0390625*v[pd[0]][k] + 0.0234375*v[pd[1]][k]
		    + 0.125*(-v[pd[3]][k] - v[pd[7]][k] + v[pd[8]][k])
		    + 0.375*(v[pd[4]][k] - v[pd[12]][k] + v[pd[13]][k])
		    + 0.09375*v[pd[9]][k] - 0.046875*v[pd[10]][k]
		    - 0.03125*v[pd[11]][k] + 0.75*v[pd[14]][k]);
    v[cd[14]][k] = v[pd[13]][k];
  }
  return;
}


static void real_d_coarse_inter4_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_inter4_2d");
  EL        *el;
  REAL_D    *v = nil;
  int       k;
  DOF       pd[N_BAS4_2D];
  const DOF *cd;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[9]][k] = v[cd[4]][k];
    v[pd[10]][k] = v[cd[2]][k];
    v[pd[12]][k] = v[cd[14]][k];
    v[pd[14]][k] = v[cd[7]][k];
  }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cd = get_dof_indices(el->child[1], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[11]][k] = v[cd[7]][k];
    v[pd[13]][k] = v[cd[14]][k];
  }

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[0]                                          */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[12]][k] = v[cd[14]][k];
    v[pd[14]][k] = v[cd[7]][k];
  }
/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[1]                                          */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
    v[pd[13]][k] = v[cd[14]][k];

  return;
}

static void real_d_coarse_restr4_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_restr4_2d");
  EL        *el;
  REAL_D    *v = nil;
  int       k;
  DOF       pd[N_BAS4_2D];
  const DOF *cd;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += (0.2734375*v[cd[3]][k]
		    + 0.0390625*(-v[cd[5]][k] - v[cd[8]][k] - v[cd[13]][k])
		    + 0.0234375*(v[cd[6]][k] + v[cd[12]][k]));
    v[pd[1]][k] += (0.0390625*(-v[cd[3]][k] - v[cd[8]][k]
			       - v[cd[12]][k] - v[cd[13]][k])
		    + 0.0234375*(v[cd[5]][k] + v[cd[6]][k]));
    v[pd[3]][k] += (0.0625*(-v[cd[6]][k] + v[cd[13]][k]) + 0.1875*v[cd[8]][k]
		    + 0.125*v[cd[12]][k]);
    v[pd[4]][k] += (-0.375*v[cd[8]][k] - 0.125*v[cd[12]][k]);
    v[pd[5]][k] += 0.5*v[cd[8]][k];
    v[pd[6]][k] += 0.5*v[cd[8]][k];
    v[pd[7]][k] += 0.375*(-v[cd[8]][k] + v[cd[12]][k]);
    v[pd[8]][k] += (-0.0625*v[cd[6]][k] + 0.1875*v[cd[8]][k]
		    - 0.125*v[cd[12]][k]
		    + 0.3125*v[cd[13]][k]);
    v[pd[9]][k] = (v[cd[4]][k] + 1.09375*v[cd[3]][k] + 0.46875*v[cd[5]][k]
		   - 0.09375*v[cd[6]][k] + 0.15625*v[cd[13]][k]
		   + 0.03125*(v[cd[8]][k] - v[cd[12]][k]));
    v[pd[10]][k] = (v[cd[2]][k] - 0.546875*v[cd[3]][k] + 0.703125*v[cd[5]][k]
		    + 0.140625*v[cd[6]][k] + 0.015625*v[cd[8]][k]
		    - 0.046875*v[cd[12]][k] - 0.234375*v[cd[13]][k]);
    v[pd[11]][k] = (0.21875*v[cd[3]][k]
		    + 0.15625*(-v[cd[5]][k] + v[cd[13]][k])
		    + 0.09375*(-v[cd[6]][k] + v[cd[12]][k])
		    + 0.03125*v[cd[8]][k]);
    v[pd[12]][k] = (v[cd[14]][k] + 0.5625*v[cd[6]][k] - 0.1875*v[cd[8]][k]
		    + 0.375*v[cd[12]][k] + 0.9375*v[cd[13]][k]);
    v[pd[13]][k] = (0.5625*v[cd[6]][k] - 0.1875*v[cd[8]][k]
		    - 0.375*v[cd[12]][k] - 0.3125*v[cd[13]][k]);
    v[pd[14]][k] = (v[cd[7]][k] + 0.75*(v[cd[8]][k] + v[cd[12]][k]));
  }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cd = get_dof_indices(el->child[1], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += (0.0234375*v[cd[6]][k] 
		    + 0.0390625*(-v[cd[8]][k] - v[cd[12]][k] - v[cd[13]][k]));
    v[pd[1]][k] += (0.0390625*(-v[cd[6]][k] - v[cd[12]][k])
		    + 0.2734375*v[cd[8]][k] + 0.0234375*v[cd[13]][k]);
    v[pd[3]][k] += 0.3125*v[cd[12]][k] - 0.125*v[cd[13]][k];
    v[pd[4]][k] += 0.375*v[cd[13]][k];
    v[pd[7]][k] += -0.125*v[cd[13]][k];
    v[pd[8]][k] += 0.0625*v[cd[12]][k] + 0.125*v[cd[13]][k];
    v[pd[9]][k] += (0.15625*(-v[cd[6]][k] + v[cd[12]][k])
		    + 0.21875*v[cd[8]][k] + 0.09375*v[cd[13]][k]);
    v[pd[10]][k] += (0.703125*v[cd[6]][k] - 0.546875*v[cd[8]][k]
		     - 0.234375*v[cd[12]][k] - 0.046875*v[cd[13]][k]);
    v[pd[11]][k] += (v[cd[7]][k] + 0.46875*v[cd[6]][k] + 1.09375*v[cd[8]][k]
		     + 0.15625*v[cd[12]][k] - 0.03125*v[cd[13]][k]);
    v[pd[12]][k] += -0.3125*v[cd[12]][k] - 0.375*v[cd[13]][k];
    v[pd[13]][k] += v[cd[14]][k] + 0.9375*v[cd[12]][k] + 0.375*v[cd[13]][k];
    v[pd[14]][k] += 0.75*v[cd[13]][k];
  }

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;

  get_dof_indices(el, admin, pd);

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[0]                                          */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += (0.0234375*(v[cd[6]][k] + v[cd[12]][k])
		    + 0.0390625*(-v[cd[8]][k] - v[cd[13]][k]));
    v[pd[1]][k] += (0.0234375*v[cd[6]][k] 
		    + 0.0390625*(-v[cd[8]][k] - v[cd[12]][k] - v[cd[13]][k]));
    v[pd[3]][k] += (0.0625*(-v[cd[6]][k] + v[cd[13]][k]) + 0.1875*v[cd[8]][k]
		    + 0.12500000*v[cd[12]][k]);
    v[pd[4]][k] += -0.375*v[cd[8]][k] - 0.125*v[cd[12]][k];
    v[pd[5]][k] += 0.5*v[cd[8]][k];
    v[pd[6]][k] += 0.5*v[cd[8]][k];
    v[pd[7]][k] += 0.375*(-v[cd[8]][k] + v[cd[12]][k]);
    v[pd[8]][k] += (-0.0625*v[cd[6]][k] + 0.1875*v[cd[8]][k]
		    - 0.125*v[cd[12]][k] + 0.3125*v[cd[13]][k]);
    v[pd[9]][k] += (-0.09375*v[cd[6]][k]
		    + 0.03125*(v[cd[8]][k] - v[cd[12]][k])
		    + 0.15625*v[cd[13]][k]);
    v[pd[10]][k] += (0.140625*v[cd[6]][k] + 0.015625*v[cd[8]][k]
		     - 0.046875*v[cd[12]][k] - 0.234375*v[cd[13]][k]);
    v[pd[11]][k] += (0.09375*(-v[cd[6]][k] + v[cd[12]][k])
		     + 0.03125*v[cd[8]][k] + 0.15625*v[cd[13]][k]);
    v[pd[12]][k] = (v[cd[14]][k] + 0.5625*v[cd[6]][k] - 0.1875*v[cd[8]][k]
		    + 0.375*v[cd[12]][k] + 0.9375*v[cd[13]][k]);
    v[pd[13]][k] = (0.5625*v[cd[6]][k] - 0.1875*v[cd[8]][k]
		    - 0.375*v[cd[12]][k] - 0.3125*v[cd[13]][k]);
    v[pd[14]][k] = v[cd[7]][k] + 0.75*(v[cd[8]][k] + v[cd[12]][k]);
  }

/*--------------------------------------------------------------------------*/
/*  values on neighbour's child[1]                                          */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += 0.0390625*(-v[cd[12]][k] - v[cd[13]][k]);
    v[pd[1]][k] += -0.0390625*v[cd[12]][k] + 0.0234375*v[cd[13]][k];
    v[pd[3]][k] += 0.3125*v[cd[12]][k] - 0.125*v[cd[13]][k];
    v[pd[4]][k] += 0.375*v[cd[13]][k];
    v[pd[7]][k] += -0.125*v[cd[13]][k];
    v[pd[8]][k] += 0.0625*v[cd[12]][k] + 0.125*v[cd[13]][k];
    v[pd[9]][k] += 0.15625*v[cd[12]][k] + 0.09375*v[cd[13]][k];
    v[pd[10]][k] += -0.234375*v[cd[12]][k] - 0.046875*v[cd[13]][k];
    v[pd[11]][k] += 0.15625*v[cd[12]][k] - 0.03125*v[cd[13]][k];
    v[pd[12]][k] += -0.3125*v[cd[12]][k] - 0.375*v[cd[13]][k];
    v[pd[13]][k] += v[cd[14]][k] + 0.9375*v[cd[12]][k] + 0.375*v[cd[13]][k];
    v[pd[14]][k] += 0.75*v[cd[13]][k];
  }

  return;
}

static BAS_FCT *phi4_2d[N_BAS4_2D] = {phi4v0_2d, phi4v1_2d, phi4v2_2d,
				      phi4e00_2d, phi4e01_2d, phi4e02_2d,
				      phi4e10_2d, phi4e11_2d, phi4e12_2d,
				      phi4e20_2d, phi4e21_2d, phi4e22_2d,
				      phi4c0_2d,  phi4c1_2d,  phi4c2_2d};
static GRD_BAS_FCT  *grd_phi4_2d[N_BAS4_2D] = {
                                grd_phi4v0_2d,  grd_phi4v1_2d,  grd_phi4v2_2d,
				grd_phi4e00_2d, grd_phi4e01_2d, grd_phi4e02_2d,
				grd_phi4e10_2d, grd_phi4e11_2d, grd_phi4e12_2d,
				grd_phi4e20_2d, grd_phi4e21_2d, grd_phi4e22_2d,
				grd_phi4c0_2d,  grd_phi4c1_2d,  grd_phi4c2_2d};

static D2_BAS_FCT  *D2_phi4_2d[N_BAS4_2D] = {
  D2_phi4v0_2d,  D2_phi4v1_2d,  D2_phi4v2_2d,
  D2_phi4e00_2d, D2_phi4e01_2d, D2_phi4e02_2d,
  D2_phi4e10_2d, D2_phi4e11_2d, D2_phi4e12_2d,
  D2_phi4e20_2d, D2_phi4e21_2d, D2_phi4e22_2d,
  D2_phi4c0_2d,  D2_phi4c1_2d,  D2_phi4c2_2d};

static BAS_FCTS  lagrange4_2d = {"lagrange4_2d", 2, N_BAS4_2D, 4,
				 {1, 3, 3, 0}, /* VERTEX, CENTER, EDGE, FACE */
				 nil,
				 phi4_2d, grd_phi4_2d, D2_phi4_2d,
				 get_dof_indices4_2d, 
				 get_bound4_2d,
				 interpol4_2d,
				 interpol_d4_2d,
				 get_int_vec4_2d,
				 get_real_vec4_2d,
				 get_real_d_vec4_2d,
				 get_uchar_vec4_2d,
				 get_schar_vec4_2d,
				 real_refine_inter4_2d, 
				 real_coarse_inter4_2d,
				 real_coarse_restr4_2d,
				 real_d_refine_inter4_2d, 
				 real_d_coarse_inter4_2d,
				 real_d_coarse_restr4_2d,
				 bary4_2d,};
