/* vim:sw=4:sts=4
 * Library to use the Gtk2 widget library from Lua 5.1
 * Copyright (C) 2007 Wolfgang Oertl
 *
 * Handle different type of widgets (objects) - refcounting, freeing.
 *
 * Exported symbols:
 *  luagtk_inc_refcount
 *  luagtk_dec_refcount
 *  luagtk_get_refcount
 *  luagtk_get_widget_type
 *  luagtk_guess_widget_type
 *  luagtk_register_widget_type
 *  luagtk_init_widget
 */

#include "luagtk.h"
#include <string.h>		// strcmp, strncpy
#include <lauxlib.h>		// luaL_error
#include <glib.h>		// g_slice_free &c

static int next_type_nr = 0;
static struct widget_type widget_types[10];

/**
 * Free a GValue.  It might contain a string, for example, meaning additional
 * allocated memory.  This is freed.
 */
static void _gvalue_free(void *p)
{
    GValue *gv = (GValue*) p;
    if (G_IS_VALUE(gv))
	g_value_unset(gv);
    g_slice_free(GValue, gv);
}


/*
 * Most functions to free a structure are called very regularly, but some
 * are not.  Therefore the class name is looked up here first, and if nothing
 * is found, a regularly named function is looked for.
 *
 * Note: such structures are usually allocated using g_slice_alloc or similar,
 * and not g_malloc.  Using an inappropriate free function (g_slice_free vs.
 * g_free) leads to memory corruption!
 */
static struct free_methods {
    const char *class_name;
    const char *func_name;
    void (*free_func)(void*);
} free_methods[] = {
    { "GdkRegion",	"gdk_region_destroy" },
    { "GdkRectangle",	NULL },		// don't bother looking
    { "GValue",	NULL, &_gvalue_free },	// special free function
    { NULL, NULL }
};

/**
 * The given structure should be freed; it has no reference counting and
 * therefore an _unref function does not exist.  Try to find a _free
 * function and call it; otherwise, just g_slice_free() it.
 *
 * @param w   Pointer to the widget to be freed.
 */
static void _free_structure(struct widget *w)
{
    char func_name[50];
    struct func_info fi;
    struct free_methods *fm;

    if (!w->p) {
	fprintf(stderr, "%s Warning: trying to free NULL structure %p %s\n",
	    msgprefix, w, WIDGET_NAME(w));
	return;
    }

    for (fm=free_methods; ; fm++) {
	if (!fm->class_name) {
	    // not found - use default name.
	    if (luagtk_make_func_name(func_name, sizeof(func_name),
		WIDGET_NAME(w), "free"))
		return;
	    break;
	}

	if (!strcmp(fm->class_name, WIDGET_NAME(w))) {

	    // direct pointer to a free function - use it.
	    if (fm->free_func) {
		fm->free_func(w->p);
		w->p = NULL;
		return;
	    }

	    // An entry with NULL function name means that g_slice_free1 should
	    // be used.  It's just an optimization, but also avoids the
	    // warning below.
	    if (!fm->func_name)
		goto free_directly;

	    strncpy(func_name, fm->func_name, sizeof(func_name));
	    break;
	}
    }

    // if no special free function is available, just free() the structure.
    if (G_UNLIKELY(!find_func(func_name, &fi))) {
	if (runtime_flags & RUNTIME_DEBUG_MEMORY)
	    fprintf(stderr, "_free_structure: %s not found, using "
		"g_slice_free1\n", func_name);
free_directly:;
	const struct type_info *ti = STRUCT_INFO(w->type_idx);
	g_slice_free1(ti->st.struct_size, w->p);
	w->p = NULL;
	return;
    }

    // The function exists - call it.
    if (G_UNLIKELY(runtime_flags & RUNTIME_DEBUG_MEMORY))
	fprintf(stderr, "%p %p freeing memory using %s\n", w, w->p, func_name);

    void (*func)(void*);
    func = fi.func;
    func(w->p);
    w->p = NULL;
}


/**
 * Increase the reference counter of the object by 1.
 *
 * If FLAG_NEW_OBJECT is set, then this is a "new" object, just created
 * by the appropriate function.  This may require not increasing the refcount,
 * or doing something else (think about floating references).
 */
void luagtk_inc_refcount(struct widget *w, int flags)
{
    struct widget_type *wt = luagtk_get_widget_type(w);
    if (wt)
	wt->handler(w, WIDGET_REF, flags);
}


/**
 * Decrease the Gdk/Gtk/GObject reference counter by one.  This is done when
 * the Lua object representing it is garbage collected.
 *
 * @param w          The widget
 */
void luagtk_dec_refcount(struct widget *w)
{
    // Not if deleted e.g. by s_list_free.
    if (!w->is_deleted) {
	struct widget_type *wt = luagtk_get_widget_type(w);
	if (wt)
	    wt->handler(w, WIDGET_UNREF, 0);
    }
}



/**
 * Return the reference counter of the Gtk widget associated with this
 * widget structure.  Not all objects have such a counter.
 *
 * A negative value indicates an error.
 */
int luagtk_get_refcount(struct widget *w)
{
    struct widget_type *wt;

    /* NULL pointer to Lua object */
    if (!w)
	return -100;

    /* Lua object contains NULL pointer to Gtk object */
    if (!w->p)
	return -99;

    wt = luagtk_get_widget_type(w);
    if (!wt)
	return -98;

    return wt->handler(w, WIDGET_GET_REFCOUNT, 0);
}


/**
 * Retrieve the struct widget_type for the given widget.
 *
 * @param w  A widget
 * @return  The widget_type, or NULL on error.  In this case, an error is
 *   printed on stderr, too.
 */
struct widget_type *luagtk_get_widget_type(struct widget *w)
{
    if (w && w->widget_type < next_type_nr)
	return widget_types + w->widget_type;
    fprintf(stderr, "%p %p luagtk_get_widget_type: invalid widget (type %d)\n",
	w, w ? w->p : NULL, w ? w->widget_type : 0);
    return NULL;
}


/**
 * Determine the widget_type for a new widget proxy object.  This type
 * determines how the memory of this widget is managed - is reference
 * counting used, should it be free()d or nothing done, etc.
 *
 * @param L  Lua State
 * @param w  The new widget
 * @param flags  any of the FLAG_xxx constants: FLAG_NEW_OBJECT, FLAG_ALLOCATED
 */
void luagtk_guess_widget_type(lua_State *L, struct widget *w, int flags)
{
    int i, type_nr=-1, score=0;

    // printf("contest for type on %s, flags=%d\n", WIDGET_NAME(w), flags);
    for (i=0; i<next_type_nr; i++) {
	int rc = widget_types[i].handler(w, WIDGET_SCORE, flags);
	// printf("   %s = %d\n", widget_types[i].name, rc);
	if (rc > score) {
	    score = rc;
	    type_nr = i;
	}
    }

    if (G_UNLIKELY(type_nr == -1)) {
	lua_pop(L, 1);
	luaL_error(L, "[gtk] internal error: no appropriate "
	    "widget_type found");
    }

    w->widget_type = type_nr;
}


/**
 * Register a widget type.  Widgets have types (maybe a better name could be
 * found?) that govern how their memory is managed.  To make the library more
 * extensible, they are registered at initialization.
 *
 * @param name  Short name for this widget type (for debugging output)
 * @param handler  A function to handle various tasks
 * @return  The type_nr assigned to this widget type.
 */
int luagtk_register_widget_type(const char *name, widget_handler handler)
{
    int type_nr = next_type_nr ++;
    struct widget_type *wt = widget_types + type_nr;
    wt->name = name;
    wt->handler = handler;
    return type_nr;
}

static inline int _is_on_stack(void *p)
{
    volatile char c[30];    // large enough so it can't be in registers.
    long int ofs = ((char*)p) - c;
    if (ofs < 0)
	ofs = -ofs;
    return ofs < 36000;
}

/**
 * Handler for objects that don't need memory management or refcounting; it is
 * the fallback for non-allocated objects.
 *
 * GSList: the start as well as any item of such a list has the same type.
 * Therefore, they can't be free()d automatically.  use list:free() for that.
 *
 * @param w  A widget
 * @param op  Type of operation to perform
 * @param flags  If FLAG_ALLOCATED is set, rather not use this handler, else
 *   yes.
 * @return  For op==WIDGET_SCORE, the score, i.e. how well this handler would
 *   be for the given widget.
 */
static int _plain_handler(struct widget *w, widget_op op, int flags)
{
    if (op == WIDGET_SCORE) {
	if (!strcmp(WIDGET_NAME(w), "GSList"))
	    return 10;
	if (_is_on_stack(w->p))
	    return 5;
	return (flags & FLAG_ALLOCATED) ? 1 : 2;
    }

    return 0;
}

/**
 * Handler for objects that are allocated by this library using g_slice_alloc,
 * and that don't have refcounting.  They exist attached to their Lua object
 * proxy and are freed when the proxy is freed.
 */
static int _malloc_handler(struct widget *w, widget_op op, int flags)
{
    switch (op) {
	case WIDGET_SCORE:
	    if (flags & FLAG_NEW_OBJECT)
		return flags & FLAG_ALLOCATED ? 4 : 3;
	    // Non-new objects might be allocated, but they are not owned
	    // by this library and therefore must not be freed - let the
	    // plain handler take precedence.
	    return 0;

	case WIDGET_REF:
	    if (!(flags & FLAG_NEW_OBJECT))
		fprintf(stderr, "ref a malloc()ed widget of type %s?\n",
		    WIDGET_NAME(w));
	    break;

	case WIDGET_UNREF:
	    _free_structure(w);
	    break;
	
	default:
	    break;
    }

    return 0;
}

/**
 * Handler for GObject derived objects (widgets).  They have refcounting, so
 * when a Lua proxy object is created, the refcount usually is increased by
 * one, and when it is garbage collected, it must be decreased again.  Of
 * course, there are various fine points to it...
 */
static int _gobject_handler(struct widget *w, widget_op op, int flags)
{
    switch (op) {
	case WIDGET_SCORE:;
	    // not applicable to objects that have been malloc()ed directly.
	    if (flags & FLAG_ALLOCATED)
		return 0;
	    GType type_nr = g_type_from_name(WIDGET_NAME(w));
	    GType type_of_gobject = g_type_from_name("GObject");
	    return g_type_is_a(type_nr, type_of_gobject) ? 100 : 0;

	case WIDGET_GET_REFCOUNT:
	    return ((GObject*)w->p)->ref_count;

	case WIDGET_REF:

#ifdef GTK_OLDER_THAN_2_10
	    if (GTK_IS_OBJECT(w->p)) {
		// GtkObjects are created with a floating ref, so this code
		// works no matter whether it is a new or existing object.
		g_object_ref(w->p);
		gtk_object_sink((GtkObject*)w->p);
	    } else if (!(flags & FLAG_NEW_OBJECT)) {
		/* XXX wrong if _is_on_stack.  is_new is 1, but actually it
		 * isn't, so the refcount isn't increased... */
		// Normal objects are created with one reference.  Only add
		// another one of this is not a new object.
		g_object_ref(w->p);
	    }
#else
	    // non-Gtk objects (e.g. GdkDrawable, GtkStyle, PangoLayout) are
	    // not referenced if new.
	    if (GTK_IS_OBJECT(w->p) || !(flags & FLAG_NEW_OBJECT))
		g_object_ref_sink(w->p);

#endif
	    /*
	    fprintf(stderr, "%p %p %s ref - refcnt after = %d, floating=%d\n",
		w, w->p, WIDGET_NAME(w), ((GObject*)w->p)->ref_count,
		g_object_is_floating(w->p));
	    */
	    break;

	case WIDGET_UNREF:;
	    int ref_count = ((GObject*)w->p)->ref_count;
	    if (ref_count <= 0) {
		fprintf(stderr, "%p %p GC  %d %s - free with this refcount?\n",
		    w, w->p, ref_count, WIDGET_NAME(w));
		return 0;
	    }

	    // fprintf(stderr, "Unref %p %p %s\n", w, w->p, WIDGET_NAME(w));

	    // sometimes triggers a glib error here. w->p is a valid object,
	    // ref_count == 1.
	    g_object_unref(w->p);

	    /*
	    fprintf(stderr, "%p %p %s unref - refcnt now %d\n",
		w, w->p, WIDGET_NAME(w), ((GObject*)w->p)->ref_count);
	    */

	    w->p = NULL;
	    break;
    }

    return -1;
}


/**
 * PangoAttrList implements refcounting and isn't derived from GObject!  What
 * a nuisance.  It has its own, incompatible refcounting.
 */
static int _pango_attr_list_handler(struct widget *w, widget_op op, int flags)
{
    switch (op) {
	case WIDGET_SCORE:;
	    return strcmp(WIDGET_NAME(w), "PangoAttrList") ? 0 : 100;
	
	case WIDGET_REF:
	    // New objects already have their refcount set to 1.
	    if (!(flags & FLAG_NEW_OBJECT))
		pango_attr_list_ref(w->p);
	    return 0;
	
	case WIDGET_UNREF:
	    pango_attr_list_unref(w->p);
	    return 0;
	
	// See pango-attributes.c of libpango sources, which contains the
	// definition of struct _PangoAttrList.  The first element is the
	// refcount.
	case WIDGET_GET_REFCOUNT:
	    return * ((guint*) w->p);
    }

    return -1;
}


/**
 * Initialize the widget type handlers defined in this module.  There currently
 * is one more (in channel.c).
 */
void luagtk_init_widget(lua_State *L)
{
    luagtk_register_widget_type("plain", _plain_handler);
    luagtk_register_widget_type("malloc", _malloc_handler);
    luagtk_register_widget_type("gobject", _gobject_handler);
    luagtk_register_widget_type("pangoattrlist", _pango_attr_list_handler);
}


