/** vim:sw=4:sts=4
 * Simple hash lookup function (i.e., not using a minimal perfect hash function)
 * by Wolfgang 2006, 2007
 */

#include <string.h>
#include "luagtk_hash.h"
#include "hash-simple.h"
#include "config.h"

#ifndef HASHFUNC
 #error "Please define HASHFUNC."
#endif


extern unsigned int HASHFUNC(const char *p, int len);


/**
 * Given a key, look it up in the hash table.  Returns NULL if not found,
 * or a pointer to the data of this entry.
 */
const unsigned char *hash_search(const struct hash_info *hi, const char *key,
    int keylen, int *datalen)
{
    unsigned int hash, index, i;
    unsigned const char *p;
    unsigned char c;
    int bytenr, count;

    /* calculate hash and get the index */
    hash = HASHFUNC(key, keylen);
    index = hash % hi->hash_size;

    /* get offset, which is stored as char, short or int */
    switch (hi->ofs_size) {
	case 1:
	    index = hi->ofs[index];
	    break;
	case 2:
	    index = ((unsigned short*)hi->ofs)[index];
	    break;
	case 4:
	    index = ((unsigned int*)hi->ofs)[index];
	    break;
    }

    if (!index)
	return NULL;

    /* index is 1-based in order to use 0 for empty buckets. */
    p = hi->data + index - 1;

    /* for full hash, compare the complete hash values */
    /* layout: bucket size, entries... */
    /* entry: hash value, data length, data bytes */
    if (hi->full_hash) {
	count = (*p++) + 1;
	for (i=count; i; i--) {
	    if (*(unsigned int*)p == hash)
		goto found1;
	    p += sizeof(int) + 1 + p[sizeof(int)];
	}
	return NULL;
found1:
	/* skip hash value */
	p += sizeof(int);
    } else {
	/* get the bucket size and the hash byte nr */
	c = *p++;
	bytenr = (c >> 6);
	count = (c & 0x3f) + 1;
	i = count;

	if (count > 1) {
	    c = ((unsigned char*)&hash)[bytenr];
	    for (; i; i--) {
		if (*p == c)
		    goto found2;
		p += p[1] + 2;
	    }
	    return NULL;
found2:
	    /* skip hash byte */
	    p ++;
	}
    }

    /* p points to the data length, followed by data bytes.  count is the
     * number of items in the bucket, whereas count-i is the number of the
     * found entry.
     */

    *datalen = *p++;

    return p;
}

