/*- vim:sw=4:sts=4
 *
 * This is a collection of a few hash functions: djb2, fnv, sdbm, jenkins.
 * Which one is used depends on the data file, but usually jenkins will be
 * used.  These functions take a key and length, and return a hash value,
 * sometimes additionally a vector of three values.
 *
 */

#include "hash-cmph.h"
#include <stdio.h>		/* fprintf */

/**
 * These simple hash functions are all from the cmph sources, where they
 * can be found in the xxx_hash.c files.  Only the Jenkins method is rather
 * large, so I put it into a separate source file.
 */
static inline unsigned int hash_djb2(const unsigned char *key, int len)
{
    unsigned int hash = 5381;
    const unsigned char *key_end = key + len;

    while (key < key_end)
	hash = hash * 33 ^ *key++;

    return hash;
}

static inline unsigned int hash_fnv(const unsigned char *key, int len)
{
    unsigned int hash = 0;
    unsigned const char *key_end = key + len;

    while (key < key_end) {
	hash = (hash << 1) + (hash << 4) + (hash << 7) + (hash << 8)
	    + (hash << 24);
	hash ^= *key++;
    }

    return hash;
}

static inline unsigned int hash_sdbm(const unsigned char *key, int len)
{
    unsigned int hash = 0;
    unsigned const char *key_end = key + len;

    while (key < key_end) {
	hash = *key + (hash << 6) + (hash << 16) - hash;
	key ++;
    }
    
    return hash;
}


 /*
 * A supposedly good hash function.
 * by Bob Jenkins
 *
 * Source:
 * http://burtleburtle.net/bob/hash/doobs.html
 */

typedef  unsigned int  ub4;   /* unsigned 4-byte quantities */
typedef  unsigned char ub1;   /* unsigned 1-byte quantities */

#define hashsize(n) ((ub4)1<<(n))
#define hashmask(n) (hashsize(n)-1)

/*
--------------------------------------------------------------------
mix -- mix 3 32-bit values reversibly.
For every delta with one or two bits set, and the deltas of all three
  high bits or all three low bits, whether the original value of a,b,c
  is almost all zero or is uniformly distributed,
* If mix() is run forward or backward, at least 32 bits in a,b,c
  have at least 1/4 probability of changing.
* If mix() is run forward, every bit of c will change between 1/3 and
  2/3 of the time.  (Well, 22/100 and 78/100 for some 2-bit deltas.)
mix() was built out of 36 single-cycle latency instructions in a 
  structure that could supported 2x parallelism, like so:
      a -= b; 
      a -= c; x = (c>>13);
      b -= c; a ^= x;
      b -= a; x = (a<<8);
      c -= a; b ^= x;
      c -= b; x = (b>>13);
      ...
  Unfortunately, superscalar Pentiums and Sparcs can't take advantage 
  of that parallelism.  They've also turned some of those single-cycle
  latency instructions into multi-cycle latency instructions.  Still,
  this is the fastest good hash I could find.  There were about 2^^68
  to choose from.  I only looked at a billion or so.
--------------------------------------------------------------------
*/
#define mix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

/*
--------------------------------------------------------------------
hash() -- hash a variable-length key into a 32-bit value
  k       : the key (the unaligned variable-length array of bytes)
  len     : the length of the key, counting by bytes
  initval : can be any 4-byte value
Returns a 32-bit value.  Every bit of the key affects every bit of
the return value.  Every 1-bit and 2-bit delta achieves avalanche.
About 6*len+35 instructions.

The best hash table sizes are powers of 2.  There is no need to do
mod a prime (mod is sooo slow!).  If you need less than 32 bits,
use a bitmask.  For example, if you need only 10 bits, do
  h = (h & hashmask(10));
In which case, the hash table should have hashsize(10) elements.

If you are hashing n strings (ub1 **)k, do it like this:
  for (i=0, h=0; i<n; ++i) h = hash( k[i], len[i], h);

By Bob Jenkins, 1996.  bob_jenkins@burtleburtle.net.  You may use this
code any way you wish, private, educational, or commercial.  It's free.

See http://burtleburtle.net/bob/hash/evahash.html
Use for hash table lookup, or anything where one collision in 2^^32 is
acceptable.  Do NOT use for cryptographic purposes.
--------------------------------------------------------------------
*/

/**
 * Calculate the hash value using the Jenkins algorithm.
 *
 * @param k  key string
 * @param length  length of the key
 * @param initval  previous hash, or an arbitrary value
 * @param vector  place to store the internal state variables a, b, c
 * @return  the hash value
 */
static ub4 hash_jenkins(const unsigned char *k, int length, ub4 initval,
    ub4 *vector)
{
    ub4 a, b, c, len;

    /* Set up the internal state */
    len = length;
    a = b = 0x9e3779b9;		/* the golden ratio; an arbitrary value */
    c = initval;		/* the previous hash value */

    /* handle most of the key */
    while (len >= 12) {
	a += (k[0] +((ub4)k[1]<<8) +((ub4)k[2]<<16) +((ub4)k[3]<<24));
	b += (k[4] +((ub4)k[5]<<8) +((ub4)k[6]<<16) +((ub4)k[7]<<24));
	c += (k[8] +((ub4)k[9]<<8) +((ub4)k[10]<<16)+((ub4)k[11]<<24));
	mix(a,b,c);
	k += 12; len -= 12;
    }

    /* handle the last 11 bytes */
    c += length;
    switch(len) {
	case 11: c+=((ub4)k[10]<<24);
	case 10: c+=((ub4)k[9]<<16);
	case 9 : c+=((ub4)k[8]<<8);

	/* the first byte of c is reserved for the length */
	case 8 : b+=((ub4)k[7]<<24);
    	case 7 : b+=((ub4)k[6]<<16);
	case 6 : b+=((ub4)k[5]<<8);
	case 5 : b+=k[4];

	case 4 : a+=((ub4)k[3]<<24);
	case 3 : a+=((ub4)k[2]<<16);
	case 2 : a+=((ub4)k[1]<<8);
	case 1 : a+=k[0];
    }
    mix(a,b,c);

    if (vector) {
	vector[0] = a;
	vector[1] = b;
	vector[2] = c;
    }

    return c;
}


/**
 * Call the correct hash function depending on what was used.
 */
unsigned int compute_hash(const struct hash_state *state,
    const unsigned char *key, int keylen, unsigned int *vector)
{
    int hf = state->hashfunc;

    // methods with (optional) vector output
    switch (hf) {
	case CMPH_HASH_JENKINS:
	    return hash_jenkins(key, keylen, state->seed, vector);
    }

    if (vector) {
	fprintf(stderr, "compute_hash called for hash method %d, which doesn't "
	    "support vectors.\n", hf);
	return -1;
    }

    // methods without vector support
    switch (hf) {
	case CMPH_HASH_DJB2:
	    return hash_djb2(key, keylen);
	
	case CMPH_HASH_FNV:
	    return hash_fnv(key, keylen);

	case CMPH_HASH_SDBM:
	    return hash_sdbm(key, keylen);
    }

    fprintf(stderr, "Unsupported hash method %d\n", hf);
    return -1;
}


