/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */


#define DEATH_COUNT		40
#define FIRE_DELAY		6
#define MAX_PLAYERS		10
#define INITIAL_HEALTH		1000
#define GHOST_HEALTH_DELTA	1

#define ABS(x) (x<0?-x:x)

#include <stdlib.h>
#include <values.h>
#include <assert.h>
#include <curses.h>

#include "snipes.h"
#include "coords.h"
#include "collision.h"
#include "chars.h"
#include "weapons.h"
#include "sound.h"
#include "screen.h"


char global_player_eyestat=0;
char global_player_wall_dangerous=0;


struct player_info {
  int attribs;
  int lives;
  int health;
  int in_use;
  char has_collided;
  coordinate position;
  int fire_time;
} player[MAX_PLAYERS];


void player_choose_new_random_position( int index, screen_coords *sc )
{
  do {
    player[index].position.x = sc->square_width/2-1 +
      (random()%sc->square_count_x)*(sc->square_width-1);
    player[index].position.y = sc->square_height/2-1 +
      (random()%sc->square_count_y)*(sc->square_height-1);
  } while( check_collisions( player[index].position.x,
			     player[index].position.y, 2, 2,
			     MAZE_OBJECT_EVERYTHING ) );
}


void init_players( screen_coords *sc, int count, int a[], int wall_danger )
{
  int i;

  global_player_eyestat=0;
  global_player_wall_dangerous=wall_danger;
  
  assert( count <= MAX_PLAYERS && count >= 0 );

  for( i=0; i<count; i++ ) {
    player[i].lives=5;
    player[i].health=INITIAL_HEALTH;
    player[i].has_collided=0;
    player[i].in_use=1;
    player_choose_new_random_position( i, sc );
    player[i].attribs=a[i];
    player[i].fire_time=0;
  }
  for( ; i<MAX_PLAYERS; i++ )
    player[i].in_use=0;
}


coordinate get_player_pos( int index )
{
  assert( index <= MAX_PLAYERS && index >= 0 );
  return player[index].position;
}


int get_player_lives( int index )
{
  assert( index <= MAX_PLAYERS && index >= 0 );
  return player[index].lives;
}


void hide_player( int index, screen_coords *sc )
{
  coordinate screen_pos;
  assert( index <= MAX_PLAYERS && index >= 0 );
  
  if( coords_check_on_screen( sc, player[index].position, 0 ) ) {
    coords_to_screen_pos( sc, player[index].position, &screen_pos );

    screen_move( screen_pos.y, screen_pos.x );
    screen_addch(0);
    screen_addch(0);

    screen_move( screen_pos.y+1, screen_pos.x );
    screen_addch(0);
    screen_addch(0);
  }
}


void output_death_char( int i )
{
  switch( i%4 )
  {
    case 0:
      screen_addch( PLAYER_DIE_1 );
      break;
    case 1:
      screen_addch( PLAYER_DIE_2 );
      break;
    case 2:
      screen_addch( PLAYER_DIE_3 );
      break;
    case 3:
      screen_addch( PLAYER_DIE_4 );
      break;
  }
}


void show_player( int index, screen_coords *sc )
{
  coordinate screen_pos;
  assert( index <= MAX_PLAYERS && index >= 0 );
  
  if( coords_check_on_screen( sc, player[index].position, 0 ) ) {
    coords_to_screen_pos( sc, player[index].position, &screen_pos );

    screen_setcolorpair(player[index].attribs);

    if( player[index].has_collided || player[index].lives == 0 ) {
      screen_move( screen_pos.y, screen_pos.x );
      output_death_char( player[index].has_collided   );
      output_death_char( player[index].has_collided+1 );

      screen_move( screen_pos.y+1, screen_pos.x );
      output_death_char( player[index].has_collided+1 );
      output_death_char( player[index].has_collided   );
    }
    else {
      screen_move( screen_pos.y, screen_pos.x );
      screen_addch( global_player_eyestat?PLAYER_SMALLEYE:PLAYER_BIGEYE );
      screen_addch( global_player_eyestat?PLAYER_SMALLEYE:PLAYER_BIGEYE );

      screen_move( screen_pos.y+1, screen_pos.x );
      screen_addch(PLAYER_LEFTFACE);
      screen_addch(PLAYER_RIGHTFACE);
    }
  }
}


void show_players( screen_coords *sc )
{
  int i;

  for( i=0; i<MAX_PLAYERS; i++ )
    if( player[i].in_use )
      show_player( i, sc );
}


void player_collision_check( int index, screen_coords *sc )
{
  assert( index <= MAX_PLAYERS && index >= 0 );

  if( player[index].has_collided ) {
    if( ++player[index].has_collided == DEATH_COUNT ) {
      player[index].has_collided = 0;
      player[index].lives--;
      collision_map_remove_obj( player[index].position.x,
				player[index].position.y, 2, 2,
				MAZE_OBJECT_PLAYER );
      if( player[index].lives != 0 ) {
	player[index].health=INITIAL_HEALTH;
	player_choose_new_random_position( index, sc );
	collision_map_place_obj( player[index].position.x,
				 player[index].position.y, 2, 2,
				 MAZE_OBJECT_PLAYER );
      }
    }
  }
  else {
    /* Ghost contact has a negative impact on a player's health */
    if( check_adjacent_collisions( player[index].position.x,
				   player[index].position.y, 2, 2,
				   MAZE_OBJECT_SNIPE_GHOST ) ) {
      player[index].health -= GHOST_HEALTH_DELTA;
      if( player[index].health < 0 ) {
	sound_post_player_death();
	player[index].has_collided = 1;
      }
    }

    /* Getting hit by a bullet has an even more negative impact on
     * a player's health
     */
    if( check_collisions( player[index].position.x,
			  player[index].position.y, 2, 2,
			  MAZE_OBJECT_SNIPE_BULLET ) ) {
      sound_post_player_death();
      player[index].has_collided = 1;
    }
  }
}


void all_player_collision_check( screen_coords *sc )
{
  int i;

  for( i=0; i<MAX_PLAYERS; i++ )
    if( player[i].in_use )
      player_collision_check( i, sc );
}


void player_move( screen_coords *sc, int index, coordinate delta )
{
  coordinate tp;
  assert( index <= MAX_PLAYERS && index >= 0 );
  
  player_collision_check( index, sc );
  
  if( player[index].has_collided )
    return;
  
  tp = player[index].position;

  coords_add_delta( sc, &tp, delta );

  if( global_player_wall_dangerous )
    if( check_collisions( tp.x, tp.y, 2, 2, MAZE_OBJECT_WALL ) ) {
      sound_post_player_death();
      player[index].has_collided = 1;
      return;
    }

  if( !check_collisions( tp.x, tp.y, 2, 2,
			 MAZE_OBJECT_EVERYTHING^MAZE_OBJECT_PLAYER ) ) {

    collision_map_remove_obj( player[index].position.x,
			      player[index].position.y, 2, 2,
			      MAZE_OBJECT_PLAYER );
    player[index].position = tp;
    collision_map_place_obj( player[index].position.x,
			     player[index].position.y, 2, 2,
			     MAZE_OBJECT_PLAYER );
  }
  if( check_collisions( tp.x, tp.y, 2, 2,
			MAZE_OBJECT_SNIPE_BULLET|MAZE_OBJECT_PLAYER_BULLET ) ) {
      player[index].has_collided = 1;
      sound_post_player_death();
      return;
  }
}


void player_fire( screen_coords *sc, int index, coordinate direction,
		  int elapsed_time )
{
  coordinate tc, tp;
  assert( index <= MAX_PLAYERS && index >= 0 );

  /* Now try to fire. */
  if( (direction.x != 0 || direction.y != 0) &&
      player[index].fire_time+FIRE_DELAY < elapsed_time ) {
    
    tc = direction;
    if( tc.x == 1 )
      tc.x = 2;
    if( tc.y == 1 )
      tc.y = 2;
  
    tp = player[index].position;
    coords_add_delta( sc, &tp, tc );

#if 0
    if( !check_collisions( tp.x, tp.y, 1, 1, MAZE_OBJECT_EVERYTHING ) ) {
      player[index].fire_time = elapsed_time;
      add_weapon( tp, direction, 1 );
    }
#else
    player[index].fire_time = elapsed_time;
    add_weapon( tp, direction, 1 );
#endif
  }
}


void find_nearest_player( screen_coords *sc, coordinate pos, coordinate *player_delta )
{
  int i;
  int deltax, deltay, dist, old_dist=MAXINT, player_index=0;
  int save_deltax=0, save_deltay=0;

  for( i=0; i<MAX_PLAYERS; i++ )
    if( player[i].in_use ) {
      deltax = player[i].position.x - pos.x;
      deltay = player[i].position.y - pos.y;
      
      if( ABS(deltax) > sc->max_x/2 ) {
	if( deltax < 0 )
	  deltax = sc->max_x + deltax;
	else
	  deltax = deltax - sc->max_x;
      }

      if( ABS(deltay) > sc->max_y/2 ) {
	if( deltax < 0 )
	  deltay = sc->max_y + deltay;
	else
	  deltay = deltay - sc->max_y;
      }

      dist = deltax * deltax + deltay * deltay;
      if( dist < old_dist ) {
	player_index = i;
	save_deltax = deltax;
	save_deltay = deltay;
	old_dist = dist;
      }
    }

  player_delta->x = save_deltax;
  player_delta->y = save_deltay;
}


void eyechange( void )
{
  global_player_eyestat=((global_player_eyestat+1)%2);
}
