
/*
    Copyright (c) 2008 Andrew Caudwell (acaudwell@gmail.com)
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:
    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. The name of the author may not be used to endorse or promote products
       derived from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
    OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
    NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
    DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
    THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
    THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "display.h"
#include "sdlapp.h"

Display display;

Display::Display() {
    clearColour = vec3f(0.0f,0.0f,0.0f);
}

Display::~Display() {
}

void Display::displayArgs(int argc, char *argv[], int* xres, int* yres, bool* fullscreen) {

    for (int i=1; i<argc; i++) {
        debugLog("argv[%d] = %s\n", i, argv[i]);

        if (!strcmp(argv[i],"-f")) *fullscreen = 1;
        else if (!strcmp(argv[i],"-w")) *fullscreen = 0;

        //get video mode
        if(strchr(argv[i], '-') != 0 && strlen(argv[i])>1) {
            std::string argstr(argv[i], 1, strlen(argv[i])-1);
            debugLog("%s\n", argstr.c_str());
            size_t x = argstr.rfind("x");

            if(x != std::string::npos) {
                std::string widthstr  = argstr.substr(0, x);
                std::string heightstr = argstr.substr(x+1);

                int width = atoi(widthstr.c_str());
                int height = atoi(heightstr.c_str());

                debugLog("w=%d, h=%d\n",width,height);

                if(width!=0 && height!=0) {
                        *xres = width;
                        *yres = height;
                }
            }
        }
    }
}

void Display::setClearColour(vec3f colour) {
    clearColour = colour;
}

int Display::SDLFlags(bool fullscreen) {
    int flags = SDL_OPENGL | SDL_HWSURFACE | SDL_ANYFORMAT | SDL_DOUBLEBUF;
    if (fullscreen) flags |= SDL_FULLSCREEN;
    return flags;
}

void Display::init(std::string window_title, int width, int height, bool fullscreen) {
    this->width  = width;
    this->height = height;

    this->fullscreen = fullscreen;

    int flags = SDLFlags(fullscreen);

    SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO);
    atexit(SDL_Quit);

    //vsync
    SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, 1);

    SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);

#ifdef _WIN32
    surface = SDL_SetVideoMode(width, height, 32, flags);
#else
    surface = SDL_SetVideoMode(width, height, 24, flags);
#endif

    if (!surface) {
        printf("SDL initalization failed: %s\n", SDL_GetError());
        exit(1);
    }

    SDL_WM_SetCaption(window_title.c_str(),0);
}

void Display::quit() {
    SDL_Quit();
}

void Display::update() {
    SDL_GL_SwapBuffers();
}

void Display::clear() {
    glClearColor(clearColour.x, clearColour.y, clearColour.z, 0.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

void Display::mode3D(float fov, float znear, float zfar) {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(fov, (GLfloat)width/(GLfloat)height, znear, zfar);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}

void Display::mode2D() {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, width, height, 0, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}

vec4f Display::currentColour() {
    vec4f colour;
   	glGetFloatv(GL_CURRENT_COLOR, colour);
   	return colour;
}

GLuint Display::createTexture(int width, int height, int mipmaps, int clamp, int trilinear, int format, unsigned int* data) {

    GLuint textureid;

    glGenTextures(1, &textureid);
    glBindTexture(GL_TEXTURE_2D, textureid);

    if(mipmaps) {
        gluBuild2DMipmaps(GL_TEXTURE_2D, 4, width, height, format, GL_UNSIGNED_BYTE, data);
    } else {
    	glTexImage2D(GL_TEXTURE_2D, 0, 4, width, height, 0, format, GL_UNSIGNED_BYTE, data);
    }

    //GL_LINEAR_MIPMAP_LINEAR  - Trilinear
    //GL_LINEAR_MIPMAP_NEAREST - Bilinear

    if(trilinear) {
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    } else {
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
    }

    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST_MIPMAP_NEAREST);

    if(clamp) {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    }

    return textureid;
}

