/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/command.h"
#include "include/message.h"

/*! \brief Konstruktor polecenia

	Wołany przez konstruktory poleceń dziedziczących z klasy
*/
LWCommand::LWCommand(): argc(0)
{
}

/*! \brief Deincjalizacja polecenia

	Metoda może być przeciążona w poleceniach dziedziczących z klasy.
*/
void LWCommand::reset1()
{
}

/*! \brief Deincjalizacja polecenia

	Deincjalizuje polecenie. Wewnętrznie woła metodę reset1.
*/
void LWCommand::reset()
{
	reset1();
	
	for (guint i=0; i < argc; i++)
		if (arg[i] != NULL)
			if (FALSE == arg[i]->isVariable()) {
				delete arg[i];
				arg[i] = NULL;
			}
		
	argc = 0;
}

/*! \brief Nadanie argumentu

	Nadaje wartość dla argumentu dla polecenia.
	\param n Numer argumentu numerowany od 0
	\param value Wartość argumentu
*/
void LWCommand::setArgument (guint n, LWValue *value)
{
	if (value != NULL)
		checkArgument (n, value);
	
	arg[n] = value;
}

/*! \brief Ustawianie ilości argumentów

	Ustawia ilość argumentów, które dostarczono do polecenia
	\param n Ilość argumentów
*/
void LWCommand::setArgc (guint n)
{
	checkArgc (n);
	
	argc = n;
}

/*! \brief Sprawdzenie poprawności argumentów

	Sprawdza poprawność argumentu. Metoda może być przeciążona
	przez polecenia dziedziczącące z klasy
	\param n Numer argumentu numerowany od 0
	\param value Wartość argumentu
*/
void LWCommand::checkArgument (guint n, LWValue *value)
{
	(void) value;
	(void) n;	
}

/*! \brief Sprawdzenie czy ilość argumentów jest poprawna

	Sprawdza czy ilość argumentów dostraczonych do polecenia
	jest dopuszczalna. Metoda może być przeciążona przez polecenia
	dziedziczącące z klasy. Nie przeciążona metoda zwraca wyjątek
	jeśli dostarczono jakiekolwiek argumenty.
	\param n Ilość argumentów
*/
void LWCommand::checkArgc (guint n)
{
	if (n != 0) 
		throw new LWMessage (LW_ERROR_WrongNumberOfArguments);
}	

/*! \brief Pobieranie argumentów

	Pobiera argumenty do tablicy, pobierając brakujące wartości
	ze stosu.
	\param args Tablica, w której znajdują się argumenty
	\param stack Stos
*/
void LWCommand::getArguments (LWValue *args[], GQueue *stack)
{
	g_return_if_fail (argc > 0);
	
	for (gint i= (gint) argc-1; i >= 0; i--)
		if (arg[i] == NULL) {
			args[i] = (LWValue *) g_queue_pop_head (stack);
		} else
			args[i] = arg[i];
}

/*! \brief Zwolnienie argumentów

	Zwalnia argumenty znajdujące się w tablicy
	\param args Tablica, w której znajdują się argumenty
*/
void LWCommand::freeArguments (LWValue *args[])
{
	for (guint i= 0; i < argc; i++)
		if (arg[i] == NULL)
			if (FALSE == args[i]->isVariable())
				delete args[i];
}


/*! \brief Zwolnienie argumentów

	Zwalnia argumenty znajdujące się w tablicy
	\param args Tablica, w której znajdują się argumenty
*/
void LWCommand::pushArguments (LWValue *args[], GQueue *stack)
{
	for (guint i= 0; i < argc; i++)
		if (arg[i] == NULL)
			if (FALSE == args[i]->isVariable())
				g_queue_push_head (stack, (gpointer) args[i]);
}

/*! \brief Przypisanie wyniku

	Przekazuje wartość wyniku na stos
	\param value Wartość
	\param stack Stos
*/
void LWCommand::setReturn (LWValue *value, GQueue *stack)
{
	g_return_if_fail (hasReturn() == TRUE);
	
	g_queue_push_head (stack, (gpointer) value);
}

/*! \brief Czy polecenie zwraca wynik

	Tylko polecenia przeciążające tą metodę w taki sposób by
	zwracała prawdę, mogę zwracać wynik. Nie przeciążona
	metoda zwraca fałsz.
*/
gboolean LWCommand::hasReturn ()
{
	return FALSE;
}

/*! \brief Czy polecenie jest pętlą

	Tylko polecenia przeciążające tą metodę w taki sposób by
	zwracała prawdę, mogę implementować pętlę. Nie przeciążona
	metoda zwraca fałsz.
*/
gboolean LWCommand::isLoop ()
{
	return FALSE;	
}

/*! \brief Czy polecenie jest parzyste

	Czy polecenie składa się z dwóch części. Przykładami dwu członowych
	poleceń są nawiasy okrągłe oraz polecenia bloków logicznych.
	Nie przeciążona metoda zwraca fałsz.
*/
gboolean LWCommand::isSegment()
{
	return FALSE;	
}

/*! \brief Wykonanie polecenia

	Wykonuje polecenie. Wewnętrznie woła metodę execute1, która musi
	być zaimplementowana w poleceniu dziedziczącemu z tej klasy.
	\param context Kontekst polecenia
*/
void LWCommand::execute (LWContext *context, gboolean resume)
{
LWValue *r;
	
if (argc != 0) {
	LWValue *args[argc];
	GNode *lastip = context->instrPtr;
	getArguments (args, context->stack);

	try {
		r = (resume == FALSE) ? execute1 (context, argc, args) : resume1 (context, argc, args);
	} catch (LWMessage *msg) {

		/* Why it doesn't work with win32 ??? */
#ifndef _WIN32
		freeArguments (args);
#endif
		throw msg;
	}
	
	if (lastip != context->instrPtr || FALSE == context->resume)
		freeArguments (args);
	else
		pushArguments (args, context->stack);
} else
	r = (resume == FALSE) ? execute1 (context, argc, NULL) : resume1 (context, argc, NULL);

if (context->resume == FALSE && TRUE == hasReturn()) {
	g_return_if_fail (r != NULL);
	setReturn (r, context->stack);
} else
	g_return_if_fail (r == NULL);
}

/*! \brief Pobranie priorytetu

	Pobiera priorytet polecenia. Nie przeciążona metoda zwraca 
	priorytet 1 (domyślny dla poleceń dla czarodzieja)
*/
guint LWCommand::getPriority ()
{
	return 1;
}

gboolean LWCommand::canBeSkiped ()
{
	return FALSE;
}

/*! \brief Pobranie rodzaju wiązania

	Pobiera rodzaj wiązania jakie posiada polecenie.
*/
LWLink LWCommand::getLinkType () 
{
	return LW_LINK_LEFT;
}

gboolean LWCommand::isCommand ()
{
	return TRUE;
}

gboolean LWCommand::canClone ()
{
	return TRUE;
}

gboolean LWCommand::requiresLogicBlock ()
{
	return isLoop();
}

LWValue *LWCommand::resume1 (LWContext *context, guint argc, LWValue *args[])
{
	return execute1 (context, argc, args);
}
