/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <cmath>
#include <cairomm/context.h>

#include "widget_chart.hpp"


using namespace LIFEO;


ChartPoint::ChartPoint( Date date_0, int value_0 )
	:	date( date_0 ), value( value_0 )
{}

ChartPoints::ChartPoints( void )
	:	value_min( -1 ), value_max( -1 ), flag_use_current( false )
{
}

void
ChartPoints::add( ChartPoint &point )
{
	push_back( point );
	if( point.value > value_max )
		value_max = point.value;
	if( point.value < value_min || value_min < 0 )
		value_min = point.value;
}

void
ChartPoints::add( Date date, int value )
{
	ChartPoint point( date, value );
	push_back( point );
	if( point.value > value_max )
		value_max = point.value;
	if( point.value < value_min || value_min < 0 )
		value_min = point.value;
}

WidgetChart::WidgetChart( BaseObjectType *cobject,
						  const Glib::RefPtr< Gtk::Builder >& )
	:	DrawingArea( cobject ), m_points( NULL ), m_overview_height( 0.0 ),
	m_flag_button_pressed ( false ), m_flag_pointer_hovered( false )
{
	set_events( Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK |
			    Gdk::POINTER_MOTION_MASK | Gdk::LEAVE_NOTIFY_MASK | Gdk::SCROLL_MASK );
	m_font_main = Cairo::ToyFontFace::create( "FreeSans",
				  Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_BOLD );
}

void
WidgetChart::update_col_geom( bool flag_first_time )
{
	if( m_points )
	{
		m_col_count = m_points->size() >= column_count_max ?
				column_count_max : m_points->size();

		int col_start_max = m_points->size() - m_col_count;
		if( flag_first_time )
		{
			if( m_points->flag_use_current )
			{
				int col_start = m_points->current - m_col_count / 2;
				if( col_start < 0 )
					m_col_start = 0;
				else
				if ( col_start > col_start_max )
					m_col_start = col_start_max;
				else
					m_col_start = col_start;
			}
			else
				m_col_start = col_start_max;
		}
		else
		if( int( m_col_start ) > col_start_max )
			m_col_start = col_start_max;

		step_x = m_col_count == 1 ? length : length / ( m_col_count - 1 );

		m_overview_height = m_col_count < m_points->size() ? OVERVIEW_HEIGHT : 0.0;
		y_max = m_height - 2 * bar_height - m_overview_height;
		y_mid = ( y_max + y_min ) / 2;
		amplitude = y_max - y_min;
		coefficient = m_points->value_max == m_points->value_min ? 0.0 :
				amplitude / ( m_points->value_max - m_points->value_min );
	}
}

void
WidgetChart::set_points( ChartPoints *points )
{
	if( m_points )
		delete m_points;
	m_points = points;
	update_col_geom( true );

	if( Glib::RefPtr< Gdk::Window > window = get_window() )
		window->invalidate( false );
}

bool
WidgetChart::on_scroll_event( GdkEventScroll *event )
{
	if( m_points )
	{
		if( event->direction == GDK_SCROLL_UP && m_col_start > 0 )
			m_col_start--;
		else
		if( event->direction == GDK_SCROLL_DOWN &&
			m_col_start < ( m_points->size() - m_col_count ) )
			m_col_start++;
		else
			return true;

		get_window()->invalidate( false );
	}
	return true;
}

bool
WidgetChart::on_button_press_event( GdkEventButton *event )
{
	if( m_points && event->button == 1 && event->y > m_height - m_overview_height )
	{
		int col_start = ( event->x / m_width ) * m_points->size() - m_col_count / 2;
		if( col_start > int( m_points->size() - m_col_count ) )
			col_start = m_points->size() - m_col_count;
		else
		if( col_start < 0 )
			col_start = 0;
		m_col_start = col_start;

		get_window()->invalidate( false );

		m_flag_button_pressed = true;
	}

	return true;
}

bool
WidgetChart::on_button_release_event( GdkEventButton *event )
{
	if( event->button == 1 )
	{
		m_flag_button_pressed = false;
	}

	return true;
}

bool
WidgetChart::on_motion_notify_event( GdkEventMotion *event )
{
	if( m_points )
	{
		if( m_flag_button_pressed )
		{
			int col_start = ( event->x / m_width ) * m_points->size() - m_col_count / 2;
			if( col_start > int( m_points->size() - m_col_count ) )
				col_start = m_points->size() - m_col_count;
			else
			if( col_start < 0 )
				col_start = 0;

			if( col_start != ( int ) m_col_start )
			{
				m_col_start = col_start;
				get_window()->invalidate( false );
			}
		}
		bool flag_pointer_hovered = ( event->y > m_height - m_overview_height );
		if( flag_pointer_hovered != m_flag_pointer_hovered )
		{
			m_flag_pointer_hovered = flag_pointer_hovered;
			get_window()->invalidate( false );	// TODO: limit to scrollbar only
		}
	}

	return Gtk::DrawingArea::on_motion_notify_event( event );
}

bool
WidgetChart::on_leave_notify_event( GdkEventCrossing *event )
{
	if( m_points )
	{
		m_flag_pointer_hovered = false;
		get_window()->invalidate( false );	// TODO: limit to scrollbar only
	}

	return true;
}

void
WidgetChart::on_size_allocate( Gtk::Allocation &allocation )
{
	Gtk::Widget::on_size_allocate( allocation );
	m_width = allocation.get_width();
	m_height = allocation.get_height();

	x_max = m_width - border_curve;
	length = x_max - x_min;
	column_count_max = ( length / COLUMN_WIDTH_MIN ) + 1;

	update_col_geom();
}

bool
WidgetChart::on_expose_event( GdkEventExpose *event )
{
	Glib::RefPtr< Gdk::Window > window = get_window();
	if( window )
	{
		Cairo::RefPtr< Cairo::Context > cr = window->create_cairo_context();

		if( event )
		{
			// clip to the area indicated by the expose event so that we only
			// redraw the portion of the window that needs to be redrawn
			cr->rectangle( event->area.x, event->area.y,
					event->area.width, event->area.height );
			cr->clip();
		}

		// FONT FACE
		cr->set_font_face( m_font_main );

		// HANDLE THERE-IS-TOO-FEW-ENTRIES-CASE SPECIALLY
		if( m_points->size() < 2 )
		{
			// FIXME: do this more elegantly
			cr->rectangle( 0.0, 0.0, m_width, m_height );
			cr->set_source_rgb( 0.8, 0.8, 0.8 );
			cr->fill();
			cr->set_font_size( 1.5 * label_height );
			cr->set_source_rgb( 0.0, 0.0, 0.0 );
			Cairo::TextExtents te;
			cr->get_text_extents( "NOT ENOUGH DATA", te );
			cr->move_to( ( m_width - te.width ) / 2 , m_height / 2 );
			cr->show_text( "NOT ENOUGH DATA" );
			return true;
		}

		// BACKGROUND
		cr->rectangle( 0.0, 0.0, m_width, y_max );
		cr->set_source_rgba( 1.0, 1.0, 1.0, 0.6 );
		cr->fill();

		// END REGIONS
		if( m_col_start == 0 )
		{
			cr->set_source_rgb( 0.8, 0.8, 0.8 );
			cr->rectangle( 0.0, 0.0, x_min, y_max );
			cr->fill();
		}
		if( m_col_start == m_points->size() - m_col_count )
		{
			cr->set_source_rgb( 0.8, 0.8, 0.8 );
			cr->rectangle( m_width - border_curve, 0.0, border_curve, y_max );
			cr->fill();
		}

		// MONTHS BAR
		cr->rectangle( 0.0, y_max, m_width, bar_height );
		cr->set_source_rgb( 0.85, 0.85, 0.85 );
		cr->fill();
		// YEARS BAR
		cr->rectangle( 0.0, y_max + bar_height, m_width, bar_height );
		cr->set_source_rgb( 0.75, 0.75, 0.75 );
		cr->fill();

		// YEAR & MONTH LABELS
		cr->set_source_rgb( 0.0, 0.0, 0.0 );
		cr->set_font_size( label_height );
		unsigned int year_last = 0;
		for( unsigned int i = 0; i < m_col_count; ++i )
		{
			Date date = m_points->at( m_col_start + i ).date;  
			cr->move_to( x_min + step_x * i, y_max + label_y );
			cr->show_text( Glib::ustring::compose( "%1", date.get_month() ) );

			if( date.get_year() != year_last )
			{
				year_last = m_points->at( m_col_start + i ).date.get_year();
				if( i == 0 )
					if( m_points->at( m_col_start ).date.get_month() + m_col_count > 16 )
						continue;
				cr->move_to( x_min + step_x * i, y_max + bar_height + label_y );
				cr->show_text( Glib::ustring::compose( "%1", year_last ) );
			}
		}

		// y LABELS
		cr->move_to( border_label, y_min - offset_label );
		cr->show_text( Glib::ustring::compose( "%1", m_points->value_max ) );
		cr->move_to( border_label, y_max - offset_label );
		cr->show_text( Glib::ustring::compose( "%1", m_points->value_min ) );

		// HORIZONTAL LINES
		cr->set_source_rgb( 0.5, 0.5, 0.5 );
		cr->set_line_width( 1.0 );
		cr->move_to( border_label, y_min );
		cr->line_to( m_width - border_label, y_min );
		cr->move_to( border_label, y_mid );
		cr->line_to( m_width - border_label, y_mid );
		cr->stroke();

		// CURRENT LINE
		if( m_points->flag_use_current &&
			m_points->current >= m_col_start &&
			m_points->current - m_col_start < m_col_count )
		{
			cr->set_line_width( 2.0 );
			cr->set_source_rgb( 0.6, 0.0, 0.5 );
			cr->move_to( x_min + step_x * ( m_points->current - m_col_start ), 0.0 );
			cr->line_to( x_min + step_x * ( m_points->current - m_col_start ), y_max );
			cr->stroke();
		}

		// GRAPH LINE
		cr->set_source_rgb( 1.0, 0.0, 0.0 );
		cr->set_line_join( Cairo::LINE_JOIN_BEVEL );
		cr->set_line_width( 3.0 );

		cr->move_to( x_min,
				y_max - coefficient *
						( m_points->at( m_col_start ).value - m_points->value_min ) );
		for( unsigned int i = 0; i < m_col_count; ++i )
		{
			cr->line_to( x_min + step_x * i,
					y_max - coefficient *
							( m_points->at( m_col_start + i ).value - m_points->value_min ) );
		}
		cr->stroke();

		// GRAPH DOTS
		for( unsigned int i = 0; i < m_col_count; ++i )
		{
			cr->arc( x_min + step_x * i,
					 y_max - coefficient *
							( m_points->at( m_col_start + i ).value - m_points->value_min ),
					 5.0, 0.0, 2.0*M_PI);
			cr->fill();
		}

		if( m_col_count < m_points->size() )
		{
			// OVERVIEW PARAMETERS
			float ampli_ov = m_overview_height - 2 * offset_label;
			float coeff_ov = m_points->value_max == m_points->value_min ? 0.5 :
					ampli_ov / ( m_points->value_max - m_points->value_min );
			float step_x_ov = m_width - 2 * offset_label;
			if( m_points->size() > 1 )
				step_x_ov /= m_points->size() - 1;

			// OVERVIEW REGION
			cr->save();
			cr->set_source_rgb( 0.7, 0.7, 0.7 );
			cr->rectangle( 0.0, m_height - m_overview_height, m_width, m_overview_height );
			cr->fill();

			if( m_flag_pointer_hovered )
				cr->set_source_rgb( 1.0, 1.0, 1.0 );
			else
				cr->set_source_rgb( 0.95, 0.95, 0.95 );
			cr->rectangle( offset_label + m_col_start * step_x_ov, m_height - m_overview_height,
						   ( m_col_count - 1 ) * step_x_ov, m_overview_height );
			cr->fill();
			cr->restore();

			// CURRENT LINE
			if( m_points->flag_use_current )
		{
			cr->save();
			cr->set_line_width( 1.0 );
			cr->set_source_rgb( 0.6, 0.0, 0.5 );
			cr->move_to( offset_label + step_x_ov * m_points->current,
						 m_height - m_overview_height );
			cr->line_to( offset_label + step_x_ov * m_points->current, m_height );
			cr->stroke();
			cr->restore();
		}

			// OVERVIEW LINE
			cr->set_line_width( 2.0 );
			cr->move_to( offset_label,
					m_height - offset_label - coeff_ov *
							( m_points->at( 0 ).value - m_points->value_min ) );
			for( unsigned int i = 1; i < m_points->size(); ++i )
			{
				cr->line_to( offset_label + step_x_ov * i,
						m_height - offset_label - coeff_ov *
								( m_points->at( i ).value - m_points->value_min ) );
			}
			cr->stroke();

			// DIVIDER
			if( m_flag_pointer_hovered )
				cr->set_source_rgb( 0.2, 0.2, 0.2 );
			else
				cr->set_source_rgb( 0.45, 0.45, 0.45 );
			cr->rectangle( 1.0, m_height - m_overview_height,
						   m_width - 2.0, m_overview_height - 1.0 );
			cr->stroke();
		}
	}

	return true;
}
