/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_DIARYDATA_HEADER
#define LIFEOGRAPH_DIARYDATA_HEADER


#include "helpers.hpp"	// i18n headers

#include <gtkmm/treemodel.h>
#include <set>


namespace LIFEO
{

typedef char SortingCriteria;
static const SortingCriteria SC_DATE	= 'd';
static const SortingCriteria SC_SIZE	= 's';
//static const SortingCriteria SC_CHANGE_DATE	= 'c';


class ListData;	// forward declaration

// DIARYELEMENT
class DiaryElement
{
	public:
		static bool 			FLAG_ALLOCATE_GUI_FOR_DIARY;

		enum Type
		{
			IT_NONE, IT_UNTAGGED, IT_TAG, IT_TAG_CATEGORY, IT_THEME,
			// entry list elements:
			IT_DIARY, IT_EARLIER, IT_CHAPTER, IT_ENTRY, IT_DATE
		};

								DiaryElement( void );
		virtual					~DiaryElement() {}

		virtual Date			get_date( void ) const
		{ return Date( 0 ); }
		virtual int				get_size( void ) const = 0;
		virtual Type			get_type( void ) const = 0;
		virtual Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const = 0;

		virtual	Glib::ustring	get_1st_str( void ) const
		{ return ""; }
		virtual	Glib::ustring	get_2nd_str( void ) const
		{ return ""; }

		virtual void			show( void ) = 0;
		virtual void			prepare_for_hiding( void ) {}
		virtual bool			get_filtered_out( void )
		{ return false; }

		ListData				*m_list_data;
};

class DiaryElementNamed : public DiaryElement
{
	public:
								DiaryElementNamed( void )
								:	m_name( "" ) {}
								DiaryElementNamed( const Glib::ustring &name )
								:	m_name( name ) {}

		Glib::ustring			get_name( void ) const
		{ return m_name; }
		void					set_name( const Glib::ustring &name )
		{ m_name = name; }

	protected:
		Glib::ustring			m_name;
};

class ListData
{
	public:
		class Colrec : public Gtk::TreeModel::ColumnRecord
		{
			public:
				Colrec()
				{
					add( ptr );
					add( id );
					add( info );
					add( icon );
				}
				// HIDDEN COLUMNS
				Gtk::TreeModelColumn< DiaryElement* >					ptr;
				// COLUMNS
				Gtk::TreeModelColumn< Glib::ustring >					id;
				Gtk::TreeModelColumn< Glib::ustring >					info;
				Gtk::TreeModelColumn< Glib::RefPtr< Gdk::Pixbuf > >		icon;
		};
		static Colrec			*colrec;

		Gtk::TreePath			treepath;
};

bool compare_listitems( DiaryElement*, DiaryElement* );
bool compare_listitems_by_name( DiaryElementNamed*, DiaryElementNamed* );
bool compare_names( const Glib::ustring&, const Glib::ustring& );

typedef bool( *FuncCompareStrings )( const Glib::ustring&, const Glib::ustring& ) ;
typedef bool( *FuncCompareDiaryElem )( DiaryElement*, DiaryElement* ) ;
typedef bool( *FuncCompareDiaryElemNamed )( DiaryElementNamed*, DiaryElementNamed* ) ;
typedef std::set< DiaryElement*, FuncCompareDiaryElem > SetDiaryElements;

template< class T >
Glib::ustring
create_unique_name_for_map( const std::map< Glib::ustring, T*, FuncCompareStrings > &map,
						 const Glib::ustring &name0 )
{
	Glib::ustring name = name0;
	for( int i = 1;
		 map.find( name ) != map.end();
		 i++ )
	{
		name = Glib::ustring::compose( "%1 %2", name0, i );
	}

	return name;
}

class DiaryElementContainer : public DiaryElementNamed
{
// NOTE: this class and its derivatives are not responsible for handling of...
// ...its children's allocation and deletion
	public:
								DiaryElementContainer( void )
								:	m_items( compare_listitems ) {}
								DiaryElementContainer( const Glib::ustring &name )
								:	DiaryElementNamed( name ), m_items( compare_listitems ) {}

		SetDiaryElements*		get_items( void )
		{ return &m_items; }
		virtual int				get_size( void ) const
		{ return m_items.size(); }

		void					clear( void )
		{ m_items.clear(); }

		DiaryElement*			add_to_list( Glib::RefPtr< Gtk::TreeStore >&,
											 const Gtk::TreeRow& );

	protected:
		SetDiaryElements		m_items;
};


// TAG =========================================================================
class Tag;	// forward declaration

// TAG CATEGORY
// TODO: should be a DiaryElementContainer as it is not responsible for memory handling
class CategoryTags : public DiaryElementNamed, public std::set< Tag*, FuncCompareDiaryElemNamed >
{
	public:
		struct Shower { virtual void show( CategoryTags& ) = 0; };
		static Shower			*shower;

								CategoryTags( const Glib::ustring& );

		void					show( void );

		virtual int				get_size( void ) const
		{ return size(); }
		Type					get_type( void ) const
		{ return IT_TAG_CATEGORY; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;

		Glib::ustring			get_2nd_str( void ) const
		{ return m_name; }

		bool					get_expanded( void ) const
		{ return m_flag_expanded; }
		void					set_expanded( bool flag_expanded )
		{ m_flag_expanded = flag_expanded; }

	protected:
		bool					m_flag_expanded;

	friend class CategoryTagsView;
	friend class PoolCategoriesTags;
	friend class TagPanel;
};

// POOL OF DEFINED TAG CATEGORIES
class PoolCategoriesTags : public std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >
{
	public:
								PoolCategoriesTags( void );
								~PoolCategoriesTags();

		bool					rename_category( CategoryTags*, const Glib::ustring& );
		void					clear( void );
};

// TAG
class Tag : public DiaryElementContainer
{
	public:
		struct Shower { virtual void show( Tag& ) = 0; };
		static Shower			*shower;

								Tag( void ) : m_ptr2category( NULL ) {}
		explicit				Tag( const Glib::ustring&, CategoryTags* );
		virtual					~Tag( void );

		void					show( void );

		CategoryTags*			get_category( void )
		{ return m_ptr2category; }
		void					set_category( CategoryTags *category );

		void					set_name( const Glib::ustring& );
		//Date					get_date( void ) const;	// if earliest entry's date is needed
		Type					get_type( void ) const
		{ return IT_TAG; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;

		Glib::ustring			get_2nd_str( void ) const
		{ return m_name; }

	protected:
		CategoryTags			*m_ptr2category;

	friend class PoolTags;
	friend class Tagset;
	friend class TagView;
};

// POOL OF DEFINED TAGS
class PoolTags : public std::map< Glib::ustring, Tag*, FuncCompareStrings >
{
	public:
								PoolTags( void )
		:	std::map< Glib::ustring, Tag*, FuncCompareStrings >( compare_names ) {}
								~PoolTags();

		bool					handle_tag_changed( );
		Tag*					create( const Glib::ustring&,
										CategoryTags* = NULL );
		bool					rename( Tag*, const Glib::ustring& );
		Tag*					get_tag( unsigned int );
		Tag*					get_tag( const Glib::ustring& );
		void					clear( void );
};

// TAGS OF AN ENTRY
// not responsible for memory handling
class Tagset : public std::set< Tag*, FuncCompareDiaryElemNamed >
{
	public:
								Tagset( void )
		:	std::set< Tag*, FuncCompareDiaryElemNamed >( compare_listitems_by_name ) {}

								~Tagset( void );
		bool					add( Tag* );
		bool					checkfor_member( const Tag* ) const;
		const Tag*				get_tag( unsigned int ) const;
		//bool					remove_tag( const Glib::ustring& );

	protected:

};

// CHAPTER =====================================================================
class Chapter : public DiaryElementContainer
{
	public:
		struct Shower { virtual void show( Chapter& ) = 0; };
		static Shower			*shower;

		explicit				Chapter( const Glib::ustring& );
								Chapter( const Glib::ustring&, Date );
		// this class should have a virtual dtor for some reason because
		// if it does not, its derived classes cannot be deleted
		virtual					~Chapter() {}

		void					show( void );

		void					set_name( const Glib::ustring& );
		Date					get_date( void ) const;
		Glib::ustring			get_date_str( void ) const;
		void					set_date( Date );
		Type					get_type( void ) const
		{ return IT_CHAPTER; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;

		Glib::ustring			get_1st_str( void ) const
		{ return Glib::ustring::compose( "<b>%1</b>", Glib::Markup::escape_text(
			m_date_begin.format_string() ) ); }
		Glib::ustring			get_2nd_str( void ) const
		//{ return Glib::ustring::compose( "<b>%1</b>", Glib::Markup::escape_text( m_name ) ); }
		{ return m_name; }

		bool					get_expanded( void ) const
		{ return m_flag_expanded; }
		void					set_expanded( bool flag_expanded )
		{ m_flag_expanded = flag_expanded; }

		bool					is_initialized( void )
		{ return m_flag_initialized; }

	protected:
		Date					m_date_begin;
		bool					m_flag_initialized;
		bool					m_flag_expanded;

	friend class CategoryChapters;
	friend class ChapterView;
};

class CategoryChapters :
		public DiaryElementNamed, public std::set< Chapter*, FuncCompareDiaryElem >
{
	public:
		explicit				CategoryChapters( const Glib::ustring& );
		virtual					~CategoryChapters();

		virtual int				get_size( void ) const
		{ return size(); }
		Type					get_type( void ) const
		{ return IT_TAG_CATEGORY; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;
		void					show( void ) {}	// not used

		Chapter*				add_chapter( const Glib::ustring& );
		void					remove_chapter( Chapter* );
		bool					rename_chapter( Chapter*, const Glib::ustring& );
		bool					set_chapter_date( Chapter*, const Glib::ustring& );
		bool					set_chapter_date( Chapter*, const Date& );

		void					clear( void );

	protected:

};

class PoolCategoriesChapters :
		public std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >
{
	public:
									PoolCategoriesChapters( void )
		:	std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >(
				compare_names ) {}
									~PoolCategoriesChapters();
		void						clear( void );
};

// THEMES
class Theme : public DiaryElementNamed
{
	public:
		// Shower sub-class is for communication between Theme and...
		// ...ThemeView classes. we may resort to the sigc::signals if needed:
		struct Shower { virtual void show( Theme& ) = 0; };

									Theme( const Glib::ustring& );
									Theme( const Glib::ustring&,
										   const Glib::ustring&,
										   const std::string&,
										   const std::string&,
										   const std::string&,
										   const std::string&,
										   const std::string& );

		void						show( void );

		int							get_size( void ) const
		{ return 0; }	// redundant
		DiaryElement::Type			get_type( void ) const
		{ return IT_THEME; }
		Glib::RefPtr< Gdk::Pixbuf >&
									get_icon( void ) const;

		Glib::ustring				get_1st_str( void ) const
		{ return Glib::Markup::escape_text( m_name ); }

		Pango::FontDescription		font;
		Gdk::Color					color_base;
		Gdk::Color					color_text;
		Gdk::Color					color_heading;
		Gdk::Color					color_subheading;
		Gdk::Color					color_highlight;

		static Theme				system_theme;

		static Shower				*shower;

	friend class Diary;
};

class PoolThemes : public std::map< Glib::ustring, Theme*, FuncCompareStrings >
{
	public:
									PoolThemes( void )
		:	std::map< Glib::ustring, Theme*, FuncCompareStrings >( compare_names ) {}
									~PoolThemes();

		void						clear( void );
		bool						rename_theme( Theme*, const Glib::ustring& );
};

} // end of namespace LIFEO

#endif

