/*
 *  Copyright 1994-2012 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef WITH_GL
#include <GL/gl.h>
#include <GL/glu.h>
#endif
#include "biniou.h"
#include "schemes.h"
#include "pictures.h"
#include "colormaps.h"
#include "brandom.h"

#define BARSIZE (HEIGHT/20) /* Colormap bar size */

static void
Context_boundary(Context_t *ctx)
{
  switch (ctx->params3d.draw_boundary) {
  case 1:
    draw_cube_3d(&ctx->params3d, active_buffer(ctx), 250);
    break;

  case 2:
    draw_sphere_3d(&ctx->params3d, active_buffer(ctx), 250);
    break;

  case 3:
    draw_sphere_wireframe_3d(&ctx->params3d, active_buffer(ctx), 250);
    break;

  default:
    break;
  }
}


static void
Context_sync(Context_t *ctx)
{
  int i;
  const float rrt = b_timer_elapsed(ctx->fps_timer);
  const float sleep = ctx->i_max_fps - rrt;
  
  for (i = 0; i < NFPS-1; i++)
    ctx->fps[i] = ctx->fps[i+1];
  ctx->fps[i] = (rrt > 0) ? (1.0/rrt) : 0.0;
  
  if (sleep > 0)
    ms_sleep(sleep*MFACTOR);
}


static void
Context_sequence(Context_t *ctx)
{
  const GList *tmp;
  uint16_t count = 0;

  tmp = ctx->sm->cur->layers;
  b_timer_start(ctx->fps_timer);

#ifdef WITH_GL
  ctx->texture_ready = ctx->texture_used = ctx->gl_done = 0;
#endif

  while (tmp != NULL) {
    Layer_t *layer = (Layer_t *)tmp->data;
    Plugin_t *P = layer->plugin;
    
    assert(P != NULL);
    if (P == ctx->sm->cur->lens)
      push_buffer(ctx);

    if (P->run != NULL)
      if (!ctx->bypass || (ctx->bypass && !(*P->options & BEQ_BYPASS))) {
	P->run(ctx);
	P->calls++;
	count++;
      }

    switch (layer->mode) {
    case NONE:
      break;

    case NORMAL:
      swap_buffers(ctx);
      break;

    case OVERLAY:
      Buffer8_overlay(active_buffer(ctx), passive_buffer(ctx));
      break;

    case XOR:
      Buffer8_XOR(active_buffer(ctx), passive_buffer(ctx));
      break;

    case AVERAGE:
      Buffer8_average(active_buffer(ctx), passive_buffer(ctx));
      break;

    case RANDOM:
      Buffer8_mix_random(active_buffer(ctx), passive_buffer(ctx));
      break;

    default:
      xerror("Unsupported layer mode %d\n", layer->mode);
      break;
    }

    tmp = g_list_next(tmp);
  }
#ifdef FEAT_TARGET
  if (!count && ctx->target) {
    assert(NULL != ctx->target_pic);
    Buffer8_copy(ctx->target_pic->buff, active_buffer(ctx));
  }
#endif
}


void
Context_run(Context_t *ctx)
{
  Plugin_t *input = ctx->input_plugin;
  GSList *outputs = ctx->outputs;

  /* Auto-change sequence */
  if (Alarm_ring(ctx->a_random)) {
    if (ctx->random_mode == BR_SCHEMES) {
      if ((NULL != schemes) && (schemes->size > 1))
	Schemes_random(ctx);
    } else if (ctx->random_mode == BR_SEQUENCES) {
      if ((NULL != sequences) && (sequences->size > 1))
	Context_random_sequence(ctx);
    } else if (ctx->random_mode == BR_BOTH) {
      if (b_rand_boolean()) {
	if ((NULL != schemes) && (schemes->size > 1)) {
	  printf("[+] Random scheme\n");
	  Schemes_random(ctx);
	}
      } else {
	if ((NULL != sequences) && (sequences->size > 1)) {
	  printf("[+] Random sequence\n");
	  Context_random_sequence(ctx);
	}
      }
    }
  }

  /* Auto-change picture */
  /* Only if there are more than one picture loaded */
  if ((pictures != NULL) && (pictures->size > 1)) {
    /* Timer elapsed ? */
    if (ctx->pf->on && Alarm_ring(ctx->a_picts))
      PictFader_random(ctx->pf);
    /* Fade picture */
    if (PictFader_ring(ctx->pf)) {
      PictFader_run(ctx->pf);
      ctx->sm->cur->picture_id = ctx->pf->dst->id;
    }
  }
  
  /* Auto-change colormap */
  assert(colormaps != NULL);
  /* Only if there are more than one colormap loaded */
  if (colormaps->size > 1) {
    /* Timer elapsed ? */
    if (ctx->cf->on && Alarm_ring(ctx->a_cmaps))
      CmapFader_random(ctx->cf);
    /* Fade colormap */
    if (CmapFader_ring(ctx->cf)) {
      CmapFader_run(ctx->cf);
      ctx->sm->cur->cmap_id = ctx->cf->dst->id;
    }
  }

  /* If we have an input plugin that runs in non-threaded mode,
   * call it now */
  if ((NULL != input) && (NULL != input->run)) {
    input->run(ctx);
    input->calls++;
  }
  
  /* 3D stuff */
  if (ctx->params3d.do_auto_rotate)
    Params3d_change_rotations(&ctx->params3d);
#ifdef Z_BUFFER
  Params3d_init_z_buffer();
#endif

#ifdef WITH_GL
  glClearColor(0, 0, 0, 0);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

  /* Initialize projection matrix */
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  gluPerspective(ctx->params3d.gl_fov, (float)WIDTH/(float)HEIGHT,
		 0.1, ZMAX);
  gluLookAt(0, 0, 3.14,
	    0, 0, -3.14,
	    0, 1, 0);

  glShadeModel(GL_SMOOTH);

  glClearDepth(ZMAX),
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);

  glEnable(GL_BLEND);

  glRotatef(ctx->params3d.gl_rotations[0], 1.0, 0.0, 0.0);
  glRotatef(ctx->params3d.gl_rotations[1], 0.0, 1.0, 0.0);
  glRotatef(ctx->params3d.gl_rotations[2], 0.0, 0.0, 1.0);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
#endif

  /* The sequence */
  Context_sequence(ctx);

  /* sync fps */
  /* FIXME peut-etre le bouger en fin de boucle ? --oliv3 */
  if (ctx->sync_fps)
    Context_sync(ctx);

  /* draw 3D boundary */
  Context_boundary(ctx);
  
  /* show the current colormap */
  if (ctx->display_colormap) {
    if (ctx->sm->cur->lens == NULL)
	push_buffer(ctx);
    Buffer8_color_bar(active_buffer(ctx), BARSIZE);
  }

  /* TODO: use plugin on_key callback */
  for ( ; outputs != NULL; outputs = g_slist_next(outputs)) {
    Plugin_t *output = (Plugin_t *)outputs->data;
    
    if ((ctx->cf != NULL) && ctx->cf->refresh && (output->set_cmap != NULL))
      output->set_cmap(ctx);

    /* TODO print a warning: an output plugin without run() is a bug */
    if (output->run != NULL) {
      output->run(ctx);
      output->calls++;
    }
  }
  /* All outputs should have refreshed their cmap by now */
  ctx->cf->refresh = 0;

  /* Process events */
  Context_process_events(ctx);

  /* Screenshot */
  if (ctx->take_screenshot) {
    Context_screenshot(ctx);
    ctx->take_screenshot = 0;
  }

  /* Restore possibly saved buffer */
  if ((ctx->sm->cur->lens != NULL) || ctx->display_colormap)
    pop_buffer(ctx);
  
  ++ctx->frames;
}
