/*
 *  Copyright 1994-2011 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "globals.h"
#include "context.h"
#include "colormaps.h"
#include "pictures.h"


void
Context_add_event(Context_t *ctx, const Event_t *e)
{
  ctx->events = g_list_append(ctx->events, (gpointer)e);
}


static void
Context_process_event(Context_t *ctx, const Event_t *e)
{
  /* find out the recipient */
  switch (e->to) {
  case BT_NONE:
    break;
    
  case BT_PICTFADER:
    if (PictFader_event(ctx->pf, e))
	Sequence_changed(ctx->sm->cur);
    break;
    
  case BT_CMAPFADER:
    if (CmapFader_event(ctx->cf, e))
	Sequence_changed(ctx->sm->cur);
    break;
    
  case BT_CONTEXT:
    (void)Context_event(ctx, e);
    break;
    
  case BT_SEQMGR:
    (void)SequenceManager_event(ctx->sm, e);
    break;
    
  case BT_PLUGINS:
    (void)Plugins_event(plugins, e);
    break;
    
  default:
    break;
  }

  ctx->nb_events++;
}


void
Context_process_events(Context_t *ctx)
{
  GList *t;

  for (t = ctx->events; t != NULL; t = g_list_next(t)) {
    Event_t *e = (Event_t *)t->data;

    Context_process_event(ctx, e);
    xfree(e);
  }
  g_list_free(ctx->events);
  ctx->events = NULL;
}


static void
Context_display_random(const Context_t *ctx)
{
  printf("[A] Auto random is ");
  if (ctx->random_mode == BR_NONE) printf("off\n");
  else if (ctx->random_mode == BR_SEQUENCES) printf("auto sequences\n");
  else if (ctx->random_mode == BR_SCHEMES) printf("auto schemes\n");
  else if (ctx->random_mode == BR_BOTH) printf("auto schemes/sequences\n");
}


int
Context_event(Context_t *ctx, const Event_t *e)
{
  int i;
  GSList *outputs = ctx->outputs;

  switch (e->cmd) {
  case BC_QUIT:
    if (e->arg0 == BA_SAVE)
      Sequence_save(ctx->sm->cur, 0);
    ctx->running = 0;
    return 1;
    break;

  case BC_SAVE:
    if (e->arg0 == BA_SCREENSHOT) {
      ctx->take_screenshot = 1;
      return 1;
    } else
      return 0;
    break;

  case BC_SWITCH:
    switch (e->arg0) {
    case BA_SELECTED:
      if (!(*plugins->selected->options & BEQ_DISABLED)) {
	if (Sequence_find(ctx->sm->cur, plugins->selected) != NULL)
	  Context_remove_plugin(ctx, plugins->selected);
	else {
	  if (g_list_length(ctx->sm->cur->layers) < MAX_SEQ_LEN)
	    Context_insert_plugin(ctx, plugins->selected);
	}
      }
      return 1;
      break;

      /* TODO: use plugin on_key callback */
    case BA_FULLSCREEN:
      ctx->fullscreen = !ctx->fullscreen;
      printf("[S] Full-screen %s\n", ctx->fullscreen ? "off" : "on");
      
      for ( ; outputs != NULL; outputs = g_slist_next(outputs)) {
	Plugin_t *output = (Plugin_t *)outputs->data;

	if (output->fullscreen != NULL)
	  output->fullscreen(ctx->fullscreen);
      }
      return 1;
      break;
      
      /* TODO: use plugin on_key callback */
    case BA_CURSOR:
      for ( ; outputs != NULL; outputs = g_slist_next(outputs)) {
	Plugin_t *output = (Plugin_t *)outputs->data;

	if (output->switch_cursor != NULL)
	  output->switch_cursor();
      }
      return 1;
      break;

    case BA_COLORMAPS:
      if (colormaps->size > 1) {
	ctx->sm->cur->auto_colormaps = 1-ctx->sm->cur->auto_colormaps;
        Sequence_changed(ctx->sm->cur);
        Context_update_auto(ctx);
        return 1;
      } else
	return 0;
      break;

    case BA_PICTURES:
      if ((pictures != NULL) && (pictures->size > 1)) {
        ctx->sm->cur->auto_pictures = 1-ctx->sm->cur->auto_pictures;
        Sequence_changed(ctx->sm->cur);
        Context_update_auto(ctx);
        return 1;
      } else
	return 0;
      break;
      
    case BA_OSD_CMAP:
      ctx->display_colormap = 1 - ctx->display_colormap;
      return 1;
      break;

    case BA_ROTATIONS:
      if (ctx->params3d.do_auto_rotate) {
	for (i = 0; i < 3; i++) {
	  ctx->params3d.auto_rotate[i] = 0;
	  ctx->params3d.rotations[i] = 0;
	  update(&ctx->params3d);
	}
	ctx->params3d.do_auto_rotate = 0;
      } else {
	/* TODO add more events to the 3d rotations stuff
	 * for now we force rotation around the Y axis */
	/* Params3d_randomize(&ctx->params3d); */
	ctx->params3d.auto_rotate[0] =
	  ctx->params3d.auto_rotate[2] = 0;
	ctx->params3d.auto_rotate[1] = 1;

	ctx->params3d.do_auto_rotate = 1;
      }
      return 1;
      break;

    case BA_BYPASS:
      ctx->bypass = !ctx->bypass;
      return 1;
      break;

    default:
      return 0;
      break;
    }
    
  case BC_PREV:
    switch (e->arg0) {
    case BA_SEQUENCE:
      Context_previous_sequence(ctx);
      return 1;

    case BA_RANDOM:
      if (ctx->random_mode == BR_NONE)
	ctx->random_mode = BR_BOTH;
      else
	--ctx->random_mode;
      
      if (ctx->random_mode == BR_SCHEMES) {
	if ((schemes == NULL) || (!Shuffler_ok(schemes->shuffler))) {
	  printf("[i] Skipping random schemes since there are no schemes available\n");
	  ctx->random_mode = BR_SEQUENCES;
	} else {
	  Schemes_random(ctx);
	  Alarm_init(ctx->a_random);
	}
      }

      if (ctx->random_mode == BR_SEQUENCES) {
	if ((sequences->seqs == NULL) || !g_list_length(sequences->seqs)) {
	  printf("[i] Skipping random sequences since there are no sequences available\n");
	  ctx->random_mode = BR_NONE;
	} else {
	  Context_random_sequence(ctx);
	  Alarm_init(ctx->a_random);
	}
      }

      Context_display_random(ctx);
      return 1;

    case BA_FPS:
      if (ctx->sync_fps && (ctx->max_fps > 5)) {
	Context_set_max_fps(ctx, ctx->max_fps-5);
	return 1;
      }
      
    default:
      return 0;
    }
    break;

  case BC_NEXT:
    switch (e->arg0) {
    case BA_OSD:
      ctx->osd_mode = (enum OSDMode)((ctx->osd_mode+1) % 4);
      return 1;

    case BA_BOUNDARY:
      ctx->params3d.draw_boundary = (ctx->params3d.draw_boundary+1) % 4;
      return 1;

    case BA_SEQUENCE:
      Context_next_sequence(ctx);
      return 1;

    case BA_RANDOM:
      ctx->random_mode = (enum RandomMode)((ctx->random_mode+1) % sizeof(RandomMode_e));
      
      if (ctx->random_mode == BR_SEQUENCES) {
	if ((sequences->seqs == NULL) || !g_list_length(sequences->seqs)) {
	  printf("[i] Skipping random sequences since there are no sequences available\n");
	  ctx->random_mode = BR_SCHEMES;
	} else {
	  Context_random_sequence(ctx);
	  Alarm_init(ctx->a_random);
	}
      }
      
      if (ctx->random_mode == BR_SCHEMES) {
	if ((schemes == NULL) || (!Shuffler_ok(schemes->shuffler))) {
	  printf("[i] Skipping random schemes since there are no schemes available\n");
	  ctx->random_mode = BR_BOTH;
	} else {
	  Schemes_random(ctx);
	  Alarm_init(ctx->a_random);
	}
      }

      Context_display_random(ctx);
      return 1;

    case BA_FPS:
      if (ctx->sync_fps) {
	Context_set_max_fps(ctx, ctx->max_fps+5);
	return 1;
      }

    default:
      return 0;
    }
    break;

  case BC_RANDOM:
    switch (e->arg0) {
    case BA_SEQUENCE:
      assert(sequences != NULL);

      if (sequences->size > 1) {
	Context_random_sequence(ctx);
	Alarm_init(ctx->a_random);
      }
      return 1;
      break;
      
    case BA_SCHEME:
      if ((schemes != NULL) && (schemes->size > 1)) {
	Schemes_random(ctx);
	Alarm_init(ctx->a_random);
      }
      return 1;
      break;

    default:
      return 0;
    }
    break;

  case BC_RESET:
    switch (e->arg0) {
    case BA_RANDOM:
      Buffer8_randomize(active_buffer(ctx));
      return 1;
      break;

    case BA_SEQUENCE:
      Buffer8_clear(active_buffer(ctx));
      return 1;
      break;
      
    default:
      /* Emergency stop, all auto_* modes -> disabled */
      ctx->sm->cur->auto_colormaps = 0;
      ctx->sm->cur->auto_pictures = 0;
      Context_update_auto(ctx);
      Sequence_changed(ctx->sm->cur);
      ctx->random_mode = BR_NONE;

      printf("[!] *** EMERGENCY STOP *** button pressed !!!\n");
      return 1;
      break;
    }

  case BC_USE_BANK:
    printf("[i] Using sequence in bank #%d\n", (e->arg0+1));
    Context_use_bank(ctx, e->arg0);
    return 1;
    break;

  case BC_CLEAR_BANK:
    printf("[i] Clear sequence in bank #%d\n", (e->arg0+1));
    Context_clear_bank(ctx, e->arg0);
    return 1;
    break;

  case BC_RELOAD:
    if (e->arg0 == BA_SEQUENCE) {
      Context_latest_sequence(ctx);
      return 1;
    }
    break;

  default:
    break;
  }

  return 0;
}


void
Context_send_event(Context_t *ctx, const enum RcptTo TO, const enum Command CMD,
	           const enum Arg ARG0)
{
  Event_t *e = xmalloc(sizeof(Event_t));
  e->to = TO;
  e->cmd = CMD;
  e->arg0 = ARG0;
  Context_add_event(ctx, e);
}
