/***************************************************************************
 *
 *  $Id: kzenplaylist.cpp,v 1.16 2005/07/03 18:27:52 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <qpainter.h>
#include <qpixmap.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <kapplication.h>
#include <klocale.h>

#include "kzenconfig.h"
#include "kzenexplorer.h"
#include "kzenplaylist.h"

int KZenPlayList::fNModified = 0;

KZenPlayList::KZenPlayList()
{
  fItem = 0;

  fTracks = new QPtrList<KZenTrack>(); 
  fTracks->setAutoDelete(false);

  fModified = false;
}

KZenPlayList::KZenPlayList(const char * name)
{
  fItem = 0;

  setName(name);

  fTracks = new QPtrList<KZenTrack>();
  fTracks->setAutoDelete(false);

  fModified = false;
}

KZenPlayList::KZenPlayList(njb_playlist_t * pl, QPtrList<KZenTrack> * tracks)
{
  fItem = 0;

  fTracks = new QPtrList<KZenTrack>();
  fTracks->setAutoDelete(false);

  fModified = false;

  njb_playlist_track_t * track;
  setName(pl->name);
  setID(pl->plid);

  KZenTrack * temp;
  NJB_Playlist_Reset_Gettrack(pl);
  while ((track = NJB_Playlist_Gettrack(pl)) ) {
    temp = 0;
    for (uint tr=0;tr<tracks->count();tr++) {
      temp = tracks->at(tr);
      if (temp->getTrackID()==track->trackid) break;
      temp = 0;
    }
    if (temp) {
      fTracks->append(temp);
    } else {
      setModified(true);
    }
  }

  NJB_Playlist_Destroy(pl);
}

KZenPlayList::~KZenPlayList()
{
  delete fTracks;
}

void KZenPlayList::setName(const char * name)
{
  fName = name;
  if (fItem) {
    fItem->setText(0,fName);
  }
}

void KZenPlayList::addTrack(KZenTrack * track)
{
  fTracks->append(track);    
  setModified(true);
}

void KZenPlayList::setModified(bool m)
{
  if (!fModified && m) {
    fNModified++;
  }
  if (fModified && !m) {
    fNModified--;
  }
  if (fNModified>0) {
    gExplorer->setUpdateAllEnabled(true);
  } else {
    gExplorer->setUpdateAllEnabled(false);
  }

  fModified = m;
  if (fItem && fModified) {
    if (fModified) {
      if (fItem->isSelected()) gExplorer->setUpdateEnabled(true);
      fItem->triggerUpdate();
    }
  }
}

void KZenPlayList::shuffle()
{
  QPtrList<KZenTrack> tempList;
  KZenTrack * track;
  
  for (uint tr=0;tr<fTracks->count();tr++) {
    track = fTracks->at(tr);
    tempList.append(track);
  }
  fTracks->clear();
  
  uint c = tempList.count();
  while (c>0) {
    track = tempList.at(KApplication::random() % c);
    fTracks->append(track);
    tempList.remove(track);
    c = tempList.count();
  }
  setModified(true);
}

uint KZenPlayList::getTrackCount()
{
  return fTracks->count();
}

uint KZenPlayList::getLength()
{
  int length = 0;
  KZenTrack * track;
  for (uint tr=0;tr<fTracks->count();tr++) {
    track = fTracks->at(tr);
    length += track->getLength();
  }

  return length;
}

KZenPlayListItem::KZenPlayListItem(QListView * parent, KZenPlayList * PlayList)
  :KZenItem(parent,KZenItem::ItemTypePlayList)
{
  fPlayList = PlayList;
  fPlayList->setItem(this);

  setPixmap(0, UserIcon("playlist.png", 16));
  setText(0,fPlayList->getName());
  
  fSortColumn = KZen::FieldUnknown;
  fSortOrder = Qt::Ascending;
}

KZenPlayListItem::~KZenPlayListItem()
{
  
}

void KZenPlayListItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
  bool sd = KZenConfig::get()->showDetailsPlayList();

  QFontMetrics fml(listView()->font());
  int margin = listView()->itemMargin()*2;
  int h = fml.lineSpacing();
  if (sd) h += fml.lineSpacing();
  if (pixmap(0)->height()>h) h = pixmap(0)->height();
  if (h % 2 > 0) h++;
  
  setHeight(h + margin + 2);

  KListView *lv = (KListView *)listView();

  static QPixmap buffer;
  buffer.resize(width, height());

  if(buffer.isNull()) {
    KListViewItem::paintCell(p, cg, column, width, align);
    return;
  }
  
  QPainter pBuf(&buffer, true);
  if (isSelected()) {
    pBuf.fillRect(buffer.rect(), cg.highlight());
  } else {
    pBuf.fillRect(buffer.rect(), KZen::backgroundColor(this));
  }

  QFont font(p->font());
  QFontMetrics fm(p->fontMetrics());

  int text_x = pixmap(0)->width() + 10;
  if (fPlayList->isModified()) text_x += 16 + 4;

  int text_y = 1;
  int textHeight = fm.lineSpacing() + lv->itemMargin() + 1;

  pBuf.drawPixmap(4, (height() - pixmap(0)->height())/2,
		  *pixmap(0));
  
  if (fPlayList->isModified()) {
    QPixmap update = UserIcon("update_small", 16);
    pBuf.drawPixmap(4 + 18, (height() - update.height())/2,
		    update);
  }
  
  if (isSelected()) {
    pBuf.setPen(cg.highlightedText());
  } else {
    pBuf.setPen(cg.text());
  }

  if (sd) font.setBold(true);
  pBuf.setFont(font);
  QFontMetrics fmName(font);

  QString name = text(0);

  if(fmName.width(name) + text_x + lv->itemMargin()*2 > width ) {
    int ellWidth = fmName.width(i18n("..."));
    QString text = QString::fromLatin1("");
    int i = 0;
    int len = name.length();
    while ( i < len && fmName.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
      text += name[i];
      i++;
    }
    name = text + i18n("...");
  }

  pBuf.drawText(text_x, text_y, width, textHeight, AlignVCenter, name);

  if (sd) {
    font.setBold(false);
    font.setItalic(true);
    pBuf.setFont(font);
    QFontMetrics fmInfo(font);

    QString info;
    info += i18n("1 Track", "%n Tracks", fPlayList->getTrackCount());
    info += QString(i18n(" - [%2]")).arg(KZen::formatTime(fPlayList->getLength()));
    
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }    
    pBuf.drawText(text_x, text_y + textHeight, width, fm.lineSpacing(), AlignVCenter, info);
  }
  
  pBuf.end();
  p->drawPixmap(0, 0, buffer);
}
