/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qcombobox.h>
#include <qlistbox.h>
#include <qlabel.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <kapplication.h>
#include <klocale.h>
#include "kxstitch.h"
#include "calibratedialog.h"
#include "schememanager.h"
#include "flossscheme.h"
#include "flosslistboxitem.h"

CalibrateDialog::CalibrateDialog(QWidget *parent)
  : CalibrateDlg(parent,"CalibrateDlg",true),
    m_item(0),
    m_sample(new QPixmap(100,100))
{
  fillSchemeList();
  fillColorList();

  connect(SchemeList,SIGNAL(activated(int)),this,SLOT(schemeChanged(int)));
  connect(ColorList,SIGNAL(selectionChanged(QListBoxItem *)),this,SLOT(selectionChanged(QListBoxItem *)));
  connect(RedSlider,SIGNAL(valueChanged(int)),this,SLOT(redChanged(int)));
  connect(GreenSlider,SIGNAL(valueChanged(int)),this,SLOT(greenChanged(int)));
  connect(BlueSlider,SIGNAL(valueChanged(int)),this,SLOT(blueChanged(int)));
  connect(ResetColor,SIGNAL(clicked()),this,SLOT(resetColor()));
  connect(HelpButton,SIGNAL(clicked()),this,SLOT(contextHelp()));
}

CalibrateDialog::~CalibrateDialog()
{
  delete m_sample;
}

void CalibrateDialog::accept()
{
  commitColor();

  QMap<QString,ChangedColors>::Iterator it;
  for (it = m_calibratedColors.begin() ; it != m_calibratedColors.end() ; ++it)
  {
    FlossScheme *scheme = ((KXStitchApplication *)kapp)->scheme(it.key());
    QPtrListIterator<Floss> itFloss = scheme->flossIterator();
    for (Floss *f = 0 ; (f = itFloss.current()) ; ++itFloss)
      if (it.data().contains(f->name))
        f->color = it.data()[f->name];
    ((KXStitchApplication *)kapp)->writeScheme(it.key());
  }
  QDialog::accept();
}

void CalibrateDialog::contextHelp()
{
  kapp->invokeHelp("CalibrateDialog");
}

void CalibrateDialog::schemeChanged(int)
{
  commitColor();
  fillColorList();
}

void CalibrateDialog::selectionChanged(QListBoxItem *item)
{
  if (m_item) commitColor();
  m_item = (FlossListBoxItem *)item;
  if (m_calibratedColors[m_scheme].contains(m_item->floss->name))
    m_sampleColor = m_calibratedColors[m_scheme][m_item->floss->name];
  else
    m_sampleColor = m_item->floss->color;
  updateSample();
  updateName(m_calibratedColors[m_scheme].contains(m_item->floss->name));
}

void CalibrateDialog::redChanged(int r)
{
  m_sampleColor.setRgb(r,m_sampleColor.green(),m_sampleColor.blue());
  updateSample();
  updateName(true);
}

void CalibrateDialog::greenChanged(int g)
{
  m_sampleColor.setRgb(m_sampleColor.red(),g,m_sampleColor.blue());
  updateSample();
  updateName(true);
}

void CalibrateDialog::blueChanged(int b)
{
  m_sampleColor.setRgb(m_sampleColor.red(),m_sampleColor.green(),b);
  updateSample();
  updateName(true);
}

void CalibrateDialog::resetColor()
{
  m_sampleColor = m_item->floss->color;
  if (m_calibratedColors[m_scheme].contains(m_item->floss->name))
    m_calibratedColors[m_scheme].remove(m_item->floss->name);
  updateSample();
  updateName(false);
  m_item->setCalibrated(false);
}

void CalibrateDialog::fillSchemeList()
{
  SchemeList->insertStringList(((KXStitchApplication *)kapp)->schemes());
}

void CalibrateDialog::fillColorList()
{
  ColorList->clear();
  m_scheme = SchemeList->currentText();
  FlossScheme *scheme = ((KXStitchApplication *)kapp)->scheme(m_scheme);
  QPtrListIterator<Floss> it = scheme->flossIterator();
  Floss *f;
  for ( ; (f = it.current()) ; ++it)
    new FlossListBoxItem(ColorList, f);
  selectionChanged(ColorList->item(0));
}

void CalibrateDialog::updateSample()
{
  m_sample->resize(ExampleColor->size());
  m_sample->fill(m_sampleColor);
  ExampleColor->setPixmap(*m_sample);
  RedSlider->setValue(m_sampleColor.red());
  GreenSlider->setValue(m_sampleColor.green());
  BlueSlider->setValue(m_sampleColor.blue());
}

void CalibrateDialog::updateName(bool m)
{
  SelectedColorName->setText(QString("%1-%2%3").arg(m_item->floss->name).arg(m_item->floss->description).arg((m)?i18n(" (Modified)"):""));
}

void CalibrateDialog::commitColor()
{
  if (m_item->floss->color != m_sampleColor)
  {
    m_calibratedColors[m_scheme][m_item->floss->name] = m_sampleColor;
    m_item->setCalibrated(true);
  }
}
