/***************************************************************************
                          kpgdatabase.cpp  -  description
                             -------------------
    begin                : � led 6 2004
    copyright            : (C) 2004 by Lummir Vanek
    email                : lvanek@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for Qt
#include <qtextcodec.h>

// include files for KDE
#include <kdebug.h>
#include <klocale.h>
#include <kcursor.h>
#include <kmessagebox.h>
#include <klocale.h>

// application specific includes
#include "kpgitemsfolder.h"
#include "kpgdatabase.h"
#include "kpgconnection.h"
#include "kpgserver.h"
#include "kpgschema.h"
#include "kpgcastsfolder.h"
#include "kpglanguagesfolder.h"
#include "../kpglinklabel.h"
#include "../kpogreview.h"
#include "../kpgconfiguration.h"
#include "../kpggeneralsettings.h"
#include "../kpgutil.h"
#include "../kpgsqldialog.h"


KPGDatabase::KPGDatabase(KPGServer *parent, const QString name, pqxx::oid _oid)
  : KPGObject(parent, name, _oid)
{
  	setPixmap(0, *m_pIconDatabase);
	m_pDbConnection = 0;
	
	m_pFolderCasts = 0;
	m_pFolderLanguages = 0;
  
  	m_bIsSystemObject = false;	
  	m_bConnectionStateChanged = false;
}

KPGDatabase::KPGDatabase(
  KPGServer *parent,
  KPGDatabase *after,
  const QString name,
  pqxx::oid _oid
  )
  : KPGObject(parent, after, name, _oid)
{
	setPixmap(0, *m_pIconDatabase);
	m_pDbConnection = 0;
	
	m_pFolderCasts = 0;
	m_pFolderLanguages = 0; 
	
	m_bIsSystemObject = false;
	m_bConnectionStateChanged = false;
}

KPGDatabase::~KPGDatabase()
{
  	if(m_pDbConnection)
   		delete m_pDbConnection;
}

void KPGDatabase::activate()
{
  	if(!m_pDbConnection && m_bAllowConnections)
  	{ 
		listView()->setCursor(KCursor::waitCursor());
		
		try
		{
            connectToDB();
            if(m_bIsTemplate)
            {
                setPixmap(0, *m_pIconDatabaseSys);
                m_bIsSystemObject = true;
            }
            else
                setPixmap(0, *m_pIconDatabaseUsr);
		}
		catch (const KPGSqlException &e)
		{
			listView()->setCursor(KCursor::arrowCursor());
			KPGSqlDialog dlg(0, e.sql(), e.message());
			dlg.exec();
		}
		catch(const std::exception &e)
		{
            QTextCodec *pTextCodec = QTextCodec::codecForLocale(); 
            KMessageBox::sorry(0, pTextCodec->toUnicode(e.what()));
		}
		
		listView()->setCursor(KCursor::arrowCursor());
	}
}

KPGServer * KPGDatabase::KPGDatabase::server() const 
{ 
	return static_cast <KPGServer *> (parent()); 
}

// Connect to PostgreSQL database server
void KPGDatabase::connectToDB() throw(const std::exception)
{
	if(m_pDbConnection)
		return; // already connected
	
	try
	{
		m_pDbConnection = new KPGConnection(connectionString());
		
		m_pDbConnection->setPassword(server()->connection()->password()); // store password for future use
		
		m_bConnectionStateChanged = true;
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << "Failed to open database " << endl;
		throw;
	}  
	
	try
    {
        refresh();
    }
    catch (const KPGSqlException &e)
    {
        KPGSqlDialog dlg(0, e.sql(), e.message());
        dlg.exec();
    }
}

void KPGDatabase::disconnectFromDB()
{
	setPixmap(0, *m_pIconDatabase);
	
	// delete all child items
  	while(QListViewItem * pItem = firstChild())
    	delete pItem;
		
	if(!m_pDbConnection)
    	return; // already disconnected
		
	delete m_pDbConnection;
	m_pDbConnection = 0;
	m_bConnectionStateChanged = true;
}

const QString KPGDatabase::connectionString()
{
	KPGTreeItem *parentItem = static_cast <KPGTreeItem *> (parent());
	if(parentItem->type() != nodeServer)
	{
		kdError() << k_funcinfo << " Cannot find parent server." << endl;
		return "";
	}
		
	KPGServer *parentServer = static_cast <KPGServer *> (parentItem);
	KPGConnection *pConnection = parentServer->connection();
	if(!pConnection)
	{
		kdError() << k_funcinfo << " Parent server has no connection." << endl;
		return "";
	}
	
	
	QString strConnectString = makeConnectionString(
		pConnection->hostname(),
		pConnection->port(),
		QString("'" + text(0) + "'"),
		pConnection->username(),
		pConnection->password()
		);
		
	return strConnectString;
}

void KPGDatabase::refresh() throw(const KPGSqlException &)
{
	// delete all child items
	while(QListViewItem * pItem = firstChild())
		delete pItem;
	
	if(m_pDbConnection == 0)
		return;	
				
	//--- Folders for casts and languages
	m_pFolderCasts = new KPGCastsFolder(this);
	m_pFolderLanguages = new KPGLanguagesFolder(this);	
	
	// Get pointer to server for version info
	//KPGServer *pServer = static_cast <KPGServer *> (parent());
		
	// obtain list of schemas
	QString strQuery("SELECT nsp.oid, nsp.nspname, description, nspacl, has_schema_privilege(nsp.oid, 'CREATE') as cancreate, pg_get_userbyid(nspowner) AS owner ");
	
	strQuery.append("FROM pg_catalog.pg_namespace nsp ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_description des ON des.objoid=nsp.oid ");
		
	strQuery.append("ORDER BY nspname");
				
	try
	{
	   m_pqxxResultSchemas = m_pDbConnection->runQuery(strQuery);
	   KPGSchema *pSchema = 0;
			
	   for (result::size_type i = 0; i != m_pqxxResultSchemas.size(); ++i)
	   {
	       pqxx::oid oid;
		   m_pqxxResultSchemas[i][0].to(oid);
		   
		   if(pSchema == 0)
		      pSchema = new KPGSchema(this, m_pqxxResultSchemas[i]["nspname"].c_str(), oid);
		   else
              pSchema = new KPGSchema(this, pSchema, m_pqxxResultSchemas[i]["nspname"].c_str(), oid);
				
			pSchema->setProperties(m_pqxxResultSchemas[i], m_pDbConnection);
									
			//kdDebug() << "------- START: " << m_pqxxResultSchemas[i]["nspname"].c_str() << endl;	
			if(pSchema->isSystemObject())
		    {
		      if(KPoGreView::configuration()->general()->autoLoadSysSchemas())
		          pSchema->refresh();
		    }
            else
            {
                if(KPoGreView::configuration()->general()->autoLoadUsrSchemas())
                  pSchema->refresh();
            }
		
			//kdDebug() << "--------END: " << m_pqxxResultSchemas[i]["nspname"].c_str() << endl;
		}
			
		//--- Folders for casts and languages
		m_pFolderCasts->refresh();
		m_pFolderLanguages->refresh();
	}
	catch (const KPGSqlException &e)
	{
		kdError() << k_funcinfo << "Routing exception up. Database: " << text(0) << endl;
		throw; // if it is KPGSqlException re-throw it
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(m_pDbConnection, e.what(), strQuery);
	} 
		
	setOpen(true);
}

// Refresh only database info, without childs objects
void KPGDatabase::refreshItem() throw(const KPGSqlException &)
{
    KPGServer *pServer = static_cast <KPGServer *> (parent());
    
    bool bVersion80_OrNewer = false;
	bool bVersion81_OrNewer = false;
	    
	// Is it 8.0 or newer ?
	if(pServer->versionMajor() > 7)
    {             
       bVersion80_OrNewer = true;
    }     
	    
    // Is it 8.1 or newer ?
	if(((pServer->versionMajor() == 8) && (pServer->versionMiddle() >= 1)) || ((pServer->versionMajor() > 8))) 
	{
		bVersion81_OrNewer = true;
	}
    
    QString strQuery("SELECT datname, pg_get_userbyid(datdba) AS datowner");
    
    if(!bVersion80_OrNewer)
        strQuery.append(", datpath "); // deprecated in PostgreSQL 8
    else
        strQuery.append(", dattablespace, spcname ");
        
    strQuery.append(", pg_encoding_to_char(encoding) AS serverencoding");
    
    if(bVersion81_OrNewer) 
	{
		strQuery.append(", pg_catalog.pg_size_pretty(pg_catalog.pg_database_size(db.oid)) AS size_pretty");
	}
    
    strQuery.append(", has_database_privilege(db.oid, 'CREATE') as cancreate, datconfig, datallowconn, datistemplate, datacl ");
           
    if(bVersion81_OrNewer) 
	{
		strQuery.append(", pg_catalog.pg_database_size(db.oid) AS database_size ");
	}
           
    strQuery.append("FROM pg_catalog.pg_database db ");
    
    if(bVersion80_OrNewer)
        strQuery.append("LEFT JOIN pg_catalog.pg_tablespace ts ON db.dattablespace=ts.oid ");
    
    strQuery.append("WHERE db.oid = " + QString("%1").arg(m_oid));  
        
    try
    {
        pqxx::result pqxxResultDatabases = m_pDbConnection->runQuery(strQuery);
        
        if(pqxxResultDatabases.size() != 1)
        {
            kdError() << k_funcinfo "Expect one row in result !" <<  endl;
            return;
        } 
        
        setProperties(pqxxResultDatabases[0], bVersion80_OrNewer, bVersion81_OrNewer); 
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(m_pDbConnection, e.what(), strQuery);
    } 
    
    refresh();
}

pqxx::result KPGDatabase::runQuery(const QString strQuery, KPGConnection::ETransType transType)
{
 	if(m_pDbConnection == 0)
	{
		kdError() << k_funcinfo << " Database not connected to server !" << endl;
   	    KMessageBox::sorry(0, i18n("Database not connected to server !"));
		pqxx::result R;
		return R;
	}
		
    return m_pDbConnection->runQuery(strQuery, transType);
}

// Set database propertie
void KPGDatabase::setProperties(const pqxx::result::tuple & pqxxTuple, 
    bool bVersion80_OrNewer, bool bVersion81_OrNewer)
{
    m_strOwner = pqxxTuple["datowner"].c_str(), 
    m_strACL = pqxxTuple["datacl"].c_str(), 
    m_strEncoding = pqxxTuple["serverencoding"].c_str();
	m_strDefaultVariables = pqxxTuple["datconfig"].c_str();
	pqxxTuple["cancreate"].to(m_bCanCreate);
    pqxxTuple["datallowconn"].to(m_bAllowConnections);
    pqxxTuple["datistemplate"].to(m_bIsTemplate);
    
	if(!bVersion80_OrNewer)
	{
	   m_strPath = pqxxTuple["datpath"].c_str();
	   m_oidTablespace = 0;
	}
	else
	{
	   pqxxTuple["dattablespace"].to(m_oidTablespace);
       m_strTablespace = pqxxTuple["spcname"].c_str();
	}
	
	if(bVersion81_OrNewer)
	{
		pqxxTuple["database_size"].to(m_llSize);
		m_strPrettySize = pqxxTuple["size_pretty"].c_str(); 
	}
}

const QString KPGDatabase::queryStatistics() const
{	
	// obtain list of statistics	
	QString strQuery(QString("SELECT numbackends, xact_commit, xact_rollback, blks_read, blks_hit FROM pg_catalog.pg_stat_database WHERE datid = '%1'").arg(m_oid));
	
	return strQuery;
}

const pqxx::result & KPGDatabase::resultCasts() const
{
	return m_pFolderCasts->resultCasts();
}

const pqxx::result & KPGDatabase::resultLanguages() const
{
	return m_pFolderLanguages->resultLanguages();
}

const pqxx::result & KPGDatabase::resultSchemas() const
{
	return m_pqxxResultSchemas;
}

const QString KPGDatabase::queryExtendedStatistics(int currentItem) const
{
	
	QString strQuery;
	
	switch(currentItem)
	{
		case 0:
			strQuery = "SELECT relid, relname, schemaname, seq_scan, seq_tup_read, idx_scan, idx_tup_fetch, n_tup_ins, n_tup_upd, n_tup_del FROM pg_catalog.pg_stat_all_tables";
			break;
				
		case 1:
			strQuery = "SELECT relid, relname, schemaname, seq_scan, seq_tup_read, idx_scan, idx_tup_fetch, n_tup_ins, n_tup_upd, n_tup_del FROM pg_catalog.pg_stat_sys_tables";
			break;
			
		case 2:
			strQuery = "SELECT relid, relname, schemaname, seq_scan, seq_tup_read, idx_scan, idx_tup_fetch, n_tup_ins, n_tup_upd, n_tup_del FROM pg_catalog.pg_stat_user_tables";
			break;
			
		case 3:
			strQuery = "SELECT relid, relname, schemaname, indexrelid, indexrelname, idx_scan, idx_tup_read, idx_tup_fetch FROM pg_catalog.pg_stat_all_indexes";
			break;
	
		case 4:
			strQuery = "SELECT relid, relname, schemaname, indexrelid, indexrelname, idx_scan, idx_tup_read, idx_tup_fetch FROM pg_catalog.pg_stat_sys_indexes";
			break;
	
		case 5:
			strQuery = "SELECT relid, relname, schemaname, indexrelid, indexrelname, idx_scan, idx_tup_read, idx_tup_fetch FROM pg_catalog.pg_stat_user_indexes";
			break;
	
		case 6:
			strQuery = "SELECT relid, relname, schemaname, heap_blks_read, heap_blks_hit, idx_blks_read, idx_blks_hit, toast_blks_read, toast_blks_hit, tidx_blks_read, \
tidx_blks_hit FROM pg_catalog.pg_statio_all_tables";
			break;      
			
		case 7:
			strQuery = "SELECT relid, relname, schemaname, heap_blks_read, heap_blks_hit, idx_blks_read, idx_blks_hit, toast_blks_read, toast_blks_hit, tidx_blks_read, \
tidx_blks_hit FROM pg_catalog.pg_statio_sys_tables";
			break;
			
		case 8:
			strQuery = "SELECT relid, relname, schemaname, heap_blks_read, heap_blks_hit, idx_blks_read, idx_blks_hit, toast_blks_read, toast_blks_hit, tidx_blks_read, \
tidx_blks_hit FROM pg_catalog.pg_statio_user_tables";
			break;
			
		case 9:
			strQuery = "SELECT relid, relname, schemaname, indexrelid, indexrelname, idx_blks_read, idx_blks_hit FROM pg_catalog.pg_statio_all_indexes";
			break;
	
		case 10:
			strQuery = "SELECT relid, relname, schemaname, indexrelid, indexrelname, idx_blks_read, idx_blks_hit FROM pg_catalog.pg_statio_sys_indexes";
			break;
	
		case 11:
			strQuery = "SELECT relid, relname, schemaname, indexrelid, indexrelname, idx_blks_read, idx_blks_hit FROM pg_catalog.pg_statio_user_indexes";
			break;
				
		case 12:
			strQuery = "SELECT relid, relname, schemaname, blks_read, blks_hit FROM pg_catalog.pg_statio_all_sequences";
			break;
	
		case 13:
			strQuery = "SELECT relid, relname, schemaname, blks_read, blks_hit FROM pg_catalog.pg_statio_sys_sequences";
			break;
	
		case 14:
			strQuery = "SELECT relid, relname, schemaname, blks_read, blks_hit FROM pg_catalog.pg_statio_user_sequences";
			break;
	
		default:
			kdError() << k_funcinfo << " Unknown choice" << endl;
			return "";
	}
	
	return strQuery;
}

const QString KPGDatabase::queryLocks() const
{
	KPGServer *pServer = static_cast <KPGServer *> (parent());
    
    // Is it 8.1 or newer ? If not, return
	if(((pServer->versionMajor() < 8) || (pServer->versionMiddle() <= 1)) && ((pServer->versionMajor() == 8))) 
	{
		throw PGSTD::runtime_error(i18n("Works only with PostgresQL 8.1 or newer !"));
	}
		
	QString strQuery("SELECT l.locktype, l.relation, rel.relname, l.transactionid, l.pid, l.mode, l.granted ");
 
    strQuery.append("FROM  pg_catalog.pg_locks l ");
    strQuery.append("INNER JOIN pg_catalog.pg_class rel ON l.relation=rel.oid ");
	strQuery.append(QString("WHERE l.database = %1").arg(m_oid));
		
	return strQuery;
}

const QString KPGDatabase::queryDiskUsage() const
{
	KPGServer *pServer = static_cast <KPGServer *> (parent());
    
    // Is it 8.1 or newer ? If not, return
	if(((pServer->versionMajor() < 8) || (pServer->versionMiddle() <= 1)) && ((pServer->versionMajor() == 8))) 
	{
		throw PGSTD::runtime_error(i18n("Works only with PostgresQL 8.1 or newer !"));
	}
		
	QString strQuery("SELECT ns.nspname, cls.oid, cls.relname, CASE cls.relkind WHEN 'r' THEN 'table' WHEN 't' THEN 'toast table' WHEN 'i' THEN 'index' END AS kind, ts.spcname, ");
 
    strQuery.append("pg_catalog.pg_size_pretty(pg_catalog.pg_relation_size(cls.oid)) AS size_pretty, ");
    strQuery.append("pg_catalog.pg_relation_size(cls.oid) AS size ");
    strQuery.append("FROM pg_catalog.pg_class cls ");
    strQuery.append("LEFT JOIN pg_catalog.pg_namespace ns ON cls.relnamespace=ns.oid ");
    
    strQuery.append("LEFT JOIN pg_catalog.pg_tablespace ts ON CASE cls.reltablespace ");
	strQuery.append(QString("WHEN 0 THEN (SELECT dattablespace FROM pg_catalog.pg_database WHERE datname='%1') ").arg(text(0)));
	strQuery.append("ELSE cls.reltablespace ");
    strQuery.append("END=ts.oid ");
    
    strQuery.append("WHERE pg_catalog.pg_relation_size(cls.oid) > 0 AND ");
    strQuery.append("relkind IN('r', 'i', 't') ");
    strQuery.append("ORDER BY size DESC");
		
	return strQuery;
}

void KPGDatabase::fillListOfObjectsForCodeCompletion(KPGOidNameList & listOfObjectOids)
{
    QListViewItem * pItem = firstChild();
	while(pItem)
	{
	   KPGTreeItem *pTreeItem = static_cast <KPGTreeItem *> (pItem);
		
	   if(pTreeItem->type() == KPGTreeItem::nodeSchema)
	   {
		  KPGSchema *pSchema = static_cast <KPGSchema *> (pTreeItem);
		
		  listOfObjectOids.append(KPGOidName(pSchema->oid(), pSchema->text(0), QString::null, pSchema->type(), pSchema->pixmap(0), pSchema->description()));
		  
		  //if(pSchema->text(0) == "public")
		  //{
		  pSchema->fillListOfObjectsForCodeCompletion(listOfObjectOids);
		  //}
	   }
		
		pItem = pItem->nextSibling();
	}
}

