// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

//#undef KDE_NO_COMPAT
#include <kxmlguiclient.h>
#include <kgenericfactory.h>

#include <kopeteonlinestatus.h>
#include <kopeteonlinestatusmanager.h>
#include <kopeteaccountmanager.h>
#include <kopetecommandhandler.h>
#include <kopetechatsession.h>
#include <kopetemessage.h>

#include "silcprotocol.h"
#include "silcaddcontactpage.h"
#include "silceditaccountwidget.h"
#include "silcchannelcontact.h"
#include "silcbuddycontact.h"
#include "silcaccount.h"
#include "silcbuddycliententry.h"


typedef KGenericFactory<SilcProtocol> SilcProtocolFactory;
K_EXPORT_COMPONENT_FACTORY(kopete_silc, SilcProtocolFactory("kopete_silc"));

SilcProtocol *SilcProtocol::_protocol = NULL;

SilcProtocol::SilcProtocol(QObject *parent, const QStringList & /* args */)
  : Kopete::Protocol(SilcProtocolFactory::componentData(), parent),

    // register online statuses ...
    statusOnline(Kopete::OnlineStatus::Online, Online, this,
		 Online, QStringList(QString()), i18n("Online"),
		 i18n("Online"), Kopete::OnlineStatusManager::Online),
    statusOnlineChannel(Kopete::OnlineStatus::Online, OnlineChannel, this,
			OnlineChannel,
			QStringList(QString::fromLatin1("silc_channel")),
			i18n("Online"), i18n("Online"),
			Kopete::OnlineStatusManager::Online,
			Kopete::OnlineStatusManager::HideFromMenu),
    statusConnecting(Kopete::OnlineStatus::Connecting, Connecting, this,
		     Connecting,
		     QStringList(QString::fromLatin1("silc_channel")),
		     i18n("Connecting"), i18n("Connecting"),
		     Kopete::OnlineStatusManager::Offline,
		     Kopete::OnlineStatusManager::HideFromMenu),
    statusHyper(Kopete::OnlineStatus::Online, Hyper, this, Hyper,
		QStringList(QString::fromLatin1("silc_hyper")),
		i18n("Hyper Active"), i18n("Hyper Active"),
		Kopete::OnlineStatusManager::FreeForChat),
    statusGone(Kopete::OnlineStatus::Away, Gone, this, Gone,
	       QStringList(QString::fromLatin1("contact_away_overlay")),
	       i18n("Gone"), i18n("Gone"),
	       Kopete::OnlineStatusManager::Away),
    statusIndisposed(Kopete::OnlineStatus::Away, Indisposed, this, Indisposed,
		     QStringList(QString::fromLatin1("contact_xa_overlay")),
		     i18n("Indisposed"), i18n("Indisposed"),
		     Kopete::OnlineStatusManager::ExtendedAway),
    statusBusy(Kopete::OnlineStatus::Away, Busy, this, Busy,
	       QStringList(QString::fromLatin1("contact_busy_overlay")),
	       i18n("Busy"), i18n("Busy"),
	       Kopete::OnlineStatusManager::Busy),
    statusOffline(Kopete::OnlineStatus::Offline, 0, this, Offline,
		  QStringList(QString()),
		  i18n("Offline"), i18n("Offline"),
		  Kopete::OnlineStatusManager::Offline),
    statusDetached(Kopete::OnlineStatus::Offline, Detached, this, Detached,
		   QStringList(QString::fromLatin1("silc_detached")),
		   i18n("Detached"), i18n("Detached"),
		   Kopete::OnlineStatusManager::Offline,
		   Kopete::OnlineStatusManager::HideFromMenu)
{
  _protocol = this;
  setCapabilities(FullRTF);

  // add our i18n catalogue ...
  KGlobal::locale()->insertCatalog("kopete_silc");

  // register silc-protocol's commands (at least these we do support) ...
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("whois"),
     SLOT(slotWhoisCommand(const QString&, Kopete::ChatSession *)),
     i18n("USAGE: /whois <nickname> - request whois information."), 1);

  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("watch"),
     SLOT(slotWatchCommand(const QString&, Kopete::ChatSession *)),
     i18n("USAGE: /watch -add|-del <nickname> - watch a nickname."), 1);
  
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("notice"),
     SLOT(slotNoticeCommand(const QString &, Kopete::ChatSession *)),
     i18n("USAGE: /notice <message> - send a notice message."), 1);

  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("me"),
     SLOT(slotAction(const QString &, Kopete::ChatSession *)),
     i18n("USAGE: /me <action> - send an action message."), 1);
  
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("getkey"),
     SLOT(slotGetkeyCommand(const QString&, Kopete::ChatSession *)),
     i18n("USAGE: /getkey <nickname> - fetch remote client's public key"), 1);
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("killme"),
    SLOT(slotKillmeCommand(const QString&, Kopete::ChatSession *)),
    i18n("USAGE: /killme"), 0);
}



SilcProtocol::~SilcProtocol()
{
  _protocol = NULL;
}



Kopete::Account *
SilcProtocol::createNewAccount(const QString &accountId)
{
  return new SilcAccount(this, accountId);
}



AddContactPage *
SilcProtocol::createAddContactWidget(QWidget *parent, Kopete::Account *account)
{
  return new SilcAddContactPage(static_cast<SilcAccount *>(account), parent);
}




KopeteEditAccountWidget *
SilcProtocol::createEditAccountWidget(Kopete::Account *account,
				      QWidget *parent)
{
  return new SilcEditAccountWidget(this, static_cast<SilcAccount *>(account),
				   parent);
}


SilcProtocol *
SilcProtocol::protocol(void) 
{
  return _protocol;
}

Kopete::Contact *
SilcProtocol::deserializeContact(Kopete::MetaContact *meta,
				 const QMap<QString, QString> &serializedData, 
				 const QMap<QString, QString> &)
{
  QString contactId = serializedData[ "contactId" ];

  QList<Kopete::Account*> accounts = 
    Kopete::AccountManager::self()->accounts(this);

  if(accounts.isEmpty()) {
    kDebug() << "No accounts loaded, configuration invalid." << endl;
    return NULL;
  }

  Kopete::Account *a = NULL;
  foreach(Kopete::Account *acct, accounts)
    if(acct->accountId() == serializedData["accountId"]) {
      a = acct;
      break;
    }

  if(! a) {
    kDebug() << "Account " << serializedData["accountId"]
	     << "used to be available, but isn't anymore" << endl;
    return NULL;
  }

  if(! a->addContact(contactId, meta))
    return NULL; 

  Kopete::Contact *contact = a->contacts()[contactId];
  if(! strcmp(contact->metaObject()->className(), "SilcBuddyContact")) {
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact *>(contact);
    buddy->setFpTrusted(serializedData["fpTrusted"].compare("yes") == 0);
    buddy->setAllowRichText(serializedData["allowRichText"]
			    .compare("yes") == 0);
  } 
  else if(! strcmp(contact->metaObject()->className(), "SilcChannelContact")) {
    SilcChannelContact *channel = static_cast<SilcChannelContact *>(contact);
    channel->setAllowRichText(serializedData["allowRichText"]
			      .compare("yes") == 0);
    channel->setAutoJoin(serializedData["autoJoin"].compare("yes") == 0);
  }

  return contact;
}

/**
 * sends a WHOIS request to the network for the given nick
 *
 * @todo if possible the public key should be used.
 */

void
SilcProtocol::slotWhoisCommand(const QString &nick, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  account->sendSilcCommand(QString("WHOIS %1").arg(nick));
}

void
SilcProtocol::slotKillmeCommand(const QString &, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  account->sendSilcCommand(QString("KILL %1 -pubkey").arg(account->nickName()));
}

void
SilcProtocol::slotWatchCommand(const QString &cmd, Kopete::ChatSession *cs)
{    
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  if(cmd.contains("-add ") || cmd.contains("-del ") || cmd.contains("-pubkey "))
    account->sendSilcCommand(QString("WATCH %1").arg(cmd));
  // @todo emit error message in case of error?
}

void
SilcProtocol::slotNoticeCommand(const QString &notice, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  Kopete::ContactPtrList members = cs->members();
  Kopete::Contact *contact = members.first();

  QByteArray text = notice.toUtf8();
  SilcTK::SilcMessageFlags flags =
    SILC_MESSAGE_FLAG_NOTICE | SILC_MESSAGE_FLAG_UTF8;

  if(! strcmp(contact->metaObject()->className(), "SilcChannelContact")) {
    // send action to channel ...
    SilcChannelContact *channel = static_cast<SilcChannelContact *>(contact);

    if(account->signChannelNotices())
      flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_channel_message
      (account->client(), account->conn(), channel->channelEntry(), NULL,
       flags, account->sha1hash, (unsigned char *) text.constData(), 
       text.length());

  }
  else if(! strcmp(contact->metaObject()->className(), "SilcBuddyContact")) {
    // send action to buddy ...
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact *>(contact);

    if(account->signPrivateNotices())
      flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_private_message
      (account->client(), account->conn(), buddy->clientEntries()(), flags,
       account->sha1hash, (unsigned char *) text.constData(), text.length());
  }
  else 
    return; // ignore for SilcServerContact ...

  Kopete::Message msg(account->myself(), cs->members());
  msg.setPlainBody(notice);
  msg.setDirection(Kopete::Message::Internal);
  msg.setType(Kopete::Message::TypeAction);

  SilcContact::prettyPrintMessage(msg, flags);

  cs->appendMessage(msg);
}

void
SilcProtocol::slotGetkeyCommand(const QString &nick, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  account->sendSilcCommand(QString("GETKEY %1").arg(nick));
}

void
SilcProtocol::slotAction(const QString &action, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  Kopete::ContactPtrList members = cs->members();
  Kopete::Contact *contact = members.first();

  QByteArray text = action.toUtf8();
  SilcTK::SilcMessageFlags flags =
    SILC_MESSAGE_FLAG_ACTION | SILC_MESSAGE_FLAG_UTF8;

  if(! strcmp(contact->metaObject()->className(), "SilcChannelContact")) {
    // send action to channel ...
    SilcChannelContact *channel = static_cast<SilcChannelContact *>(contact);

    if(account->signChannelActions())
      flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_channel_message
      (account->client(), account->conn(), channel->channelEntry(), NULL,
       flags, account->sha1hash, (unsigned char *) text.constData(), 
       text.length());

  }
  else if(! strcmp(contact->metaObject()->className(), "SilcBuddyContact")) {
    // send action to buddy ...
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact *>(contact);

    if(account->signPrivateActions())
      flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_private_message
      (account->client(), account->conn(), buddy->clientEntries()(), flags,
       account->sha1hash, (unsigned char *) text.constData(), text.length());
  }
  else 
    return; // ignore for SilcServerContact ...

  Kopete::Message msg(account->myself(), cs->members());
  msg.setPlainBody(action);
  msg.setDirection(Kopete::Message::Outbound);
  msg.setType(Kopete::Message::TypeAction);
  SilcContact::prettyPrintMessage(msg, flags);

  cs->appendMessage(msg);

}

#include "silcprotocol.moc"
