#include "kz-downloader.h"

#include <gcutter.h>
#include "kz-test-utils.h"

void test_new (void);
void test_new_with_filename (void);
void test_to_buffer (void);
void test_to_file (void);
void test_uri (void);
void test_filename (void);

static KzDownloader *downloader;

static gint n_start_received;
static gint n_completed_received;
static gint n_error_received;
static gchar *error_message;
static gchar *expected_buffer;
static gchar *actual_buffer;
static gchar *file_path;
static const gchar target_uri[] = "http://svn.sourceforge.jp/view/kazehakase/trunk/test/fixtures/bookmarkbar.xml?view=co&root=kazehakase";
static const gchar download_filename[] = "downloaded-file";

void
setup (void)
{
    downloader =  NULL;

    n_start_received = 0;
    n_completed_received = 0;
    n_error_received = 0;

    error_message = NULL;
    expected_buffer = NULL;
}

void
teardown (void)
{
    if (downloader)
        g_object_unref(downloader);
    g_free(expected_buffer);
    g_free(actual_buffer);
    g_free(file_path);

    cut_remove_path(download_filename, NULL);
}

void
test_new (void)
{
    downloader = kz_downloader_new(target_uri);
    cut_assert(downloader);
}

void
test_new_with_filename (void)
{
    downloader = kz_downloader_new_with_filename(target_uri,
                                                 download_filename);
    cut_assert(downloader);
}

static gboolean
cb_check_emitted (gpointer data)
{
    gboolean *emitted = data;

    *emitted = TRUE;
    return FALSE;
}

static void
wait_signal (gint *n_signals)
{
    gboolean timeout_emitted = FALSE;
    guint timeout_id;

    timeout_id = g_timeout_add_seconds(1, cb_check_emitted, &timeout_emitted);
    while (!timeout_emitted && *n_signals == 0)
        g_main_context_iteration(NULL, TRUE);

    g_source_remove(timeout_id);

    cut_assert_false(timeout_emitted, "timeout");
}

static void 
cb_start (KzDownloader *downloader, gpointer data)
{
    n_start_received++;
}

static void 
cb_completed (KzDownloader *downloader, gpointer data)
{
    n_completed_received++;
}

static void 
cb_error (KzDownloader *downloader, const gchar *message, gpointer data)
{
    n_error_received++;
    error_message = g_strdup(message);
}

static void
setup_signals (void)
{
    g_signal_connect(downloader, "start",
                     G_CALLBACK(cb_start), NULL);
    g_signal_connect(downloader, "completed",
                     G_CALLBACK(cb_completed), NULL);
    g_signal_connect(downloader, "error",
                     G_CALLBACK(cb_error), NULL);
}

void
test_to_buffer (void)
{
    GError *error = NULL;
    gsize expected_size;

    cut_trace(test_new());

    setup_signals();
    cut_assert_true(kz_downloader_to_buffer(downloader));

    wait_signal(&n_start_received);
    cut_assert_equal_int(1, n_start_received);
    wait_signal(&n_completed_received);
    cut_assert_equal_int(0, n_error_received);
    cut_assert_equal_int(1, n_completed_received);

    file_path = g_build_filename(kz_test_get_base_dir(),
                                 "fixtures", "bookmarkbar.xml",
                                 NULL);

    g_file_get_contents(file_path, &expected_buffer, &expected_size, &error);
    gcut_assert_error(error);

    cut_assert_equal_string(expected_buffer, kz_downloader_get_buffer(downloader));
    gcut_assert_equal_uint64(expected_size,
                             kz_downloader_get_size(downloader));
}

void
test_to_file (void)
{
    GError *error = NULL;
    gsize expected_size;

    cut_trace(test_new_with_filename());

    setup_signals();
    cut_assert_true(kz_downloader_to_file(downloader));

    wait_signal(&n_start_received);
    cut_assert_equal_int(1, n_start_received);
    wait_signal(&n_completed_received);
    cut_assert_equal_int(0, n_error_received);
    cut_assert_equal_int(1, n_completed_received);

    cut_assert_file_exist(download_filename);
    g_file_get_contents(download_filename, &actual_buffer, NULL, &error);
    gcut_assert_error(error);

    file_path = g_build_filename(kz_test_get_base_dir(),
                                 "fixtures", "bookmarkbar.xml",
                                 NULL);

    g_file_get_contents(file_path, &expected_buffer, &expected_size, &error);
    gcut_assert_error(error);

    cut_assert_equal_string(expected_buffer, actual_buffer);
    gcut_assert_equal_uint64(expected_size,
                             kz_downloader_get_size(downloader));
}

void
test_uri (void)
{
    cut_trace(test_new());

    cut_assert_equal_string(target_uri,
                            kz_downloader_get_uri(downloader));
}

void
test_filename (void)
{
    cut_trace(test_new_with_filename());

    cut_assert_equal_string(download_filename,
                            kz_downloader_get_filename(downloader));
}

/*
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
