//---------------------------------------------------------------------
// Ipe preference settings
//---------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipexml.h"

#include "ipeq.h"
#include "ipeprefs.h"

#if defined(WIN32)
#include <windows.h>
#endif
#include <stdlib.h>

#include <QDir>
#include <QFile>

// --------------------------------------------------------------------

/*! \class IpePreferences
  \brief The configurable settings of the Ipe program.

  There are compile-time defaults, which can be overridden by
  environment variables.  The compression levels are saved to
  <tt>$HOME/.iperc</tt> (on Unix) or <tt>$HOME/_ipecfg</tt> (on
  Windows).

  The interesting data members are all public.

*/

#ifdef WIN32
#define SEP ';'
#else
#define SEP ':'
#endif

const char * const stdFontMap[] = {
  "Times-Roman",
  "Times-Italic",
  "Times-Bold",
  "Times-BoldItalic",
  "Helvetica",
  "Helvetica-Oblique",
  "Helvetica-Bold",
  "Helvetica-BoldOblique",
  "Courier",
  "Courier-Oblique",
  "Courier-Bold",
  "Courier-BoldOblique",
  "Symbol",
  "ZapfDingbats",
};

const char * const stdFontMapAlias[] = {
   "NimbusRomNo9L-Regu",
   "NimbusRomNo9L-ReguItal",
   "NimbusRomNo9L-Medi",
   "NimbusRomNo9L-MediItal",
   "NimbusSanL-Regu",
   "NimbusSanL-ReguItal",
   "NimbusSanL-Bold",
   "NimbusSanL-BoldItal",
   "NimbusMonL-Regu",
   "NimbusMonL-ReguObli",
   "NimbusMonL-Bold",
   "NimbusMonL-BoldObli",
   "StandardSymL",
   "Dingbats",
};

const int ConfigFileRevisionLevel = 27;

IpePreferences *IpePreferences::This = 0;

//! Return a pointer to the unique IpePreferences instance.
IpePreferences *IpePreferences::Static()
{
  if (!This) {
    This = new IpePreferences;
  }
  return This;
}

//! Constructor sets up preferences.
/*! Preferences are read from the preferences file
  (<tt>$HOME/.iperc</tt> on Unix, <tt>$HOME/_ipecfg</tt> on Windows).
  If the file doesn't exist or is not readable, built-in defaults are
  used.

  The directory settings are computed from the location of the
  executable on Windows, and taken from compile-time settings on Unix.
  In both cases, environment variables override these settings.
*/

IpePreferences::IpePreferences()
{
  // First set compile-time defaults
  iCompressLevel = 9;
  iSelectDistance = 36;
  iSnapDistance = 16;
  iRightMouseSelects = true;
  iWhitePaper = false;
  iAntiAlias = true;
  iDoubleBuffer = true;
  iPropertiesShape = 0;
  iGridVisible = true;
  iMaximize = false;
  iGridSize = 16;
  iStyleSheet = QString::null;
  iStyleSheet2 = QString::null;
  iAutoExport = ENoAutoExport;
  iExportExist = false;
  iBitmapSize = 600;
  iLanguage = QString::null;
  iMikTeX = false;
  QDir home = QDir::home();
#ifdef WIN32
  char exename[OFS_MAXPATHNAME];
  GetModuleFileNameA(0, exename, OFS_MAXPATHNAME);
  QFileInfo finfo = QFileInfo(QLatin1String(exename));
  QDir ipedir = finfo.absoluteDir();
  ipedir.cdUp();
  iDocDir = ipedir.filePath(QLatin1String("doc"));
  iFontmap = ipedir.filePath(QLatin1String("data/fontmap.xml"));
  iStdKeysFileName = ipedir.filePath(QLatin1String("data/ipekeys.xml"));
  iLangDir = ipedir.filePath(QLatin1String("languages"));
  iIpeletPath.append(ipedir.filePath(QLatin1String("ipelets")));
  iLatexDir = ipedir.filePath(QLatin1String("latexrun"));
#else
  QDir ipedir = home.filePath(QLatin1String(".ipe"));
  if (!ipedir.exists())
    home.mkdir(QLatin1String(".ipe"));
  iLatexDir = ipedir.filePath(QLatin1String("latexrun"));
  iDocDir = QLatin1String(IPEDOCDIR);
  iFontmap = QLatin1String(IPEFONTMAP);
  iStdKeysFileName = QLatin1String(IPEKEYSFILE);
  iBrowser = QLatin1String(IPEBROWSER);
  iIpeletPath.append(QLatin1String(IPELETPATH));
  iLangDir = QLatin1String(IPELANGDIR);
#endif
  iPdfLatex = QLatin1String("pdflatex");
  const char *p;
  p = getenv("IPELATEXDIR");
  if (p) iLatexDir = QLatin1String(p);
  p = getenv("IPEPDFLATEX");
  if (p) iPdfLatex = QLatin1String(p);
  p = getenv("IPEDOCDIR");
  if (p) iDocDir = QLatin1String(p);
  p = getenv("IPELANGDIR");
  if (p) iLangDir = QLatin1String(p);
  p = getenv("IPEFONTMAP");
  if (p) iFontmap = QLatin1String(p);
  p = getenv("IPEKEYSFILE");
  if (p) iStdKeysFileName = QLatin1String(p);
  p = getenv("IPEBROWSER");
  if (p) iBrowser = QLatin1String(p);
  p = getenv("IPELETPATH");
  if (p)
    iIpeletPath = QString(QLatin1String(p)).split(QLatin1Char(SEP));
  p = getenv("TEXINPUTS");
  if (p)
    iTexInputs = p;

#ifdef WIN32
  iPrefsFileName = home.filePath(QLatin1String("_ipecfg"));
  iKeysFileName = home.filePath(QLatin1String("_ipekeys.xml"));
#else
  iPrefsFileName = ipedir.filePath(QLatin1String("iperc"));
  iKeysFileName = ipedir.filePath(QLatin1String("ipekeys.xml"));
#endif
  iDialogDir = QDir::current().path();
  QFile file(iPrefsFileName);
  if (!file.open(QIODevice::ReadOnly))
    return;
  QDataStream ds(&file);
  int revLevel;
  ds >> revLevel;
  if (revLevel != ConfigFileRevisionLevel)
    return;
  qint8 flag;
  int val;
  ds >> iCompressLevel;
  ds >> iSelectDistance;
  ds >> iSnapDistance;
  ds >> flag; iWhitePaper = !!flag;
  ds >> flag; iAntiAlias = !!flag;
  ds >> flag; iDoubleBuffer = !!flag;
  ds >> flag; iGridVisible = !!flag;
  ds >> flag; iMaximize = !!flag;
  ds >> iGridSize;
  ds >> iPropertiesShape;
  ds >> flag; iExportExist = !!flag;
  ds >> val; iAutoExport = TExportMode(val);
  ds >> iStyleSheet;
  ds >> iFont;
  ds >> iBitmapSize;
  ds >> iLanguage;
  ds >> flag; iMikTeX = !!flag;
  ds >> iToolbarState;
  FindStandardFonts();
}

//! Save preferences to preferences file.
bool IpePreferences::Save()
{
  QFile file(iPrefsFileName);
  if (!file.open(QIODevice::WriteOnly))
    return false;
  QDataStream ds(&file);
  ds << ConfigFileRevisionLevel
     << iCompressLevel
     << iSelectDistance
     << iSnapDistance
     << qint8(iWhitePaper)
     << qint8(iAntiAlias)
     << qint8(iDoubleBuffer)
     << qint8(iGridVisible)
     << qint8(iMaximize)
     << iGridSize
     << iPropertiesShape
     << qint8(iExportExist)
     << int(iAutoExport)
     << iStyleSheet
     << iFont
     << iBitmapSize
     << iLanguage
     << qint8(iMikTeX)
     << iToolbarState;
  file.close();
  emit Changed();
  return true;
}

// --------------------------------------------------------------------

class FontmapParser : public IpeXmlParser {
public:
  explicit FontmapParser(IpeDataSource &source) :
    IpeXmlParser(source) { /* nothing */ }
  bool ParseFontmap();

  QString iGlyphs[14];
};

// Parse a fontmap.
bool FontmapParser::ParseFontmap()
{
  IpeXmlAttributes att;
  IpeString tag = ParseToTag();

  if (tag == "?xml") {
    if (!ParseAttributes(att, true))
      return false;
    tag = ParseToTag();
  }

  if (tag != "fontmap")
    return false;
  if (!ParseAttributes(att))
    return false;
  tag = ParseToTag();

  while (tag == "font") {
    if (!ParseAttributes(att))
      return false;
    if (att["format"] == "type1") {
      IpeString name = att["name"];
      IpeString glyphs = att["glyphs"];
      int i = 0;
      while (i < 14
	     && name != IpeString(stdFontMap[i])
	     && name != IpeString(stdFontMapAlias[i]))
	++i;
      if (i < 14)
	iGlyphs[i] = QIpe(glyphs);
    }
    tag = ParseToTag();
  }

  if (tag != "/fontmap")
    return false;
  return true;
}


// --------------------------------------------------------------------

//! Parse the fontmap and store absolute paths for standard fonts.
void IpePreferences::FindStandardFonts()
{
  QFile file(iFontmap);
  QDir dir = QFileInfo(file).dir();

  if (!file.open(QIODevice::ReadOnly)) {
    qDebug("Could not open fontmap '%s'", iFontmap.toLatin1().constData());
    return;
  }

  QDataSource source(&file);
  FontmapParser parser(source);
  if (parser.ParseFontmap()) {
    for (int i = 0; i < 14; ++i) {
      if (parser.iGlyphs[i] != QString::null) {
	QString fname = dir.absoluteFilePath(parser.iGlyphs[i]);
	if (QFile::exists(fname))
	  iStandardFont[i] = fname;
      }
    }
  }
  file.close();

  for (int i = 0; i < 14; ++i) {
    if (iStandardFont[i] == QString::null) {
      qDebug("No font file found for '%s'", stdFontMap[i]);
    } else {
      ipeDebug("Standard font '%s' uses font file '%s",
	       stdFontMap[i], iStandardFont[i].toLatin1().constData());
    }
  }
}

//! Return contents of font file for standard font \a name.
IpeBuffer IpePreferences::StandardFont(IpeString name)
{
  for (int i = 0; i < 14; ++i) {
    if (name == stdFontMap[i]) {
      // if it's known and not yet loaded, load it
      if (iStandardFont[i] != QString::null
	  && iStandardFontBuffer[i].size() == 0) {
	QFile file(iStandardFont[i]);
	if (file.open(QIODevice::ReadOnly)) {
	  QByteArray a = file.readAll();
	  file.close();
	  iStandardFontBuffer[i] = IpeBuffer(a.data(), a.size());
	}
      }
      return iStandardFontBuffer[i];
    }
  }
  return IpeBuffer();
}

// --------------------------------------------------------------------
