/* daemon.c: various functions and helper things for daemon support */
/*
 * ----------------------------------------------------------------------------
 * "THE BEER-WARE LICENSE" (Revision 42):
 * <wouter@debian.org> wrote this software. As long as you retain this
 * notice you can do whatever you want with this stuff. If we meet some
 * day, and you think this stuff is worth it, you can buy me a beer in
 * return. Wouter Verhelst.
 * ----------------------------------------------------------------------------
 * (with thanks to phk@FreeBSD.org)
 */
#include <ipcfg/commands.h>
#include <ipcfg/daemon.h>
#include <ipcfg/macros.h>

#include <ipcfg/backend/pid.h>

#include <ipcfg/private/config.h>

#include <errno.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>

/* Socket */

int csock;
int ssock;

/* Daemon status functions */

static bool daemonized = false;
static bool allowed = true;

bool ipcfg_daemon_allowed() {
	return allowed;
}

bool ipcfg_am_daemon() {
	return daemonized;
}

bool ipcfg_daemon_is_running() {
	char* reply;
	if(ipcfg_am_daemon()) {
		return true;
	}
	if((reply=ipcfg_daemon_send_command("PING"))>=0) {
		free(reply);
		return true;
	}
	return false;
}

int ipcfg_go_daemon() {
	int retval;
	int pidfile;
	char buf[6];

	if(ipcfg_daemon_is_running()) {
		/* Another process is already running in daemon mode. We can't
		 * become a daemon in that case */
		return 1;
	}
	if(!ipcfg_daemon_allowed()) {
		/* daemon mode was disabled in the config file */
		return 2;
	}
	if(geteuid()) {
		/* Only root can start the daemon */
		return 3;
	}
	/* Okay, set up stuff now:
	 * - PID file
	 * - UNIX domain socket to communicate with non-daemonized processes
	 */
    retry:
	if(!(pidfile = open(PIDFILEDIR "/ipcfgd", O_CREAT | O_EXCL | O_RDWR))) {
		snprintf(buf, 6, "%d", getpid());
		write(pidfile, buf, strlen(buf));
		write(pidfile, "\n", 1);
		close(pidfile);
	} else {
		/* PID file already exists. Verify whether the daemon is in
		 * fact still running */
		if(!be_pidfile_isrunning(PIDFILEDIR "/ipcfgd")) {
			unlink(PIDFILEDIR "/ipcfgd");
			goto retry;
		}
	}
	retval = daemon(0, 0);
	daemonized = true;
	return retval;
}

int ipcfg_forbid_daemon(void) {
	if(ipcfg_daemon_is_running()) {
		/* XXX die a screaming and hollering death */
		exit(EXIT_FAILURE);
	}
	allowed = false;
	return 0;
}

/* Daemon server bits */

/* Daemon client bits */

char* ipcfg_daemon_send_command(char* command) {
	char buf[256];
	char* p=NULL;
	size_t size=256;
	ssize_t read_bytes;
	bool cont=true;

	if(IPCFG_EXPECT_FALSE(!csock && !ssock)) {
		int sock_flags;
		if(IPCFG_EXPECT_FALSE(csock == 0)) {
			/* Can't be client and server at the same time */
			return NULL;
		}
		struct sockaddr_un sa;
		csock = socket(AF_UNIX, SOCK_STREAM, 0);
		sock_flags = fcntl(csock, F_GETFL, 0);
		fcntl(csock, F_SETFL, sock_flags & O_NONBLOCK);
		if(csock<0) {
			return NULL;
		}
		sa.sun_family = AF_UNIX;
		strcpy(sa.sun_path, PIDFILEDIR "/ipcfgd.sock");
		connect(csock, (const struct sockaddr*)&sa, sizeof(sa));
	}
	write(csock, command, strlen(command));
	while(cont) {
		if((read_bytes=read(csock, buf, 256))<0) {
			switch(errno) {
				case EAGAIN:
#if EAGAIN != EWOULDBLOCK
				case EWOULDBLOCK:
#endif
					cont=false;
					break;
				case EINTR:
					break;
				default:
					return NULL;
			}
		}
		if(cont) {
			p=realloc(p, size+read_bytes);
			memcpy(p+size, buf, read_bytes);
			size+=read_bytes;
		}
	}
	return p;
}
