/** 
* @file quadtree.c
* @brief quadtree part
* @author serge guelton
* @date 2008-02-13
*/
/*
 * This file is part of hyantes.
 *
 * hyantes is free software; you can redistribute it and/or modify
 * it under the terms of the CeCILL-C License
 *
 * You should have received a copy of the CeCILL-C License
 * along with this program.  If not, see <http://www.cecill.info/licences>.
 */

#include "hs_compat.h"
#include "quadtree.h"

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <string.h>
#include <math.h>
#include <assert.h>


/** 
* @brief indicates the quadtree contains children
*/
#define QUADTREE_NOT_ALONE (-1.f)
/** 
* @brief indicates the quadtree does not contain any child
*/
#define QUADTREE_EMPTY (-2.f)

/* static */
static double is_alone(const QuadT tree, const Stock * stocks,
   size_t stocks_size, double *lat, double *lon)
/*@modifies lat@*/
/*@modifies lon@*/ ;

/*@dependent@*/ static QuadT create_quadtree();

#define STACK_SIZE 65536
#define STACK_HEIGHT 1024

static QuadT qstack[STACK_HEIGHT];
static int qstack_curr = (-1);
static size_t qstack_index = STACK_SIZE;

/** 
* @brief recursievly builds a quadtree from an array of stocks of given size
* 
* @param stocks array containg all read potential, sorted
* @param stocks_size  size of this array
* @param tree filled tree
* 
* @return 
*   1 if a son was built
*   0 otherwise
*/
int build_quadtree(const Stock * stocks, size_t stocks_size, QuadT tree)
{
    QuadT fbg = NULL;
    QuadT fbd = NULL;
    QuadT fhg = NULL;
    QuadT fhd = NULL;
    double lat = 0.f, lon = 0.f;
    double res;
    assert(stocks != NULL);
    assert(tree != NULL);
    assert(tree->fbd == NULL);
    assert(tree->fbg == NULL);
    assert(tree->fhd == NULL);
    assert(tree->fhg == NULL);

    res = is_alone(tree, stocks, stocks_size, &lat, &lon);

    /* node is not alone */
    if(fabs(res - QUADTREE_NOT_ALONE) < FLT_EPSILON)
    {

        /* initialize tree size */
        tree->size = 1;

        /* create children */
        fbd = create_quadtree();

        fbd->coords.mLat = (tree->coords.mLat + tree->coords.MLat) / 2;
        fbd->coords.mLon = tree->coords.mLon;
        fbd->coords.MLat = tree->coords.MLat;
        fbd->coords.MLon = (tree->coords.mLon + tree->coords.MLon) / 2;

        if(build_quadtree(stocks, stocks_size, fbd) == 1)
        {
            tree->fbd = fbd;
            tree->size += tree->fbd->size;
            tree->value += tree->fbd->value;
        }
        else
        {
            /*free_quadtree(fbd); */
            --qstack_index;
            memset(fbd, 0, sizeof(*fbd));
        }

        fbg = create_quadtree();

        fbg->coords.mLat = tree->coords.mLat;
        fbg->coords.mLon = tree->coords.mLon;
        fbg->coords.MLat = (tree->coords.mLat + tree->coords.MLat) / 2;
        fbg->coords.MLon = (tree->coords.mLon + tree->coords.MLon) / 2;

        if(build_quadtree(stocks, stocks_size, fbg) == 1)
        {
            tree->fbg = fbg;
            tree->size += fbg->size;
            tree->value += fbg->value;
        }
        else
        {
            /*free_quadtree(fbg); */
            --qstack_index;
            memset(fbg, 0, sizeof(*fbg));
        }

        fhg = create_quadtree();

        fhg->coords.mLat = tree->coords.mLat;
        fhg->coords.mLon = (tree->coords.mLon + tree->coords.MLon) / 2;
        fhg->coords.MLat = (tree->coords.mLat + tree->coords.MLat) / 2;
        fhg->coords.MLon = tree->coords.MLon;

        if(build_quadtree(stocks, stocks_size, fhg) == 1)
        {
            tree->fhg = fhg;
            tree->size += fhg->size;
            tree->value += fhg->value;
        }
        else
        {
            /*free_quadtree(fhg); */
            --qstack_index;
            memset(fhg, 0, sizeof(*fhg));
        }

        fhd = create_quadtree();

        fhd->coords.mLat = (tree->coords.mLat + tree->coords.MLat) / 2;
        fhd->coords.mLon = (tree->coords.mLon + tree->coords.MLon) / 2;
        fhd->coords.MLat = tree->coords.MLat;
        fhd->coords.MLon = tree->coords.MLon;

        if(build_quadtree(stocks, stocks_size, fhd) == 1)
        {
            tree->fhd = fhd;
            tree->size += fhd->size;
            tree->value += fhd->value;
        }
        else
        {
            /*free_quadtree(fhd); */
            --qstack_index;
            memset(fhd, 0, sizeof(*fhd));
        }

        return 1;
    }

    /* node is alone */
    else if(fabs(res - QUADTREE_EMPTY) < FLT_EPSILON)
    {
        return 0;
    }
    else
    {
        tree->value = res;
        tree->size = 1;
        tree->coords.mLat = tree->coords.MLat = lat;
        tree->coords.mLon = tree->coords.MLon = lon;
        return 1;
    }
}


/*****************************************************************************/


/** 
* @brief checks if a son exists for given quadtree, and fills info if a son is found
* 
* @param tree tree to check a son for
* @param stocks sorted array of all potential read
* @param stocks_size size of this array
* @param lat pointer to be filled by the latitude of the first son, if any
* @param lon  pointer to be filled by the longitude of the first son, if any
* 
* @return
*   QUADTREE_NOT_ALONE if several sons were found
*   QUADTREE_EMPTY if no son was found
*   potential of the son's stock if one son was found
*/
static double is_alone(const QuadT tree, const Stock * stocks,
   size_t stocks_size, double *lat, double *lon)
{

    size_t i;
    size_t count_valid;
    double res;
    size_t debut0 = 0;
    size_t fin0 = stocks_size - 1;
    size_t debut1 = 0;
    size_t fin1 = stocks_size - 1;
    size_t milieu0 = (debut0 + fin0) / 2;
    size_t milieu1 = (1 + debut1 + fin1) / 2;
    assert(stocks_size > 0);
    assert(stocks != NULL);

    for(i = 0; i < 20; i++)
    {
        if(tree->coords.mLat - stocks[milieu0].lat > FLT_EPSILON)
        {
            debut0 = milieu0;
            milieu0 = (debut0 + fin0) / 2;
        }
        else
        {
            fin0 = milieu0;
            milieu0 = (debut0 + fin0) / 2;
        }
    }


    for(i = 0; i < 20; i++)
    {
        if(stocks[milieu1].lat - tree->coords.MLat > FLT_EPSILON)
        {
            fin1 = milieu1;
            milieu1 = (1 + debut1 + fin1) / 2;
        }
        else
        {
            debut1 = milieu1;
            milieu1 = (1 + debut1 + fin1) / 2;
        }
    }


    count_valid = 0;
    res = 0.f;
    for(i = fin0; i <= debut1; i++)
    {
        if((stocks[i].lon - tree->coords.mLon > FLT_EPSILON
              || fabs(stocks[i].lon - tree->coords.mLon) < FLT_EPSILON)
           && (tree->coords.MLon - stocks[i].lon > FLT_EPSILON
              || fabs(stocks[i].lon - tree->coords.MLon) < FLT_EPSILON))
        {
            if(count_valid++ > 0)
                return QUADTREE_NOT_ALONE;
            *lat = stocks[i].lat;
            *lon = stocks[i].lon;
            res = stocks[i].pot;
        }
    }
    return (count_valid == 0) ? QUADTREE_EMPTY : res;

}
/** 
* @brief allocates new Quad tree and sets fields to zero
* 
* @return allocated quadtree, or NULL if an error occured
*/
/*@-unqualifiedtrans@*/
QuadT create_quadtree()
{
    if(qstack_index >= STACK_SIZE)
    {
        ++qstack_curr;
        if(qstack_curr >= STACK_HEIGHT)
        {
            fprintf(stderr,
               "[qalloc::] overflow of quadtree stack size\n");
            exit(EXIT_FAILURE);
        }
        qstack_index = 0;
        qstack[qstack_curr] = (QuadT) calloc(STACK_SIZE, sizeof(**qstack));
        if(qstack[qstack_curr] == NULL)
        {
            perror("[qalloc::calloc] ");
            exit(EXIT_FAILURE);
        }
    }
#ifndef NDEBUG
    {
        QuadT tree = qstack[qstack_curr] + (qstack_index);
        assert(tree != NULL);
        assert(tree->fbd == NULL);
        assert(tree->fbg == NULL);
        assert(tree->fhd == NULL);
        assert(tree->fhg == NULL);
    }
#endif
    return qstack[qstack_curr] + (qstack_index++);
}

/** 
* @brief free a quadtree
* 
* @return freed quadtree. Note that the tree itself is not deallocated
*/
void free_quadtree()
{
    while(qstack_curr >= 0)
    {
        free(qstack[qstack_curr--]);

    }
}
