/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-searchbar.h"

enum {
    SEARCH_PINNED,
    LAST_SIGNAL
};

G_DEFINE_TYPE (HrnSearchBar, hrn_search_bar, NBTK_TYPE_ENTRY);

#define HRN_SEARCH_BAR_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_SEARCH_BAR, \
                                HrnSearchBarPrivate))

struct _HrnSearchBarPrivate
{
  ClutterActor *pin_button;
  guint32 pin_id;

  ClutterActor *clear_button;
  HrnPinManager *pin_manager;

  char *current_pin;
};

static GObject * hrn_search_bar_constructor (GType type, guint n_params,
                                    GObjectConstructParam *params);
static void hrn_search_bar_dispose (GObject               *object);
static void hrn_searchbar_allocate (ClutterActor          *self,
                                    const ClutterActorBox *box,
                                    ClutterAllocationFlags flags);
static void hrn_searchbar_paint    (ClutterActor          *actor);
static void hrn_searchbar_pick     (ClutterActor          *actor,
                                    const ClutterColor    *color);

static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_searchbar_map (ClutterActor *self)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->map (self);

  if (priv->pin_button)
    clutter_actor_map (priv->pin_button);
  if (priv->clear_button)
    clutter_actor_map (priv->clear_button);
}


static void
hrn_searchbar_unmap (ClutterActor *self)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->unmap (self);

  if (priv->pin_button)
    clutter_actor_unmap (priv->pin_button);
  if (priv->clear_button)
    clutter_actor_unmap (priv->clear_button);
}

static gboolean
hrn_search_bar_style_changed (NbtkWidget *widget)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR (widget)->priv;

  g_signal_emit_by_name (priv->pin_button, "style-changed", 0);
  g_signal_emit_by_name (priv->clear_button, "style-changed", 0);

  return FALSE;
}

static void
hrn_search_bar_class_init (HrnSearchBarClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose      = hrn_search_bar_dispose;
  gobject_class->constructor  = hrn_search_bar_constructor;
  actor_class->paint          = hrn_searchbar_paint;
  actor_class->pick           = hrn_searchbar_pick;
  actor_class->map            = hrn_searchbar_map;
  actor_class->unmap          = hrn_searchbar_unmap;
  actor_class->allocate       = hrn_searchbar_allocate;

  g_type_class_add_private (gobject_class, sizeof (HrnSearchBarPrivate));
  signals[SEARCH_PINNED] = g_signal_new ("search-pinned",
                                         G_TYPE_FROM_CLASS (klass),
                                         G_SIGNAL_RUN_FIRST |
                                         G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                         g_cclosure_marshal_VOID__BOOLEAN,
                                         G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
}

static void
hrn_search_bar_init (HrnSearchBar *self)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (self);

  self->priv = priv;
  memset (priv, 0, sizeof (priv));
  g_signal_connect (self, "style-changed", G_CALLBACK (hrn_search_bar_style_changed), NULL);
}

static void
cleared (NbtkButton *togglebutton, HrnSearchBar *search_bar)
{
  nbtk_entry_set_text (NBTK_ENTRY (search_bar), "");
}

static void
pin_toggled (NbtkButton   *togglebutton,
             HrnSearchBar *search_bar)
{
    const char *search;

    if (nbtk_button_get_checked (togglebutton) == FALSE) {
        g_signal_emit (search_bar, signals[SEARCH_PINNED], 0, FALSE);
        return;
    }

    search = nbtk_entry_get_text (NBTK_ENTRY (search_bar));
    if (search == NULL || search[0] == '\0') {
        g_print ("Unpinning here\n");
        nbtk_button_set_checked (togglebutton, FALSE);
        return;
    }

    g_signal_emit (search_bar, signals[SEARCH_PINNED], 0, TRUE);
}

void
hrn_search_bar_set_pinned (HrnSearchBar *searchbar,
                           gboolean      pinned)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (searchbar);

  if (pinned) {
      nbtk_button_set_checked (NBTK_BUTTON (priv->pin_button), TRUE);
  } else {
      if (nbtk_button_get_checked (NBTK_BUTTON (priv->pin_button))) {
          nbtk_button_set_checked (NBTK_BUTTON (priv->pin_button), FALSE);
    }
  }
}


static GObject *
hrn_search_bar_constructor (GType type, guint n_params,
                            GObjectConstructParam *params)
{
  GObject             *object;
  HrnSearchBar        *search_bar;
  HrnSearchBarPrivate *priv;

  object = G_OBJECT_CLASS (hrn_search_bar_parent_class)->constructor (
    type, n_params, params);

  search_bar         = HRN_SEARCH_BAR (object);
  priv               = HRN_SEARCH_BAR_GET_PRIVATE (object);
  priv->pin_button   = (void*) nbtk_button_new ();
  priv->clear_button = (void*) nbtk_button_new ();

  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->pin_button), TRUE);
  clutter_actor_set_name (CLUTTER_ACTOR (priv->pin_button),
                          "fav-toggle");
  clutter_actor_set_name (CLUTTER_ACTOR (priv->clear_button),
                          "clear-query");

  clutter_actor_set_parent (CLUTTER_ACTOR (priv->pin_button),
                            CLUTTER_ACTOR (object));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->clear_button),
                            CLUTTER_ACTOR (object));

  priv->pin_id = g_signal_connect (priv->pin_button, "clicked",
                                   G_CALLBACK (pin_toggled), object);
  g_signal_connect (priv->clear_button, "clicked", G_CALLBACK (cleared), object);

  clutter_actor_set_size (priv->pin_button, 32, 32);
  clutter_actor_set_size (priv->clear_button, 32, 32);
  clutter_actor_set_anchor_point_from_gravity (priv->pin_button,
                                               CLUTTER_GRAVITY_NORTH_EAST);
  clutter_actor_set_anchor_point_from_gravity (priv->clear_button,
                                               CLUTTER_GRAVITY_NORTH_EAST);

  return object;
}

static void
hrn_search_bar_dispose (GObject *object)
{
  /* XXX: dispose of children */
  G_OBJECT_CLASS (hrn_search_bar_parent_class)->dispose (object);
}


static void
hrn_searchbar_allocate (ClutterActor *self, const ClutterActorBox *box,
                        ClutterAllocationFlags flags)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (self);

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->allocate (self, box, flags);

  clutter_actor_set_x (CLUTTER_ACTOR (priv->pin_button),
                       clutter_actor_get_width (self));
  clutter_actor_set_x (CLUTTER_ACTOR (priv->clear_button),
                       clutter_actor_get_x (priv->pin_button) -
                       20);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (
                                           priv->pin_button), flags);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (
                                           priv->clear_button), flags);
}

static void
hrn_searchbar_paint (ClutterActor *actor)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (actor);

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->paint (actor);
  clutter_actor_paint (CLUTTER_ACTOR (priv->pin_button));
  clutter_actor_paint (CLUTTER_ACTOR (priv->clear_button));
}

static void
hrn_searchbar_pick (ClutterActor *actor, const ClutterColor *color)
{
  hrn_searchbar_paint (actor);
}

HrnSearchBar *
hrn_searchbar_new (HrnPinManager *manager)
{
  HrnSearchBar *bar;
  HrnSearchBarPrivate *priv;

  bar = g_object_new (HRN_TYPE_SEARCH_BAR, NULL);
  priv = bar->priv;

  priv->pin_manager = manager;

  return bar;
}

static void
pin_selected_cb (HrnPinManager *pin_manager,
                 HrnPin        *pin,
                 HrnSearchBar  *searchbar)
{
  HrnSearchBarPrivate *priv = searchbar->priv;

  g_free (priv->current_pin);

  if (pin) {
    priv->current_pin = g_strdup (pin->group);
    nbtk_entry_set_text ((NbtkEntry *) searchbar, pin->query);
  } else {
    priv->current_pin = NULL;
    nbtk_entry_set_text ((NbtkEntry *) searchbar, "");
  }

  g_signal_handler_block (priv->pin_button, priv->pin_id);
  nbtk_button_set_checked (NBTK_BUTTON (priv->pin_button),
                           pin ? TRUE : FALSE);
  g_signal_handler_unblock (priv->pin_button, priv->pin_id);
}

void
hrn_searchbar_set_pin_manager (HrnSearchBar  *searchbar,
                               HrnPinManager *pin_manager)
{
  HrnSearchBarPrivate *priv = searchbar->priv;

  priv->pin_manager = pin_manager;
  g_signal_connect (pin_manager, "pin-selected",
                    G_CALLBACK (pin_selected_cb), searchbar);
}

void
hrn_searchbar_set_pin (HrnSearchBar *searchbar,
                       HrnPin       *pin)
{
  HrnSearchBarPrivate *priv = searchbar->priv;

  g_free (priv->current_pin);
  priv->current_pin = g_strdup (pin->group);
}
