/* -*- mode: C; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
#include <config.h>
#include <glib/gi18n-lib.h>
#include <hippo/hippo-canvas-context.h>
#include <hippo/hippo-canvas-style.h>
#include <hippo/hippo-canvas-theme.h>
#include <gtk/gtkcontainer.h>
#include "hippo-canvas-widget.h"
#include "hippo-canvas-helper.h"
#include <gtk/gtkprivate.h> /* for GTK_WIDGET_ALLOC_NEEDED */
#include <gtk/gtkwindow.h>
#include <gtk/gtklabel.h>

#include <gdk/gdkx.h>
#include <cairo-xlib.h>

/* Gap between the area we are tipping and the tooltip */
#define TOOLTIP_PADDING 4
/* Maximum width of a tippable area before we position the tooltip by the pointer */
#define MAX_TOOLTIP_AREA_WIDTH 400
/* Maximum height of a tippable area before we position the tooltip by the pointer */
#define MAX_TOOLTIP_AREA_HEIGHT 40
/* When positioning by the pointer, X/Y offset */
#define TOOLTIP_OFFSET 15

typedef struct
{
    HippoCanvasItem *item;
    GtkWidget       *widget;
} RegisteredWidgetItem;

static void hippo_canvas_helper_init       (HippoCanvasHelper       *helper);
static void hippo_canvas_helper_class_init (HippoCanvasHelperClass  *klass);
static void hippo_canvas_helper_dispose    (GObject                 *object);
static void hippo_canvas_helper_finalize   (GObject                 *object);
static void hippo_canvas_helper_iface_init (HippoCanvasContextIface *klass);


static void hippo_canvas_helper_set_property (GObject      *object,
                                              guint         prop_id,
                                              const GValue *value,
                                              GParamSpec   *pspec);
static void hippo_canvas_helper_get_property (GObject      *object,
                                              guint         prop_id,
                                              GValue       *value,
                                              GParamSpec   *pspec);

static PangoLayout*     hippo_canvas_helper_create_layout          (HippoCanvasContext *context);
static cairo_surface_t* hippo_canvas_helper_create_surface         (HippoCanvasContext *context,
                                                                    cairo_content_t     content,
                                                                    int                 width,
                                                                    int                 height);
static cairo_surface_t* hippo_canvas_helper_load_image             (HippoCanvasContext *context,
                                                                    const char         *image_name);
static guint32          hippo_canvas_helper_get_color              (HippoCanvasContext *context,
                                                                    HippoStockColor     color);
static void             hippo_canvas_helper_register_widget_item   (HippoCanvasContext *context,
                                                                    HippoCanvasItem    *item);
static void             hippo_canvas_helper_unregister_widget_item (HippoCanvasContext *context,
                                                                    HippoCanvasItem    *item);
static void             hippo_canvas_helper_translate_to_widget    (HippoCanvasContext *context,
                                                                    HippoCanvasItem    *item,
                                                                    int                *x_p,
                                                                    int                *y_p);
static void             hippo_canvas_helper_translate_to_screen    (HippoCanvasContext *context,
                                                                    HippoCanvasItem    *item,
                                                                    int                *x_p,
                                                                    int                *y_p);

HippoCanvasStyle *    hippo_canvas_helper_get_style      (HippoCanvasContext *context);
double                hippo_canvas_helper_get_resolution (HippoCanvasContext *context);
PangoFontDescription *hippo_canvas_helper_get_font       (HippoCanvasContext *context);

HippoAnimationManager *hippo_canvas_helper_get_animation_manager   (HippoCanvasContext *context);

static void             hippo_canvas_helper_fixup_resize_state     (HippoCanvasHelper  *canvas);

static void       on_animation_manager_after_frame (HippoAnimationManager *manager,
                                                    guint                  frame_serial,
                                                    HippoCanvasHelper     *helper);

static void       tooltip_window_update   (GtkWidget      *tip,
                                           GdkScreen      *screen,
                                           int             mouse_x,
                                           int             mouse_y,
                                           HippoRectangle *for_area,
                                           const char     *text);
static GtkWidget* tooltip_window_new      (void);



struct _HippoCanvasHelper {
    GObject parent;

    GtkWidget *widget;

    HippoCanvasTheme *theme;
    HippoCanvasStyle *style;
    
    HippoCanvasItem *root;

    HippoCanvasPointer pointer;

    GtkWidget *tooltip_window;

    int width;

    guint tooltip_timeout_id;
    int last_window_x;
    int last_window_y;

    int last_allocated_border;
    
    GSList *widget_items;

    HippoAnimationManager *animation_manager;
    guint frame_serial;

    unsigned int root_hovering : 1;
    unsigned int fixing_up_resize_state : 1;
    unsigned int need_background_paint : 1;
    unsigned int frame_pending : 1;
};

struct _HippoCanvasHelperClass {
    GObjectClass parent_class;
};

enum {
    NO_SIGNALS_YET,
    LAST_SIGNAL
};

/* static int signals[LAST_SIGNAL]; */

enum {
    PROP_0
};

G_DEFINE_TYPE_WITH_CODE(HippoCanvasHelper, hippo_canvas_helper, G_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(HIPPO_TYPE_CANVAS_CONTEXT,
                                              hippo_canvas_helper_iface_init));

static void
hippo_canvas_helper_init(HippoCanvasHelper *helper)
{
    helper->width = -1;
    helper->pointer = HIPPO_CANVAS_POINTER_UNSET;
    helper->last_window_x = -1;
    helper->last_window_y = -1;

    helper->animation_manager = hippo_animation_manager_new();
    g_signal_connect(helper->animation_manager, "after-frame",
                     G_CALLBACK(on_animation_manager_after_frame), helper);
}

static void
hippo_canvas_helper_class_init(HippoCanvasHelperClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    
    object_class->set_property = hippo_canvas_helper_set_property;
    object_class->get_property = hippo_canvas_helper_get_property;

    object_class->dispose = hippo_canvas_helper_dispose;
    object_class->finalize = hippo_canvas_helper_finalize;
}

static void
hippo_canvas_helper_iface_init (HippoCanvasContextIface *klass)
{
    klass->create_layout = hippo_canvas_helper_create_layout;
    klass->create_surface = hippo_canvas_helper_create_surface;
    klass->load_image = hippo_canvas_helper_load_image;
    klass->get_color = hippo_canvas_helper_get_color;
    klass->register_widget_item = hippo_canvas_helper_register_widget_item;
    klass->unregister_widget_item = hippo_canvas_helper_unregister_widget_item;
    klass->translate_to_widget = hippo_canvas_helper_translate_to_widget;
    klass->translate_to_screen = hippo_canvas_helper_translate_to_screen;
    klass->get_style = hippo_canvas_helper_get_style;
    klass->get_resolution = hippo_canvas_helper_get_resolution;
    klass->get_font = hippo_canvas_helper_get_font;
    klass->get_animation_manager = hippo_canvas_helper_get_animation_manager;
}

static void
cancel_tooltip(HippoCanvasHelper *helper)
{
    if (helper->tooltip_timeout_id) {
        g_source_remove(helper->tooltip_timeout_id);
        helper->tooltip_timeout_id = 0;
    }
     
    if (helper->tooltip_window)
        gtk_widget_hide(helper->tooltip_window);
}

static void
hippo_canvas_helper_dispose(GObject *object)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(object);

    if (helper->root != NULL) {
        HippoCanvasItem *old_root = g_object_ref(helper->root);
        hippo_canvas_helper_set_root(helper, NULL);
        hippo_canvas_helper_set_theme(helper, NULL);
        hippo_canvas_item_destroy(old_root);
        g_object_unref(old_root);
    }

    g_assert(helper->widget_items == NULL);

    cancel_tooltip(helper);
    if (helper->tooltip_window) {
        gtk_object_destroy(GTK_OBJECT(helper->tooltip_window));
        helper->tooltip_window = NULL;
    }

    if (helper->animation_manager) {
        g_signal_handlers_disconnect_by_func(helper->animation_manager,
                                             (gpointer)on_animation_manager_after_frame,
                                             helper);
        
        g_object_unref(helper->animation_manager);
        helper->animation_manager = NULL;
    }

    G_OBJECT_CLASS(hippo_canvas_helper_parent_class)->dispose(object);
}

static void
hippo_canvas_helper_finalize(GObject *object)
{
    /* HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(object); */

    G_OBJECT_CLASS(hippo_canvas_helper_parent_class)->finalize(object);
}

static void
hippo_canvas_helper_set_property(GObject      *object,
                                 guint         prop_id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
#if 0
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(object);
#endif

    switch (prop_id) {
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
hippo_canvas_helper_get_property(GObject    *object,
                                 guint       prop_id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
#if 0
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(object);
#endif

    switch (prop_id) {
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

HippoCanvasHelper*
hippo_canvas_helper_new(GtkContainer *base_container)
{
    HippoCanvasHelper *helper;

    g_return_val_if_fail(GTK_IS_CONTAINER(base_container), NULL);

    helper = g_object_new(HIPPO_TYPE_CANVAS_HELPER, NULL);

    helper->widget = GTK_WIDGET(base_container);

    return helper;
}

static void
set_pointer(HippoCanvasHelper *helper,
            HippoCanvasPointer pointer)
{
    GdkCursor *cursor;
    GtkWidget *widget;
    GdkWindow *event_window;
    
    /* important optimization since we do this on all motion notify */
    if (helper->pointer == pointer)
        return;

    widget = helper->widget;

    helper->pointer = pointer;

    if (pointer == HIPPO_CANVAS_POINTER_UNSET ||
        pointer == HIPPO_CANVAS_POINTER_DEFAULT)
        cursor = NULL;
    else {
        GdkCursorType type = GDK_X_CURSOR;
        switch (pointer) {
        case HIPPO_CANVAS_POINTER_HAND:
            type = GDK_HAND2;
            break;
        case HIPPO_CANVAS_POINTER_UNSET:
        case HIPPO_CANVAS_POINTER_DEFAULT:
            g_assert_not_reached();
            break;
            /* don't add a default, breaks compiler warnings */
        }
        cursor = gdk_cursor_new_for_display(gtk_widget_get_display(widget),
                                            type);
    }

    event_window = widget->window;
    gdk_window_set_cursor(event_window, cursor);
    
    gdk_display_flush(gtk_widget_get_display(widget));

    if (cursor != NULL)
        gdk_cursor_unref(cursor);
}

static void
get_root_item_window_coords(HippoCanvasHelper *helper,
                            int               *x_p,
                            int               *y_p)
{
    GtkWidget *widget = helper->widget;

    if (x_p)
        *x_p = GTK_CONTAINER(widget)->border_width;
    if (y_p)
        *y_p = GTK_CONTAINER(widget)->border_width;
    
    if (GTK_WIDGET_NO_WINDOW(widget)) {
        if (x_p)
            *x_p += widget->allocation.x;
        if (y_p)
            *y_p += widget->allocation.y;
    }
}

static void
update_tooltip(HippoCanvasHelper *helper,
               gboolean           show_if_not_already)
{
    char *tip;
    HippoRectangle for_area;
    GtkWidget *toplevel;
    int mouse_x;
    int mouse_y;

    if ((helper->tooltip_window == NULL || !GTK_WIDGET_VISIBLE(helper->tooltip_window)) &&
        !show_if_not_already)
        return;
    
    toplevel = gtk_widget_get_ancestor(helper->widget,
                                       GTK_TYPE_WINDOW);

    tip = NULL;
    if (helper->root != NULL &&
        toplevel && GTK_WIDGET_VISIBLE(toplevel) &&
        GTK_WIDGET_VISIBLE(helper->widget)) {
        int window_x, window_y;
        get_root_item_window_coords(helper, &window_x, &window_y);
        mouse_x = helper->last_window_x - window_x;
        mouse_y = helper->last_window_y - window_y;
        tip = hippo_canvas_item_get_tooltip(helper->root,
                                            mouse_x, mouse_y,
                                            &for_area);
        for_area.x += window_x;
        for_area.y += window_y;
    }

    if (tip != NULL) {
        int screen_x, screen_y;
        
        if (helper->tooltip_window == NULL) {
            helper->tooltip_window = tooltip_window_new();
        }

        gdk_window_get_origin(helper->widget->window, &screen_x, &screen_y);

        for_area.x += screen_x;
        for_area.y += screen_y;
        mouse_x += screen_x;
        mouse_y += screen_y;

        tooltip_window_update(helper->tooltip_window,
                              gtk_widget_get_screen(helper->widget),
                              mouse_x, mouse_y,
                              &for_area,
                              tip);

        gtk_widget_show(helper->tooltip_window);
        
        g_free(tip);
    }
}

static gboolean
allocate_pending_on_widget(GtkWidget *widget)
{
    return GTK_WIDGET_ALLOC_NEEDED(widget);
}

static gboolean
expose_pending_on_window(GdkWindow *window)
{
    /* This will not work properly for the OS X port of GTK+ */

    return ((GdkWindowObject *)window)->update_area != NULL;
}

static void
on_animation_manager_after_frame (HippoAnimationManager *manager,
                                  guint                  frame_serial,
                                  HippoCanvasHelper     *helper)
{
    /* A frame has finished; we now need to wait until all resizing
     * and redrawing and then call
     * hippo_animation_manager_frame_complete().
     */
    
    if (!allocate_pending_on_widget(helper->widget) &&
        !expose_pending_on_window(helper->widget->window))
    {
        /* Nothing to do, we can call frame_complete() immediately */
        hippo_animation_manager_frame_complete(manager, frame_serial);
    }
    else
    {
        helper->frame_serial = frame_serial;
        helper->frame_pending = TRUE;
    }
}

gboolean
hippo_canvas_helper_expose_event(HippoCanvasHelper *helper,
                                 GdkEventExpose    *event)
{
    cairo_t *cr;

    cr = gdk_cairo_create(event->window);

    if (helper->need_background_paint) {
        HippoCanvasStyle *style = hippo_canvas_context_get_style(HIPPO_CANVAS_CONTEXT(helper));
        guint32 color = hippo_canvas_style_get_background_color(style);

        cairo_save(cr);
        cairo_set_source_rgba(cr,
                              ((color & 0xFF000000) >> 24) / 255.,
                              ((color & 0x00FF0000) >> 16) / 255.,
                              ((color & 0x0000FF00) >> 8) / 255.,
                              ((color & 0x000000FF)) / 255.);
        cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
        cairo_paint(cr);
        cairo_restore(cr);
    }

    if (helper->root != NULL) {
        int window_x, window_y;
        HippoRectangle damage_box;
    
        get_root_item_window_coords(helper, &window_x, &window_y);

        damage_box.x = event->area.x;
        damage_box.y = event->area.y;
        damage_box.width = event->area.width;
        damage_box.height = event->area.height;
        hippo_canvas_item_process_paint(helper->root, cr, &damage_box,
                                        window_x, window_y);
    }
    
    cairo_destroy(cr);

    if (helper->frame_pending) {
        /* Resize occurs before redraw, so when we are done redrawing, we know we are
         * done with the frame
         */

        /* In theory, we should actually wait for the frame to paint on the X server;
         * we could do this by changing a property and watching for the PropertyNotify
         * event or by using the XSync extension. Even better would be to communicate
         * with the compositing manager and wait until a frame containing the new
         * contents of our window has been drawn to the front buffer.
         *
         * In practice, the difference between that and just going ahead and starting
         * the next frame is going to be pretty small unless the balance
         * of server vs. client side efficiency changes a lot for cairo... the client
         * side of cairo currently doesn't keep up with the server.
         */
        helper->frame_pending = FALSE;
        hippo_animation_manager_frame_complete(helper->animation_manager, helper->frame_serial);
    }
    
    return FALSE;
}

void
hippo_canvas_helper_size_request(HippoCanvasHelper *helper,
                                 GtkRequisition    *requisition)
{
    /* g_debug("gtk request on canvas root %p canvas %p", helper->root, canvas); */

    hippo_canvas_helper_fixup_resize_state(helper);
    
    requisition->width = 0;
    requisition->height = 0;

    if (helper->root != NULL) {
        int min_width, min_height;
        hippo_canvas_item_get_width_request(helper->root, &min_width, NULL);
        hippo_canvas_item_get_height_request(helper->root, MAX(helper->width, min_width), &min_height, NULL);
        requisition->width = min_width;
        requisition->height = min_height;
    }

    requisition->width += GTK_CONTAINER(helper->widget)->border_width * 2;
    requisition->height += GTK_CONTAINER(helper->widget)->border_width * 2;
}

void
hippo_canvas_helper_size_allocate(HippoCanvasHelper *helper,
                                  GtkAllocation     *allocation)
{
    /* g_debug("gtk allocate on canvas root %p canvas %p", helper->root, canvas); */

    if (helper->root != NULL) {
        int border_width = GTK_CONTAINER(helper->widget)->border_width;
        int child_width = allocation->width - border_width * 2;
        int child_height = allocation->height - border_width * 2;
        gboolean border_changed;
        
        border_changed = border_width = helper->last_allocated_border;
        helper->last_allocated_border = border_width;

        if (border_changed)
            gtk_widget_queue_draw(helper->widget);
        
        hippo_canvas_item_allocate(helper->root,
                                   child_width, child_height,
                                   border_changed);

        /* Tooltip might be in the wrong place now */
        update_tooltip(helper, FALSE);
    }

    if (helper->frame_pending) {
        if (!helper->widget->window || !expose_pending_on_window(helper->widget->window)) {
            /* We resized, and there was nothing to draw, we can can call frame_complete() */
            
            helper->frame_pending = FALSE;
            hippo_animation_manager_frame_complete(helper->animation_manager, helper->frame_serial);
        }
    }
}

gboolean
hippo_canvas_helper_button_press(HippoCanvasHelper *helper,
                                 GdkEventButton    *event)
{
    int window_x, window_y;
    int count;
    
    if (helper->root == NULL)
        return FALSE;

    get_root_item_window_coords(helper, &window_x, &window_y);
    
    /*
    g_debug("canvas button press at %d,%d allocation %d,%d", (int) event->x, (int) event->y,
            widget->allocation.x, widget->allocation.y);
    */
    count = 1;
    if (event->type == GDK_2BUTTON_PRESS)
    	count = 2;
    else if (event->type == GDK_3BUTTON_PRESS)
    	count = 3;
    
    hippo_canvas_item_emit_button_press_event(helper->root,
                                              event->x - window_x, event->y - window_y,
                                              event->button,
                                              event->x_root, event->y_root,
                                              event->time,
                                              count);

    return TRUE;
}

gboolean
hippo_canvas_helper_button_release(HippoCanvasHelper *helper,
                                   GdkEventButton    *event)
{
    int window_x, window_y;
    
    if (helper->root == NULL)
        return FALSE;

    get_root_item_window_coords(helper, &window_x, &window_y);
    
    /*
    g_debug("canvas button release at %d,%d allocation %d,%d", (int) event->x, (int) event->y,
            widget->allocation.x, widget->allocation.y);
    */
    
    hippo_canvas_item_emit_button_release_event(helper->root,
                                                event->x - window_x, event->y - window_y,
                                                event->button,
                                                event->x_root, event->y_root,
                                                event->time);

    return TRUE;
}

static gboolean
tooltip_timeout(void *data)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(data);

    update_tooltip(helper, TRUE);
    
    helper->tooltip_timeout_id = 0;
    return FALSE;
}

#define TIP_DELAY (1000*1.5)

static void
add_tooltip_timeout(HippoCanvasHelper *helper,
                    int                delay)
{
    if (helper->tooltip_timeout_id != 0)
        g_source_remove(helper->tooltip_timeout_id);
    
    helper->tooltip_timeout_id = g_timeout_add(delay, tooltip_timeout, helper);
}

static void
handle_new_mouse_location(HippoCanvasHelper *helper,
                          GdkWindow         *event_window,
                          HippoMotionDetail  detail)
{
    int mouse_x, mouse_y;
    int root_x_origin, root_y_origin;
    int root_x, root_y;
    int w, h;
    gboolean was_hovering;

    if (event_window != helper->widget->window)
        return;
    
    gdk_window_get_pointer(event_window, &mouse_x, &mouse_y, NULL);

    if (detail == HIPPO_MOTION_DETAIL_LEAVE) {
        cancel_tooltip(helper);       
    } else if (mouse_x != helper->last_window_x || mouse_y != helper->last_window_y) {
        cancel_tooltip(helper);       
        helper->last_window_x = mouse_x;
        helper->last_window_y = mouse_y;
        add_tooltip_timeout(helper, TIP_DELAY);
    }

    get_root_item_window_coords(helper, &root_x_origin, &root_y_origin);
    root_x = mouse_x - root_x_origin;
    root_y = mouse_y - root_y_origin;
    
    hippo_canvas_item_get_allocation(helper->root, &w, &h);

#if 0
    g_debug("%p mouse %d,%d root origin %d,%d root %d,%d root size %dx%d", helper->root,
            mouse_x, mouse_y, root_x_origin, root_y_origin, root_x, root_y, w, h);
#endif

    was_hovering = helper->root_hovering;

    if (detail == HIPPO_MOTION_DETAIL_LEAVE)
        helper->root_hovering = FALSE;
    else
        helper->root_hovering = TRUE;

    /* g_debug("   was_hovering %d root_hovering %d", was_hovering, helper->root_hovering); */
    
    if (was_hovering && !helper->root_hovering) {
        set_pointer(helper, HIPPO_CANVAS_POINTER_UNSET);
        hippo_canvas_item_emit_motion_notify_event(helper->root, root_x, root_y,
                                                   HIPPO_MOTION_DETAIL_LEAVE);
    } else {
        HippoCanvasPointer pointer;
        
        pointer = hippo_canvas_item_get_pointer(helper->root, root_x, root_y);
        set_pointer(helper, pointer);
    
        if (helper->root_hovering && !was_hovering) {
            hippo_canvas_item_emit_motion_notify_event(helper->root, root_x, root_y,
                                                       HIPPO_MOTION_DETAIL_ENTER);
        } else if (helper->root_hovering) {
            hippo_canvas_item_emit_motion_notify_event(helper->root, root_x, root_y,
                                                       HIPPO_MOTION_DETAIL_WITHIN);
        }
    }
}

gboolean
hippo_canvas_helper_enter_notify(HippoCanvasHelper *helper,
                                 GdkEventCrossing  *event)
{
    HippoMotionDetail detail;

    /* g_debug("motion notify GDK ENTER on %p root %p root_hovering %d", widget, helper->root, helper->root_hovering); */
    
    if (helper->root == NULL)
        return FALSE;

    if (event->detail == GDK_NOTIFY_INFERIOR || event->window != helper->widget->window)
        detail = HIPPO_MOTION_DETAIL_WITHIN;
    else
        detail = HIPPO_MOTION_DETAIL_ENTER;
        
    handle_new_mouse_location(helper, event->window, detail);
    
    return FALSE;
}

gboolean
hippo_canvas_helper_leave_notify(HippoCanvasHelper *helper,
                                 GdkEventCrossing  *event)
{
    HippoMotionDetail detail;

    /* g_debug("motion notify GDK LEAVE on %p root %p root_hovering %d", widget, helper->root, helper->root_hovering); */
    
    if (helper->root == NULL)
        return FALSE;

    if (event->detail == GDK_NOTIFY_INFERIOR || event->window != helper->widget->window)
        detail = HIPPO_MOTION_DETAIL_WITHIN;
    else
        detail = HIPPO_MOTION_DETAIL_LEAVE;
        
    handle_new_mouse_location(helper, event->window, detail);
    
    return FALSE;
}

gboolean
hippo_canvas_helper_motion_notify(HippoCanvasHelper *helper,
                                  GdkEventMotion    *event)
{
    /* g_debug("motion notify GDK MOTION on %p root %p root_hovering %d", widget, helper->root, helper->root_hovering); */
    
    if (helper->root == NULL)
        return FALSE;

    handle_new_mouse_location(helper, event->window, HIPPO_MOTION_DETAIL_WITHIN);
    
    return FALSE;
}

gboolean
hippo_canvas_helper_scroll (HippoCanvasHelper *helper,
                            GdkEventScroll    *event)
{
    int window_x, window_y;    
    
    if (helper->root == NULL)
        return FALSE;

    get_root_item_window_coords(helper, &window_x, &window_y);
    
    g_assert(GDK_SCROLL_UP == HIPPO_SCROLL_UP);
    g_assert(GDK_SCROLL_DOWN == HIPPO_SCROLL_DOWN);
    g_assert(GDK_SCROLL_LEFT == HIPPO_SCROLL_LEFT);
    g_assert(GDK_SCROLL_RIGHT == HIPPO_SCROLL_RIGHT);
    
    hippo_canvas_item_emit_scroll_event(helper->root,
                                        event->x - window_x, event->y - window_y,
                                        event->direction);
    
    return FALSE;
}

void
hippo_canvas_helper_realize(HippoCanvasHelper *helper)
{
}

void
hippo_canvas_helper_unmap(HippoCanvasHelper *helper)
{
    /* This is actually unnecessary and not useful, though harmless ... we don't 
     * reliably get an unmap if some ancestor is hidden, but we do reliably
     * get a leave event, which we handle elsewhere.
     */
    cancel_tooltip(helper);
}

void
hippo_canvas_helper_hierarchy_changed (HippoCanvasHelper *helper,
                                       GtkWidget         *old_toplevel)
{
    cancel_tooltip(helper);
}

static int
premultiply(int component,
            int alpha)
{
    int temp = component * alpha + 0x80;
    return ((temp >> 8) + temp) >> 8;
}

void
hippo_canvas_helper_set_window_background (HippoCanvasHelper *helper,
                                           GdkWindow         *window)
{
    HippoCanvasStyle *style = hippo_canvas_context_get_style(HIPPO_CANVAS_CONTEXT(helper));
    guint32 color;
    
    helper->need_background_paint = FALSE;

    /* We don't use hippo_canvas_style_get_background_color() here because we treat
     * background-color: transparent on the canvas as being different from
     * not specifying it ... which should give us the theme color. Perhaps
     * a cleaner thing to do would be to support the CSS2 system colors
     * and have a default stylesheet with canvas { background-color: Window; }
     */
    if (hippo_canvas_style_get_color(style, "background-color", FALSE, &color)) {
        GdkColormap *colormap = gdk_window_get_colormap(window);
        GdkVisual *visual = gdk_colormap_get_visual(colormap);
        GdkColor color_gdk;

        /* Special-case the RGBA visual. If visuals other than ARGB32 were in use
         * (say ABGR32) then we should support them here as well.
         */
        if (visual->depth == 32 &&
            visual->red_mask == 0xff0000 &&
            visual->green_mask == 0x00ff00 &&
            visual->blue_mask  == 0x0000ff)
        {
            int alpha = color & 0x000000FF;

            if (alpha != 0xFF)
                /* GTK+ (as of 2.12) doesn't handle clearing to a RGBA color correctly
                 * so we'll need to fix up the double-buffer pixmap ourself.
                 */
                helper->need_background_paint = TRUE;

            color_gdk.pixel = ((alpha << 24) |
                               (premultiply((color & 0xFF000000) >> 24, alpha) << 16) |
                               (premultiply((color & 0x00FF0000) >> 16, alpha) << 8) |
                               (premultiply((color & 0x0000FF00) >> 8,  alpha)));
	}
        else
        {
            color_gdk.red =   (color & 0xFF000000) >> 24;
            color_gdk.red = color_gdk.red * 0x101;
            color_gdk.green = (color & 0x00FF0000) >> 16;
            color_gdk.green = color_gdk.green * 0x101;
            color_gdk.blue =  (color & 0x0000FF00) >> 8;
            color_gdk.blue = color_gdk.blue * 0x101;
            
            gdk_rgb_find_color(colormap, &color_gdk);
        }

        gdk_window_set_background(window, &color_gdk);
    } else {
        gtk_style_set_background (helper->widget->style, window, GTK_STATE_NORMAL);
    }
}

void
hippo_canvas_helper_add(HippoCanvasHelper *helper,
                        GtkWidget         *widget)
{
    g_warning("hippo_canvas_add called, you have to just add an item with a widget in it, you can't do gtk_container_add directly");
}

void
hippo_canvas_helper_remove(HippoCanvasHelper *helper,
                           GtkWidget         *widget)
{
    GSList *link;

    /* We go a little roundabout here - we remove the widget from the canvas
     * item, which causes us to remove it from ourselves.
     * The only time we expect gtk_container_remove to be called is from
     * gtk_object_destroy on e.g. the toplevel window, or something of
     * that nature.
     */
    
    for (link = helper->widget_items;
         link != NULL;
         link = link->next) {
        RegisteredWidgetItem *witem = link->data;

        if (witem->widget == widget) {
            g_object_set(G_OBJECT(witem->item), "widget", NULL, NULL);
            return;
        }
    }

    g_warning("tried to remove widget %p that is not in the canvas", widget);
}

void
hippo_canvas_helper_forall(HippoCanvasHelper *helper,
                           gboolean           include_internals,
                           GtkCallback        callback,
                           gpointer           callback_data)
{
    GSList *link;
    
    for (link = helper->widget_items;
         link != NULL;
         link = link->next) {
        RegisteredWidgetItem *witem = link->data;

        if (witem->widget)
            (* callback) (witem->widget, callback_data);
    }
}

GType
hippo_canvas_helper_child_type(HippoCanvasHelper *helper)
{
    /* FIXME: this is wrong, since you can't call add() */
    return GTK_TYPE_WIDGET;
}

static PangoLayout*
hippo_canvas_helper_create_layout(HippoCanvasContext *context)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    return gtk_widget_create_pango_layout(helper->widget, NULL);
}

static cairo_surface_t*
hippo_canvas_helper_create_surface(HippoCanvasContext *context,
                                   cairo_content_t     content,
                                   int                 width,
                                   int                 height)
{
    /* This feels like a really round-about way of doing things, but
     * I can't think of a better way. There's no way to get a cairo
     * xlib surface without a visual other than create_similar(), and
     * no visual for CAIRO_FORMAT_ALPHA.
     */
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    GdkScreen *screen = gtk_widget_get_screen(helper->widget);
    GdkDisplay *display = gdk_screen_get_display(screen);
    GdkWindow *root = gdk_screen_get_root_window(screen);
    GdkVisual *visual = gdk_drawable_get_visual(root);
    cairo_surface_t *root_surface;
    cairo_surface_t *surface;

    root_surface = cairo_xlib_surface_create(GDK_DISPLAY_XDISPLAY(display),
                                             GDK_DRAWABLE_XID(root),
                                             GDK_VISUAL_XVISUAL(visual),
                                             gdk_screen_get_width(screen),
                                             gdk_screen_get_height(screen));
    surface = cairo_surface_create_similar(root_surface, content, width, height);
    cairo_surface_destroy(root_surface);

    return surface;
}

static HippoCanvasLoadImageHook hippo_canvas_helper_load_image_hook = NULL;

void
hippo_canvas_helper_set_load_image_hook(HippoCanvasLoadImageHook hook)
{
    hippo_canvas_helper_load_image_hook = hook;
}

static cairo_surface_t*
hippo_canvas_helper_load_image(HippoCanvasContext *context,
                               const char         *image_name)
{
    if (hippo_canvas_helper_load_image_hook) {
        return hippo_canvas_helper_load_image_hook(context, image_name);
    } else {
        return NULL;
    }
}

static guint32
convert_color(GdkColor *gdk_color)
{
    guint32 rgba;
    
    rgba = gdk_color->red / 256;
    rgba <<= 8;
    rgba |= gdk_color->green / 256;
    rgba <<= 8;
    rgba |= gdk_color->blue / 256;
    rgba <<= 8;
    rgba |= 0xff; /* alpha */

    return rgba;
}

static guint32
hippo_canvas_helper_get_color(HippoCanvasContext *context,
                              HippoStockColor     color)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    GtkWidget *widget = helper->widget;
    GtkStyle *style = gtk_widget_get_style(widget);

    if (style == NULL) /* not realized yet, should not happen really */
        return 0;
    
    switch (color) {
    case HIPPO_STOCK_COLOR_BG_NORMAL:
        return convert_color(&style->bg[GTK_STATE_NORMAL]);
        break;
    case HIPPO_STOCK_COLOR_BG_PRELIGHT:
        return convert_color(&style->bg[GTK_STATE_PRELIGHT]);
        break;
    case HIPPO_STOCK_COLOR_FG:
        return convert_color(&style->fg[GTK_STATE_NORMAL]);
        break;
    }

    g_warning("unknown stock color %d", color);
    return 0;
}

static void
update_widget(HippoCanvasHelper    *helper,
              RegisteredWidgetItem *witem)
{
    GtkWidget *new_widget;

    new_widget = NULL;
    g_object_get(G_OBJECT(witem->item), "widget", &new_widget, NULL);

    if (new_widget == witem->widget) {
        if (new_widget)
            g_object_unref(new_widget);
        return;
    }
    
    if (new_widget) {
        /* note that this ref/sinks the widget */
        gtk_widget_set_parent(new_widget, helper->widget);
    }

    if (witem->widget) {
        /* and this unrefs the widget */
        gtk_widget_unparent(witem->widget);
    }
    
    witem->widget = new_widget;

    if (new_widget)
        g_object_unref(new_widget);
}

static void
on_item_widget_changed(HippoCanvasItem *item,
                       GParamSpec      *arg,
                       void            *data)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(data);
    RegisteredWidgetItem *witem;
    GSList *link;
    
    witem = NULL;
    for (link = helper->widget_items;
         link != NULL;
         link = link->next) {
        witem = link->data;
        if (witem->item == item) {
            update_widget(helper, witem);
            return;
        }
    }

    g_warning("got widget changed for an unregistered widget item");
}

static void
add_widget_item(HippoCanvasHelper *helper,
                HippoCanvasItem   *item)
{
    RegisteredWidgetItem *witem = g_new0(RegisteredWidgetItem, 1);

    witem->item = item;
    g_object_ref(witem->item);
    helper->widget_items = g_slist_prepend(helper->widget_items, witem);

    update_widget(helper, witem);
    
    g_signal_connect(G_OBJECT(item), "notify::widget",
                     G_CALLBACK(on_item_widget_changed),
                     helper);
}

static void
remove_widget_item(HippoCanvasHelper *helper,
                   HippoCanvasItem   *item)
{
    RegisteredWidgetItem *witem;
    GSList *link;
    
    witem = NULL;
    for (link = helper->widget_items;
         link != NULL;
         link = link->next) {
        witem = link->data;
        if (witem->item == item)
            break;
    }
    if (link == NULL) {
        g_warning("removing a not-registered widget item");
        return;
    }

    helper->widget_items = g_slist_remove(helper->widget_items, witem);
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(witem->item),
                                         G_CALLBACK(on_item_widget_changed),
                                         helper);
    if (witem->widget) {
        gtk_widget_unparent(witem->widget);
        witem->widget = NULL;
    }
    g_object_unref(witem->item);
    g_free(witem);
}

static void
hippo_canvas_helper_register_widget_item(HippoCanvasContext *context,
                                         HippoCanvasItem    *item)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    
    add_widget_item(helper, item);
}

static void
hippo_canvas_helper_unregister_widget_item (HippoCanvasContext *context,
                                            HippoCanvasItem    *item)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);

    remove_widget_item(helper, item);
}

static void
hippo_canvas_helper_translate_to_widget(HippoCanvasContext *context,
                                        HippoCanvasItem    *item,
                                        int                *x_p,
                                        int                *y_p)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    GtkWidget *widget = helper->widget;

    /* convert coords of root canvas item to coords of
     * widget->window
     */

    if (GTK_WIDGET_NO_WINDOW(widget)) {
        if (x_p)
            *x_p += widget->allocation.x;
        if (y_p)
            *y_p += widget->allocation.y;
    }

    if (x_p)
        *x_p += GTK_CONTAINER(widget)->border_width;
    if (y_p)
        *y_p += GTK_CONTAINER(widget)->border_width;
}

static void
hippo_canvas_helper_translate_to_screen(HippoCanvasContext *context,
                                        HippoCanvasItem    *item,
                                        int                *x_p,
                                        int                *y_p)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    GtkWidget *widget = helper->widget;
    GdkWindow *window = widget->window;
    gint window_x, window_y;

    g_assert(window != NULL);

    /* convert coords of root canvas item to coords of
     * widget->window
     */

    if (GTK_WIDGET_NO_WINDOW(widget)) {
        if (x_p)
            *x_p += widget->allocation.x;
        if (y_p)
            *y_p += widget->allocation.y;
    }

    gdk_window_get_origin(window, &window_x, &window_y);

    if (x_p)
        *x_p += window_x;
    if (y_p)
        *y_p += window_y;
}

HippoCanvasStyle *
hippo_canvas_helper_get_style (HippoCanvasContext *context)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);

    if (helper->style == NULL) {
        helper->style = hippo_canvas_style_new(context, NULL, helper->theme,
                                               G_TYPE_NONE, NULL, NULL);
    }

    return helper->style;
}

double
hippo_canvas_helper_get_resolution (HippoCanvasContext *context)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);
    
    return gdk_screen_get_resolution(gtk_widget_get_screen(helper->widget));
}

PangoFontDescription *
hippo_canvas_helper_get_font (HippoCanvasContext *context)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);

    return helper->widget->style->font_desc;
}

HippoAnimationManager *
hippo_canvas_helper_get_animation_manager (HippoCanvasContext *context)
{
    HippoCanvasHelper *helper = HIPPO_CANVAS_HELPER(context);

    return helper->animation_manager;
}

static void
canvas_root_destroy(HippoCanvasItem   *root,
                    HippoCanvasHelper *helper)
{
    hippo_canvas_helper_set_root(helper, NULL);
}

static void
canvas_root_request_changed(HippoCanvasItem   *root,
                            HippoCanvasHelper *helper)
{
    /* g_debug("queuing resize on canvas root %p canvas %p canvas container %p",
       root, canvas, helper->widget->parent); */
    if (!helper->fixing_up_resize_state)
        gtk_widget_queue_resize_no_redraw(helper->widget);
}

static void
canvas_root_paint_needed(HippoCanvasItem      *root,
                         const HippoRectangle *damage_box,
                         HippoCanvasHelper    *helper)
{
    GtkWidget *widget = helper->widget;
    int window_x, window_y;
    
    get_root_item_window_coords(helper, &window_x, &window_y);

    gtk_widget_queue_draw_area(widget,
                               damage_box->x + window_x,
                               damage_box->y + window_y,
                               damage_box->width, damage_box->height);
}

static void
canvas_root_tooltip_changed(HippoCanvasItem   *root,
                            HippoCanvasHelper *helper)
{
    update_tooltip(helper, FALSE);
}

void
hippo_canvas_helper_set_root(HippoCanvasHelper *helper,
                             HippoCanvasItem   *root)
{
    GtkWidget *widget;
    gboolean was_hovering = FALSE;

    g_return_if_fail(HIPPO_IS_CANVAS_HELPER(helper));
    g_return_if_fail(root == NULL || HIPPO_IS_CANVAS_ITEM(root));

    widget = helper->widget;

    if (root == helper->root)
        return;

    if (helper->root != NULL) {
        g_signal_handlers_disconnect_by_func(helper->root,
                                             G_CALLBACK(canvas_root_destroy),
                                             helper);
        g_signal_handlers_disconnect_by_func(helper->root,
                                             G_CALLBACK(canvas_root_request_changed),
                                             helper);
        g_signal_handlers_disconnect_by_func(helper->root,
                                             G_CALLBACK(canvas_root_paint_needed),
                                             helper);
        g_signal_handlers_disconnect_by_func(helper->root,
                                             G_CALLBACK(canvas_root_tooltip_changed),
                                             helper);
        hippo_canvas_item_set_context(helper->root, NULL);
        g_object_unref(helper->root);
        helper->root = NULL;

        was_hovering = helper->root_hovering;
        helper->root_hovering = FALSE;
    }

    if (root != NULL) {
        g_object_ref(root);
        hippo_canvas_item_sink(root);
        helper->root = root;
        g_signal_connect(root, "destroy",
                         G_CALLBACK(canvas_root_destroy),
                         helper);
        g_signal_connect(root, "request-changed",
                         G_CALLBACK(canvas_root_request_changed),
                         helper);
        g_signal_connect(root, "paint-needed",
                         G_CALLBACK(canvas_root_paint_needed),
                         helper);
        g_signal_connect(root, "tooltip-changed",
                         G_CALLBACK(canvas_root_tooltip_changed),
                         helper);
        hippo_canvas_item_set_context(helper->root, HIPPO_CANVAS_CONTEXT(helper));

        if (was_hovering)
            handle_new_mouse_location(helper, widget->window,
                                      HIPPO_MOTION_DETAIL_ENTER);
    }

    gtk_widget_queue_resize(widget);
}

void
hippo_canvas_helper_set_theme(HippoCanvasHelper *canvas_helper,
                              HippoCanvasTheme  *theme)
{
    g_return_if_fail(HIPPO_IS_CANVAS_HELPER(canvas_helper));
    g_return_if_fail(theme == NULL || HIPPO_IS_CANVAS_THEME(theme));

    if (theme == canvas_helper->theme)
        return;

    if (canvas_helper->theme)
        g_object_unref(canvas_helper->theme);

    canvas_helper->theme = theme;

    if (canvas_helper->theme)
        g_object_ref(canvas_helper->theme);
    
    if (canvas_helper->style) {
        g_object_unref(canvas_helper->style);
        canvas_helper->style = NULL;
    }

    hippo_canvas_context_emit_style_changed(HIPPO_CANVAS_CONTEXT(canvas_helper), TRUE);
}

GtkWidget *
hippo_canvas_helper_get_widget(HippoCanvasHelper *helper)
{
    g_return_val_if_fail(HIPPO_IS_CANVAS_HELPER(helper), NULL);

    return helper->widget;
}

void
hippo_canvas_helper_set_width(HippoCanvasHelper *helper,
                              int                width)
{
    g_return_if_fail(HIPPO_IS_CANVAS_HELPER(helper));

    if (helper->width == width)
        return;

    helper->width = width;

    gtk_widget_queue_resize_no_redraw(helper->widget);
}

/*
 * This is a bad hack because GTK does not have a "resize queued" signal
 * like our request-changed; this means that if a widget inside a HippoCanvasWidget
 * queues a resize, the HippoCanvasWidget does not emit request-changed.
 *
 * Because all canvas widget items are registered with the HippoCanvas widget
 * they are inside, when we get the GTK size_request or size_allocate,
 * we go through and emit the missing request-changed before we request/allocate
 * the root canvas item.
 */
static void
hippo_canvas_helper_fixup_resize_state(HippoCanvasHelper *helper)
{
    RegisteredWidgetItem *witem;
    GSList *link;

    if (helper->fixing_up_resize_state) {
        g_warning("Recursion in %s", G_GNUC_PRETTY_FUNCTION);
        return;
    }
    
    helper->fixing_up_resize_state = TRUE;
    
    witem = NULL;
    for (link = helper->widget_items;
         link != NULL;
         link = link->next) {
        witem = link->data;

        if (witem->widget &&
            (GTK_WIDGET_REQUEST_NEEDED(witem->widget) ||
             GTK_WIDGET_ALLOC_NEEDED(witem->widget))) {
            hippo_canvas_item_emit_request_changed(witem->item);
        }
    }

    helper->fixing_up_resize_state = FALSE;
}

static gint
tooltip_expose_handler(GtkWidget *tip, GdkEventExpose *event, void *data)
{
    gtk_paint_flat_box(tip->style, tip->window,
                       GTK_STATE_NORMAL, GTK_SHADOW_OUT, 
                       &event->area, tip, "tooltip",
                       0, 0, -1, -1);
    
    return FALSE;
}

static gint
tooltip_motion_handler(GtkWidget *tip, GdkEventMotion *event, void *data)
{
    gtk_widget_hide(tip);
    return FALSE;
}

static void
tooltip_window_update(GtkWidget      *tip,
                      GdkScreen      *gdk_screen,
                      int             mouse_x,
                      int             mouse_y,
                      HippoRectangle *for_area,
                      const char     *text)
{
    GdkRectangle monitor;
    gint mon_num;
    int x, y;
    int w, h;
    GtkWidget *label;
    int screen_right_edge;
    int screen_bottom_edge;
    
    gtk_window_set_screen(GTK_WINDOW(tip), gdk_screen);
    mon_num = gdk_screen_get_monitor_at_point(gdk_screen, mouse_x, mouse_y);
    gdk_screen_get_monitor_geometry(gdk_screen, mon_num, &monitor);
    screen_right_edge = monitor.x + monitor.width;
    screen_bottom_edge = monitor.y + monitor.height;

    label = GTK_BIN(tip)->child;
    
    gtk_label_set(GTK_LABEL(label), text);
    
    gtk_window_get_size(GTK_WINDOW(tip), &w, &h);

    if (for_area->width < MAX_TOOLTIP_AREA_WIDTH) {
        /* Center on the area horizontally */
        x = for_area->x + (for_area->width - w) / 2;
    } else {
        x = mouse_x + TOOLTIP_OFFSET;
    }

    /* Clamp onscreen */
    if (x + w > screen_right_edge)
        x = screen_right_edge - w;
    if (x < 0)
        x = 0;

    if (for_area->height < MAX_TOOLTIP_AREA_HEIGHT) {
        /* And place it below if there is enough space, otherwise place it above */
        y = for_area->y + for_area->height + TOOLTIP_PADDING;
        if (y + h > screen_bottom_edge)
            y = for_area->y - h - TOOLTIP_PADDING;
    } else {
        y = mouse_y + TOOLTIP_OFFSET;

    }

    /* Clamp onscreen */
    if (y + h > screen_bottom_edge)
        y = screen_bottom_edge - h;
    if (y < 0)
        y = 0;
    
    gtk_window_move(GTK_WINDOW(tip), x, y);
}

static GtkWidget*
tooltip_window_new(void)
{
    GtkWidget *tip;
    GtkWidget *label;
    
    tip = gtk_window_new(GTK_WINDOW_POPUP);
    
    gtk_widget_set_app_paintable(tip, TRUE);
    gtk_window_set_policy(GTK_WINDOW(tip), FALSE, FALSE, TRUE);
    gtk_widget_set_name(tip, "gtk-tooltips");
    gtk_container_set_border_width(GTK_CONTAINER(tip), 4);
    
    g_signal_connect(tip, "expose-event",
                     G_CALLBACK(tooltip_expose_handler), NULL);
    g_signal_connect(tip, "motion-notify-event",
                     G_CALLBACK(tooltip_motion_handler), NULL);
    
    label = gtk_label_new(NULL);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    gtk_widget_show(label);
    
    gtk_container_add(GTK_CONTAINER(tip), label);
    
    return tip;
}
