-- |
-- Module:     Control.Wire.Prefab.Random
-- Copyright:  (c) 2011 Ertugrul Soeylemez
-- License:    BSD3
-- Maintainer: Ertugrul Soeylemez <es@ertes.de>
--
-- Wires for generating random noise.

module Control.Wire.Prefab.Random
    ( -- * Random noise
      noise,
      noiseR,

      -- * Specific types
      noiseF,
      noiseF1,
      wackelkontakt
    )
    where

import Control.Arrow
import Control.Wire.Classes
import Control.Wire.Types
import System.Random


-- | Generate random noise.

noise :: (ArrowRandom (>~), Random b) => Wire e (>~) a b
noise = mkFix $ arr Right <<< arrRand


-- | Generate random noise in range 0 <= x < 1.

noiseF :: ArrowRandom (>~) => Wire e (>~) a Double
noiseF = noise


-- | Generate random noise in range -1 <= x < 1.

noiseF1 :: ArrowRandom (>~) => Wire e (>~) a Double
noiseF1 = mkFix (arr (Right . (*2) . pred) <<< arrRand)


-- | Generate random noise in a certain range given by the input signal.
--
-- * Depends: Current instant.

noiseR :: (ArrowRandom (>~), Random b) => Wire e (>~) (b, b) b
noiseR = mkFix $ arr Right <<< arrRandR


-- | Generate a random boolean, where the input signal is the
-- probability to be 'True'.
--
-- * Depends: Current instant.

wackelkontakt :: ArrowRandom (>~) => Wire e (>~) Double Bool
wackelkontakt =
    mkFix $ proc p -> do
        s <- arrRand -< ()
        returnA -< Right (not (s >= p))
