// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Thread for scanning the connected devices
// ****************************************************************************

#include <QtCore>
#include <QtDBus/QtDBus>

#include <dlfcn.h>

#include "toolconstants.h"

#include "common.h"
#include "config.h"
#include "device.h"
#include "qtutil.h"
#include "threadscan.h"

// --------------------------
//         Constants
// --------------------------

const int THREADSCAN_WAIT_MAX         = 5000;
const int THREADSCAN_WAIT_GRANULARITY = 100;

#define   LIBPARTED  "libparted.so"

// -----------------------------------
//  Utility functions used by workers
// -----------------------------------

static APIRET ThreadScanMarkLocalDevices (const t_pDeviceList pDeviceList)
{
   QStringList *pLocalDevices;
   t_pDevice    pDevice;
   int           i;

   CHK_EXIT (CfgGetLocalDevices (&pLocalDevices))
   for (i=0; i<pDeviceList->count(); i++)
   {
      pDevice = pDeviceList->at (i);
      pDevice->Local = pLocalDevices->contains (pDevice->SerialNumber) ||
                       pLocalDevices->contains (pDevice->LinuxDevice );
   }
   return NO_ERROR;
}

// --------------------------
//  t_ThreadScanWorkerParted
// --------------------------

// Scan the devices using libparted and bash command to gather device info

typedef  void       (*t_pFnLibPartedProbeAll) (void) ;
typedef  PedDevice* (*t_pFnLibPartedGetNext ) (const PedDevice* pDev);
typedef  void       (*t_pFnLibPartedFreeAll ) (void);

class t_ThreadScanWorkerPartedLocal
{
   public:
      t_ThreadScanWorkerPartedLocal ()
         :LibPartedLoaded      (false), pLibPartedHandle     (NULL),
          pFnLibPartedProbeAll (NULL) ,
          pFnLibPartedGetNext  (NULL) ,
          pFnLibPartedFreeAll  (NULL)
      {
      };

   public:
      bool         LibPartedLoaded;
      void       *pLibPartedHandle;
      t_pFnLibPartedProbeAll pFnLibPartedProbeAll;
      t_pFnLibPartedGetNext  pFnLibPartedGetNext;
      t_pFnLibPartedFreeAll  pFnLibPartedFreeAll;
};


static APIRET ThreadScanGetSerialNumber (char const *pcDeviceName, QString &SerialNumber)
{
   QString Command;
   APIRET  rc;

   Command = CONFIG (CommandGetSerialNumber);
   Command.replace ("%dev", pcDeviceName, Qt::CaseInsensitive);

   rc = QtUtilProcessCommand (Command, SerialNumber);
   if (rc == ERROR_QTUTIL_COMMAND_TIMEOUT)
      return rc;
   CHK(rc)
   SerialNumber = SerialNumber.trimmed();

   return NO_ERROR;
}

t_ThreadScanWorkerParted::t_ThreadScanWorkerParted (APIRET &rc)
{
   char *pErrStr;

   pOwn = new t_ThreadScanWorkerPartedLocal();

   pOwn->pLibPartedHandle = dlopen (LIBPARTED, RTLD_NOW);
   if (pOwn->pLibPartedHandle == NULL)
   {
      LOG_INFO ("%s cannot be found (%s).", LIBPARTED, dlerror());
      rc = ERROR_THREADSCAN_LIBPARTED_NOTWORKING;
      return;
   }

   // GETFN: See dlsym documentation concerning the usage of dlerror for error checking
   #define GETFN(pFn, Type, pFnName)                                           \
      dlerror();                                                               \
      pFn = (Type)dlsym (pOwn->pLibPartedHandle, pFnName);                     \
      pErrStr = dlerror();                                                     \
      if (pErrStr)                                                             \
      {                                                                        \
         LOG_INFO ("Function %s cannot be found (%s).", pFnName, pErrStr);     \
         rc = ERROR_THREADSCAN_LIBPARTED_NOTWORKING;                           \
         return;                                                               \
      }

   GETFN (pOwn->pFnLibPartedProbeAll, t_pFnLibPartedProbeAll, "ped_device_probe_all")
   GETFN (pOwn->pFnLibPartedGetNext , t_pFnLibPartedGetNext , "ped_device_get_next" )
   GETFN (pOwn->pFnLibPartedFreeAll , t_pFnLibPartedFreeAll , "ped_device_free_all" )

   rc = NO_ERROR;
}

t_ThreadScanWorkerParted::~t_ThreadScanWorkerParted (void)
{
   int rc;

   if (pOwn->pLibPartedHandle)
   {
      rc = dlclose (pOwn->pLibPartedHandle);
      if (rc)
         LOG_INFO ("dlclose returned %d (%s).", rc, dlerror());
   }

   delete pOwn;
}


void t_ThreadScanWorkerParted::SlotRescan (void)
{
   t_pDeviceList  pDeviceList;
   t_pDevice      pDevice;
   PedDevice     *pPedDev;
   QString         SerialNumber;
   APIRET          rc;

   emit (SignalScanStarted ());
   LOG_INFO ("Rescanning devices")

   pOwn->pFnLibPartedProbeAll ();

   pDeviceList = new t_DeviceList;
   pPedDev     = NULL;
   while ((pPedDev = pOwn->pFnLibPartedGetNext (pPedDev)))  //lint !e820
   {
      rc = ThreadScanGetSerialNumber (pPedDev->path, SerialNumber);
      if (rc == ERROR_QTUTIL_COMMAND_TIMEOUT)
      {
         LOG_INFO ("Device scan aborted due to timeout while trying to get serial number for %s", pPedDev->path)
         pOwn->pFnLibPartedFreeAll ();
         delete pDeviceList;
         return;
      }
      pDevice = pDeviceList->AppendNew (SerialNumber, pPedDev);
      if (pDevice->LinuxDevice.startsWith ("/dev/fd",  Qt::CaseSensitive)) // Not a good way for checking this, but I don't know how to extract the information from
         pDevice->Removable = true;                                        // PedDevice. BTW, this won't work with other removable devices, for instance memory sticks.
   }
   pOwn->pFnLibPartedFreeAll ();
   CHK_EXIT (ThreadScanMarkLocalDevices (pDeviceList))

   emit (SignalScanFinished (pDeviceList));
}

// --------------------------
//  t_ThreadScanWorkerHAL
// --------------------------

// Scan the devices using DBUS/HAL

#define HAL_SERVICE      "org.freedesktop.Hal"
#define HAL_MANAGER_PATH "/org/freedesktop/Hal/Manager"
#define HAL_MANAGER_IF   "org.freedesktop.Hal.Manager"
#define HAL_DEVICE_IF    "org.freedesktop.Hal.Device"

class t_ThreadScanWorkerHALLocal
{
   public:
      t_ThreadScanWorkerHALLocal (void)
      {
         pDBusConnection = new QDBusConnection (QDBusConnection::systemBus());
         pDBusInterface  = NULL;
      }

     ~t_ThreadScanWorkerHALLocal ()
      {
         QDBusConnection::disconnectFromBus (pDBusConnection->baseService());
         delete pDBusConnection;
         pDBusConnection = NULL;
         pDBusInterface  = NULL;
      }

   public:
      QDBusConnection          *pDBusConnection;
      QDBusConnectionInterface *pDBusInterface;
};


t_ThreadScanWorkerHAL::t_ThreadScanWorkerHAL (APIRET &rc)
{
   pOwn = new t_ThreadScanWorkerHALLocal;
   if (!pOwn->pDBusConnection->isConnected())
   {
      LOG_INFO ("DBus connection failed")
      rc = ERROR_THREADSCAN_DBUSHAL_NOTWORKING;
      return;
   }

   pOwn->pDBusInterface = pOwn->pDBusConnection->interface();
   if (!pOwn->pDBusInterface->isServiceRegistered (HAL_SERVICE))
   {
      LOG_INFO ("HAL not registered")
      rc = ERROR_THREADSCAN_DBUSHAL_NOTWORKING;
      return;
   }
   rc = NO_ERROR;
}

t_ThreadScanWorkerHAL::~t_ThreadScanWorkerHAL (void)
{
   delete pOwn;
}

QList<QVariant> t_ThreadScanWorkerHAL::CallMethod (const QString &Device, const QString &Method, const QString &Argument)
{
   QList<QVariant> Args;
   QDBusMessage    Message = QDBusMessage::createMethodCall (HAL_SERVICE, Device, HAL_DEVICE_IF, Method);
   QDBusMessage    Reply;

   Args += Argument;
   Message.setArguments (Args);
   Reply = pOwn->pDBusConnection->call (Message);

   if (Reply.type() == QDBusMessage::ErrorMessage)
   {
      LOG_ERROR ("DBus returned '%s' for method %s on device %s", QSTR_TO_PSZ (Reply.errorName()),
                                                                  QSTR_TO_PSZ (Method),
                                                                  QSTR_TO_PSZ (Device))
      return QList<QVariant>(); // return empty list
   }

   return Reply.arguments();
}

QVariant t_ThreadScanWorkerHAL::CallMethodSingle (const QString &Device, const QString &Method, const QString &Argument)
{
   QList<QVariant> Results;

   Results = CallMethod (Device, Method, Argument);
   if (Results.first().isNull())
      return QVariant();

   return Results.first();
}

APIRET t_ThreadScanWorkerHAL::GetProperty (const QString &Device, const QString &Property, QList<QVariant> &VarList)
{
   QVariant Exists;

   Exists = CallMethodSingle (Device, "PropertyExists", Property);
   if (Exists.toBool())
        VarList = CallMethod (Device, "GetProperty", Property);
   else VarList = QList<QVariant>();  // Construct an empty, invalid list to show that the property doesn't exist

   return NO_ERROR;
}

APIRET t_ThreadScanWorkerHAL::GetPropertySingle (const QString &Device, const QString &Property, QVariant &Var)
{
   Var = CallMethodSingle (Device, "PropertyExists", Property);
   if (Var.toBool())
        Var = CallMethodSingle (Device, "GetProperty", Property);
   else Var = QVariant();  // Construct an empty, invalid QVariant to show that the property doesn't exist

   return NO_ERROR;
}

bool t_ThreadScanWorkerHAL::PropertyContains (const QString &Device, const QString &Property, const QString &Str)
{
   QList<QVariant> PropertyElements;
   bool            Found = false;

   CHK_EXIT (GetProperty (Device, Property, PropertyElements))

   foreach (QVariant Element, PropertyElements)
   {
      if (Element.isValid())
      {
         switch (Element.type())
         {
            case QVariant::String    : Found = (Element.toString() == Str);                                  break;
            case QVariant::StringList: Found =  Element.toStringList().contains (Str, Qt::CaseInsensitive);  break;
            default                  : break;
         }
      }
      if (Found)
         break;
   }
   return Found;
}

void t_ThreadScanWorkerHAL::SlotRescan (void)
{
   t_pDeviceList   pDeviceList=NULL;
   t_pDevice       pDevice;
   QDBusMessage     Message;
   QDBusMessage     Reply;
   QStringList      DeviceList;
   QVariant         Category;
   QVariant         SerialNumber;
   QVariant         LinuxDevice;
   QVariant         Vendor;
   QVariant         Product;
   QVariant         Size;
   QVariant         Removable;
   QVariant         RemovableAvailable;
   QString          Model;

   if (thread() != QThread::currentThread())
      CHK_EXIT (ERROR_THREADSCAN_CALLED_FROM_WRONG_THREAD) // As Qt's DBus system is quite sensible to this kind of
                                                           // mistake (resulting in many QTimer "cannot be stopped/started
                                                           // from another thread) we prefer to do the check here!
   emit (SignalScanStarted ());
   LOG_INFO ("Device scan started")

   Message = QDBusMessage::createMethodCall (HAL_SERVICE, HAL_MANAGER_PATH, HAL_MANAGER_IF, "GetAllDevices");
   Reply   = pOwn->pDBusConnection->call (Message);
   if (Reply.type() == QDBusMessage::ErrorMessage)
   {
      LOG_ERROR ("DBus GetAllDevices returned %s", QSTR_TO_PSZ(Reply.errorName()))
   }
   else
   {
      pDeviceList = new t_DeviceList;
      DeviceList  = Reply.arguments().first().toStringList();

      //lint -save -e155 -e521
      foreach (QString HalDevice, DeviceList)
      //lint -restore
      {
         // Do not check the info.category any longer. It has been observed, that it may contain many different
         // strings. For example, a phone had "portable_audio_player" in its info.category and was not detected,
         // eventhough it was perfectly readable by dd.
//         CHK_EXIT (GetPropertySingle (HalDevice, "info.category", Category))
//         if (!Category.isValid())
//            continue;
//         if (Category != "storage")
//            continue;

         CHK_EXIT (GetPropertySingle (HalDevice, "block.device", LinuxDevice))
         if (!LinuxDevice.isValid())
            continue;

         if (PropertyContains (HalDevice, "info.capabilities", "volume"))  // we only are interested in complete device (/dev/sda), not in volumes (/dev/sda1)
            continue;

         CHK_EXIT (GetPropertySingle (HalDevice, "storage.removable", Removable))
         if (!Removable.isValid())
         {
            LOG_INFO ("Strange, %s is a block device but has no storage.removable property", QSTR_TO_PSZ(HalDevice))
            continue;
         }
         if (Removable.toBool())
         {
            CHK_EXIT (GetPropertySingle (HalDevice, "storage.removable.media_available", RemovableAvailable))
            if (RemovableAvailable.toBool())
                 CHK_EXIT (GetPropertySingle (HalDevice, "storage.removable.media_size", Size))
            else continue;
         }
         else
         {
            CHK_EXIT (GetPropertySingle (HalDevice, "storage.size", Size))
         }

         CHK_EXIT (GetPropertySingle (HalDevice, "storage.serial", SerialNumber))
         CHK_EXIT (GetPropertySingle (HalDevice, "info.vendor"   , Vendor      ))
         CHK_EXIT (GetPropertySingle (HalDevice, "info.product"  , Product     ))
         if (!SerialNumber.isValid())
            SerialNumber = "";  // Attention: Empty string must be used, else t_DeviceList::MatchDevice doesn't work
         if (!Vendor.isValid() && !Product.isValid())
              Model = "--";
         else Model = Vendor.toString() + " " + Product.toString();
         Model = Model.trimmed();

//         if (SerialNumber.toString().contains("Hitachi", Qt::CaseInsensitive))
//            Size  = QVariant ((qulonglong)1024*1024*1024*32);  // for test purposes
         pDevice = pDeviceList->AppendNew (SerialNumber.toString(), LinuxDevice.toString(), Model,
                                           DEVICE_DEFAULT_SECTOR_SIZE, DEVICE_DEFAULT_SECTOR_SIZE, Size.toULongLong());
         pDevice->Removable = Removable.toBool();
      }
   }
   CHK_EXIT (ThreadScanMarkLocalDevices (pDeviceList))
   LOG_INFO ("Device scan finished")

   emit (SignalScanFinished (pDeviceList));
}

// --------------------------
//        t_ThreadScan
// --------------------------

static APIRET ThreadScanRegisterErrorCodes (void)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      Initialised = true;
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_NOT_STARTED             ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_NOT_STOPPED             ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_EXITCODE_NONZERO        ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_PROCESS_NOTSTARTED      ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_PROCESS_NOTFINISHED     ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_DBUSHAL_NOTWORKING      ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_LIBPARTED_NOTWORKING    ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_PROPERTY_NONEXISTENT    ))
      CHK (TOOL_ERROR_REGISTER_CODE (ERROR_THREADSCAN_CALLED_FROM_WRONG_THREAD))
   }
   return NO_ERROR;
}

t_ThreadScan::t_ThreadScan (void)
{
   CHK_EXIT (ThreadScanRegisterErrorCodes())
   ppoWorker = NULL;  //lint -esym(613, t_ThreadScan::ppoWorker)  Possible use of NULL pointer
}


static APIRET ThreadScanAskUser (bool SwitchToDBusHAL, bool &Ok)
{
   QMessageBox::StandardButton   Button;
   const char                  *pFrom, *pTo;

   pFrom =  SwitchToDBusHAL ? "libparted" : "DBus/HAL";
   pTo   = !SwitchToDBusHAL ? "libparted" : "DBus/HAL";
   LOG_INFO ("%s not accessible, asking user if he wants to switch to %s...", pFrom, pTo)
   Button = QMessageBox::question (NULL, QObject::tr ("%1 not accessible", "Dialog title") .arg(pFrom),
                                         QObject::tr ("Guymager is configured to use %1 for scanning the connected devices, but this "
                                                      "did not work. Alternatively, you may use %2. Do you want to switch to %2?")
                                                      .arg(pFrom) .arg(pTo), QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes);
   Ok = (Button == QMessageBox::Yes);
   if (Ok)
   {
      LOG_INFO ("User switches to %s", pTo)
      CONFIG (ScanUsingDbusHal) = SwitchToDBusHAL;
   }
   else
   {
      LOG_INFO ("User doesn't want to switch. Exiting now.")
   }

   return NO_ERROR;
}

APIRET t_ThreadScan::Start (t_ThreadScanWorker **ppWorker)
{
   int  Wait;
   bool Ok;
   bool SwitchToDBusHAL;

   #define LAUNCH_WORKER                                                                    \
   {                                                                                        \
      *ppWorker   = NULL;                                                                   \
      ppoWorker   = ppWorker;                                                               \
        oWorkerRc = NO_ERROR;                                                               \
      start();                                                                              \
      /* start(QThread::HighPriority); */                                                   \
      for ( Wait =  0;                                                                      \
           (Wait <  THREADSCAN_WAIT_MAX) && (*ppWorker == NULL) && (oWorkerRc==NO_ERROR);   \
            Wait += THREADSCAN_WAIT_GRANULARITY)                                            \
         msleep (THREADSCAN_WAIT_GRANULARITY);                                              \
   }

   LAUNCH_WORKER;

   if ((oWorkerRc == ERROR_THREADSCAN_DBUSHAL_NOTWORKING) ||
       (oWorkerRc == ERROR_THREADSCAN_LIBPARTED_NOTWORKING))
   {
      SwitchToDBusHAL = (oWorkerRc == ERROR_THREADSCAN_LIBPARTED_NOTWORKING);
      CHK (ThreadScanAskUser (SwitchToDBusHAL, Ok))
      if (Ok)
         LAUNCH_WORKER;
   }
   #undef LAUNCH_WORKER

   CHK (oWorkerRc)
   if (*ppWorker == NULL)
      CHK (ERROR_THREADSCAN_NOT_STARTED)

   return NO_ERROR;
}

APIRET t_ThreadScan::Stop ()
{
   time_t BeginT, NowT;
   int    Wait;

   quit(); // This tells t_ThreadScan::run to quit the exec() call

   time (&BeginT);  // As we do not know how much time is spent in the different calls
   do               // to processEvents, we have to measure the time ourselves
   {
      QCoreApplication::processEvents (QEventLoop::ExcludeUserInputEvents, THREADSCAN_WAIT_GRANULARITY);
      time (&NowT);
      Wait = (NowT-BeginT)*MSEC_PER_SECOND;
   } while ((Wait < THREADSCAN_WAIT_MAX) && (*ppoWorker != NULL));

   if (*ppoWorker != NULL)
      CHK (ERROR_THREADSCAN_NOT_STOPPED)

   return NO_ERROR;
}


void t_ThreadScan::run (void)
{
   QTimer *pTimer;
   int      rc;

   LOG_INFO ("Thread Scan started")

   if (CONFIG(ScanUsingDbusHal))
        *ppoWorker = new t_ThreadScanWorkerHAL    (oWorkerRc); // We have to create this object as we want to work with signals
   else *ppoWorker = new t_ThreadScanWorkerParted (oWorkerRc); // and slots in this new thread. t_ThreadScan itself belongs to
                                                               // the main thread and thus can't be used for signals and slots.
   if (oWorkerRc)
      return;

//   pTimer = new QTimer(this); // Do not take "this" as parent, as "this" doesn't belong to the current thread (would lead to the error "Cannot create children for a parent that is in a different thread").
   pTimer = new QTimer();
   CHK_QT_EXIT (connect (pTimer, SIGNAL(timeout()), *ppoWorker, SLOT(SlotRescan())))
   pTimer->start (CONFIG(ScanInterval) * MSEC_PER_SECOND);

   rc = exec(); // Enter event loop
   if (rc)
   {
      LOG_ERROR ("ThreadScan exits with code %d", rc)
      CHK_EXIT (ERROR_THREADSCAN_EXITCODE_NONZERO)
   }
   LOG_INFO ("Thread Scan ended")
   delete pTimer;
   delete *ppoWorker;
   *ppoWorker = NULL;
}



