/****************************************************************************
 **
 ** Copyright (C) 2000-2006 lorn potter. All rights reserved.
 **
 **
 **
 ** GPL or QPL
 **
 ** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 ** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 **
 ****************************************************************************/
  //http://www.gutenberg.org/files/1163/1163-h/1163-h.htm
  //ftp://ftp.ibiblio.org/pub/docs/books/gutenberg/
  //http://mirror.pacific.net.au/gutenberg/


#include "guten.h"
#include "gutendb.h"
#include "networkdialog.h"
#include "settingsdialog.h"

//#include "zip/zip.h"
#include "zip/unzip.h"

#include <QTimer>
#include <QStatusBar>
#include <QMenu>
#include <QMenuBar>
#include <QSettings>
#include <QFontDialog>
#include <QTemporaryFile>
#include <QMouseEvent>
#include <QModelIndex>
#include <QDir>
#include <QRegion>
#include <QProcess>
#include <QCursor>
#include <QDirIterator>


#ifdef Q_WS_QWS
#include <qtopia/qsoftmenubar.h>
#endif

#define TEMPFILE "/tmp/guten.tmp"


// http://www.gutenberg.org/feeds/catalog.rdf.zip
// http://www.gutenberg.org/feeds/today.rss
// http://www.gutenberg.org/dirs/GUTINDEX.ALL
// http://www.gutenberg.org/MIRRORS.ALL

GutenBase::GutenBase( QWidget *parent, Qt::WFlags f )
        : QMainWindow( parent, f )
{
    setupUi( this );
}

GutenBase::~GutenBase()
{
}

Guten::Guten( QWidget *parent, Qt::WFlags f )
        : GutenBase( parent, f ),
          downloadId(0)
{
    if( !QDir(QDir::homePath() + ".gutenbrowser").exists()) {
        QDir::home().mkdir(".gutenbrowser");
    }
    
    setWindowIcon(QIcon(":images/gutenbrowser.png"));
    
    gutdb = new gutenDb();
    gutdb->connectDatabase();
       
    createActions();
    createMenu();
    createToolbar();

    connect(searchButton,SIGNAL(clicked()),this,SLOT(doSearch()));
    connect(treeWidget,SIGNAL(itemDoubleClicked(QTreeWidgetItem *,int)),
            this, SLOT(treeClicked(QTreeWidgetItem *,int)));
    
    connect(treeWidget,SIGNAL(itemSelectionChanged()),
            this, SLOT(treeSelectionChanged()));

    connect(webView, SIGNAL(loadStarted()), this,SLOT(loadStarted()));
    connect(webView, SIGNAL(loadFinished(bool)), this,SLOT(loadFinished(bool)));
    connect(webView,SIGNAL(loadProgress(int)),this,SLOT(loadProgress(int)));
    connect(this,SIGNAL(openSuccess()),this,SLOT(openOk()));

    treeWidget->hideColumn(3);// file
    treeWidget->hideColumn(4);// book id
    
    treeWidget->sortByColumn(4);
    treeWidget->setAlternatingRowColors(true);
    comboBox->setEditable(true);

//     status = new QStatusBar(this);
      //    QTimer::singleShot(0, this, SLOT(showMirrorSettings()));
    
    statusBar()->showMessage(tr("Ready"));
    readSettings();
    gutenLibraryDir = QDir::homePath()+"/Gutenberg";
    
    showInstalled();
    
    qWarning() << webView->page()->settings()->fontFamily(QWebSettings::StandardFont);
    
    webView->page()->settings()->setFontFamily(QWebSettings::FixedFont, font().family());
    
    resultsTab->hide();
    
    if( gutdb->databaseNeedsUpdate)
        QTimer::singleShot(500, this,SLOT(showDatabaseWarning()));
}

Guten::~Guten()
{
      //   writeSettings();
    delete gutdb;
    gutdb = 0;
}


/*!
  \fn Guten::createActions()
*/
void Guten::createActions()
{
    addFileAction = new QAction( tr("&Add"), this);
    addFilesAction = new QAction( tr("Add Directory"), this);
    removeAction = new QAction( tr("Remove"), this);
    searchNetAction = new QAction( tr("More Info"), this);
    showAllAction = new QAction( tr("Gutenberg"), this);
    showInstalledAction = new QAction( tr("Local"), this);
    backAction =  new QAction( tr("Back"), this);
    forwardAction = new QAction( tr("Forward"), this);
    
//     pageUp = new QAction( tr("Page Up"), this);
//     pageDown = new QAction( tr("Page DOwn"), this);

//  connect( , SIGNAL(triggered()), this, SLOT( ()));
    connect( addFileAction, SIGNAL(triggered()), this, SLOT(addFile()));
    connect( addFilesAction, SIGNAL(triggered()), this, SLOT(addFiles()));
    connect( removeAction , SIGNAL(triggered()), this, SLOT(removeBook()));
    connect( searchNetAction , SIGNAL(triggered()), this, SLOT(searchNet()));
    connect( showAllAction, SIGNAL(triggered()), this, SLOT(showAll()));
    connect( showInstalledAction, SIGNAL(triggered()), this, SLOT(showInstalled()));
    
    connect( backAction, SIGNAL(triggered()), webView, SLOT(back()));
    connect( forwardAction, SIGNAL(triggered()), webView, SLOT(forward()));
    
      // connect( , SIGNAL(triggered()), this, SLOT( ()));
    aboutAct = new QAction(tr("About"), this);
    aboutQtAct = new QAction(tr("About Qt"), this);
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));
    connect(aboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()));   
}

void Guten::createMenu()
{
#ifdef Q_WS_QWS
    QMenu *contextMenu = QSoftMenuBar::menuFor(this);
    contextMenu->addAction( tr("Update Library"), this, SLOT( updateDb()));
    contextMenu->addAction( tr("Show All"), this, SLOT( showAll()));
    contextMenu->addAction( tr("Show Installed"), this, SLOT( showInstalled()));
    
    contextMenu->addAction( tr("Server"), this, SLOT( showMirrorSettings()));
    
//  contextMenu->addAction( tr("Local Library"), this, SLOT( showAll()));
    
    contextMenu->addSeparator();
#else
    QMenuBar *menubar;
    menubar = menuBar();
    menubar->setObjectName(QString::fromUtf8("menubar"));
    
    QMenu *fileMenu = menubar->addMenu(tr("&File"));
    fileMenu->addAction(addFileAction);
    fileMenu->addAction(addFilesAction);
    fileMenu->addAction(removeAction);
    fileMenu->addAction(searchNetAction);
    fileMenu->addSeparator();
    fileMenu->addAction(tr("&Quit"),this, SLOT(close()));
    
    QMenu *viewMenu = menubar->addMenu(tr("&View"));
    viewMenu->addAction(tr("&Top"), this, SLOT(moveToTop()));
    viewMenu->addAction(tr("&Start"), this, SLOT(moveToStart()));
//    viewMenu->addAction(tr("&Remove"), this, SLOT(removeBook()));
    
    QMenu *libraryMenu = menubar->addMenu(tr("&Library"));
    libraryMenu->addAction(showAllAction);
    libraryMenu->addAction(showInstalledAction);
    libraryMenu->addAction(tr("&Search"), this, SLOT(showSearchTab()));
    libraryMenu->addSeparator();
    libraryMenu->addAction(tr("&Update"),this,SLOT(updateDb()));
    
    QMenu *settingsMenu = menubar->addMenu(tr("&Settings"));
    settingsMenu->addAction(tr("&Font"),this,SLOT(doNewFont()));
    settingsMenu->addAction(tr("&Server Settings"),this,SLOT(showMirrorSettings()));
    
    QMenu* helpMenu = menuBar()->addMenu(tr("Help"));
    helpMenu->addAction(aboutAct);
    helpMenu->addAction(aboutQtAct);
    
      //gridLayout->addWidget(menubar, 0, 0, 1, 1);
    
      //QDockWidget *dockBar;
      //dockBar = new QDockWidget("Tools", this);
      //dockBar->setAllowedAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea);
    
// QPushButton *searchButton;
// searchButton = new QPushButton("Lookup",this);
// searchButton->setMaximumWidth(60);
//  dockBar->setWidget(searchButton);
    
// connect(searchButton,SIGNAL(clicked()), this, SLOT(lookupClicked()));
    
    
//     addDockWidget(Qt::RightDockWidgetArea, dockBar);
    
#endif
    
}

/*!
  \fn Guten::createToolbar()
*/
void Guten::createToolbar()
{
  
    QToolBar *toolbar;
    toolbar = addToolBar("tools");

    toolbar->addAction(backAction);
    toolbar->addAction(forwardAction);
      //   toolbar->addAction(removeAction);
    toolbar->addAction(searchNetAction);
    toolbar->setFloatable(false);
    toolbar->setMovable(false);
      //  toolbar->setAllowedAreas(Qt::RightToolBarArea);
    
      //   toolbar->addAction(showAllAction);
      //   toolbar->addAction(showInstalledAction);
 
    toolbar->setGeometry( width() - 25, 0, width()/3, 25);
}

void Guten::updateDb()
{
    switch(QMessageBox::warning(this, tr("Gutenbrowser update"),
                                tr("<p>Really update the Gutenberg Library?<br>This can take a long time.\n\n</p>"),
                                tr("Ok"),
                                tr("Cancel"), 0, 0, 1)) {
      case 0:
          statusBar()->showMessage(tr("Long process here. Please wait..."));
          loadStarted();        
          qApp->processEvents();
          gutdb->updateLibraryDatabase();
          break;
      case 1:
          break;
    };
    statusBar()->showMessage(tr("Ready"));
    loadFinished(true);
    showAll();
}


void Guten::doSearch()
{
    statusBar()->showMessage(tr("Searching the Gutenberg library."));
    treeWidget->clear();
    tabWidget->setTabText(1, tr("Gutenberg Library"));
    QString searchString = comboBox->currentText();
    if( authorCheckBox->isChecked())
        fillTree(gutdb->lookup("author",searchString));
    else if (titleCheckBox->isChecked())
        fillTree(gutdb->lookup("title",searchString));

    QSettings *settings = getSettings();
    settings->setValue("search/lastSearch", searchString);

}

void Guten::showAll()
{
    qApp->processEvents();
    statusBar()->showMessage(tr("Long process here. Please wait..."));
    
    loadStarted();        

    qApp->processEvents();
    treeWidget->clear();
    fillTree(gutdb->returnAllEtext());
    loadFinished(true);
    tabWidget->setTabText(1, tr("Gutenberg Library"));   
}

void Guten::showInstalled()
{
    tabWidget->setTabText(1, tr("Local Library"));
    loadStarted();        
    qApp->processEvents();
    treeWidget->clear();
    fillTree(gutdb->returnInstalled(), true);
    loadFinished(true);
}

void Guten::treeClicked(QTreeWidgetItem *item, int column)
{
    if (!item)
        return;
    
    if(item->text(2) == "X") {
        if(openEtext(item->text(4) )) {
            emit openSuccess();            
        }
        return;
    }

    Q_UNUSED(column);
    qWarning() << __PRETTY_FUNCTION__ << item->text(3).toLocal8Bit() << item->text(0).toLocal8Bit();
    
    preferredExtension = "-h.zip";
    fetchEbook(item->text(3), item->text(4).toInt());

}


void Guten::fillTree(QList<QStringList> resultList, bool isInstalled)
{
    Q_UNUSED(isInstalled);
    QList<QTreeWidgetItem *> itemList;
    QStringList list;
      //    treeWidget->setSortingEnabled(false);
    for (int i = 0; i < resultList.size(); ++i) {
        qApp->processEvents();
          // qWarning(resultList.at(i)[1].toLocal8Bit());
        QTreeWidgetItem *item;
        item = new QTreeWidgetItem();

          /* from db:
             0     title
             1     author
             2     installed
             3     file
             4     id
          */

        item->setText(0,resultList.at(i)[0]);// title
        item->setText(1,resultList.at(i)[1]); //author
        item->setText(2,resultList.at(i)[2]); //installed
        item->setText(3,resultList.at(i)[3]); //file
        item->setText(4,resultList.at(i)[4]); //book id
        
        itemList << item;
    }
    
    treeWidget->addTopLevelItems(itemList);
    tabWidget->setCurrentIndex(1);
    treeWidget->resizeColumnToContents(0);
    treeWidget->resizeColumnToContents(1);
      //  treeWidget->setSortingEnabled(true);
    QString text = QString("Showing %1 Gutenberg Etexts").arg(resultList.size());
    statusBar()->showMessage(text);

}

void Guten::showMirrorSettings()
{
    statusBar()->showMessage(tr("Gutenberg server mirrors."));
    settingsDialog *sDialog;
    sDialog = new settingsDialog();
      //sDialog->settingsTreeWidget->resizeColumnToContents(0);
    sDialog->stackedWidget->setCurrentWidget(sDialog->mirrorsPage);
    sDialog->exec();

      //  sDialog->settingsTreeWidget->resizeColumnToContents(1);
    statusBar()->showMessage(tr("Ready."));
}


QString Guten::findFile(const QString &fileToBeFound, const QString &preferred)
{ // "-h.zip" ".zip" ".txt"
    qWarning() << __PRETTY_FUNCTION__ << fileToBeFound << preferred;
    QFile file(TEMPFILE);
    QString fileFound;
    QString ext = preferred;
    bool foundTxt = false;
    bool foundZip = false;
    bool foundhtml = false;
    
    if ( file.open(QIODevice::ReadWrite)) {
          // file opened successfully
        QTextStream indexStream( &file );
        QString indexLine;

        while ( !indexStream.atEnd() ) {
            indexLine = indexStream.readLine();
            
            if(indexLine.contains(fileToBeFound)) {
                
                int i_first = indexLine.indexOf(">"+fileToBeFound,0)+1;
                int i_last = indexLine.indexOf("</a>   ",0);

                qWarning("%d",i_first);
                qWarning("%d",i_last);

                fileFound = indexLine.mid(i_first, i_last - i_first );
                  //fileFound = fileFound.left( fileFound.length()-4);
                
                qWarning(fileFound.toLocal8Bit());
                
                if(indexLine.contains("-h.zip")) {
                    foundhtml = true;
                    break;
                } else if(indexLine.contains(".zip")) {
                    foundZip = true;
                    break;
                } else if(indexLine.contains(".txt")) {
                    foundTxt = true;
                    break;
                }
            }
        }
        file.close();
    }
    else {
        return "";
    }
         
    if(foundhtml || foundZip || foundTxt)
        return fileFound;    
//        file.remove();
    return "";
}


bool Guten::openEtext(const QString &file)
{
    qWarning() << __PRETTY_FUNCTION__ << file;
    webView->load(QUrl());
    tabWidget->setTabText(2, "Loading...");
      //     
      //   QString zipFileName = findFileName( gutenLibraryDir + file, "zip");
    
    QString textFileName;
    QString id = treeWidget->currentItem()->text(4);
      //QString id = file.right( file.length() - file.lastIndexOf("/")-1);
    
    textFileName = gutdb->getEtextPath(id);
    qWarning() << textFileName << id;
    QFileInfo fi(textFileName);
    QString textFilePath = fi.absolutePath();
    
    if(fi.exists() &&fi.suffix() == "zip") { // unzip it
        qWarning() << "Ok to unzip!";
        if(!decompress( textFileName, textFilePath )) {
  
            switch(QMessageBox::warning(this, tr("Decompress"),
                                        tr("<p>Could not unzip etext.<br>Remove files?</p>"),
                                        tr("Ok"),
                                        tr("Cancel"), 0, 0, 1)) {
              case 0:
                  removeFile(textFileName);
                  break;
              case 1:
                  break;
            };
            return false;
            
        } else {
            removeFile(textFileName);
        }
        if(textFileName.contains("-h")) {
            textFileName = findFileName( textFilePath, "htm");
        } else {
            textFileName = textFileName.replace(QString(".zip"), QString(".txt"));
        }
              
        if(QFileInfo(textFileName).exists()) {
            gutdb->setEtextPath( id, textFileName );
         }
    } else {
        if( textFileName == "")
            if(textFileName.contains("-h")) {
                textFileName = findFileName( textFilePath, "htm");
            } else {
                textFileName = textFileName.replace(QString(".zip"), QString(".txt"));
            }
    }
    
    
// go ahead and open it.

    qWarning() << "Open it" << textFileName;

    if(QFileInfo(textFileName).isFile()) {
        webView->load(QUrl(textFileName));
        webView->show();
    } else {
        QString msg = QString("<p>Could not open %1</p>").arg(textFileName);
        QMessageBox::information(this,tr("Gutenbrowser"), msg);

        return false;
    }

    tabWidget->setCurrentIndex(2);
    return true;
      //startAutoScroll();
/*    QString text = QString("Showing %1 installed Gutenberg Etexts").arg(fileList.size());
      statusBar()->showMessage(text);*/
}

bool Guten::decompress(const QString& file, const QString& out/*, const QString* pwd*/)
{
    statusBar()->showMessage(tr("Decompressing...."));

    qWarning() << "decompress" << file << out;
    
    if (!QFile::exists(file))  {
        qWarning( "File does not exist");
        return false;
    }

    UnZip::ErrorCode ec;
    UnZip uz;
      //    if (pwd != 0)
      //  uz.setPassword(*pwd);

    ec = uz.openArchive(file);
    if (ec != UnZip::Ok)  {
        qWarning() << "Unzip error code: " << ec;
        QString errorCode = tr("Error Code: %1").arg(QString::number(ec));
        QMessageBox::critical(0, tr("Critical"),
                              tr("<p>There was a problem opening ebook archive.</P>")
                              +errorCode ,
                              QMessageBox::Cancel,
                              QMessageBox::NoButton);        
        statusBar()->showMessage(tr("Opening the Zip Archive failed"));
        return false;
    }

    ec = uz.extractAll(QDir(out));
    if (ec != UnZip::Ok) {
        qWarning() << "Unzip error code: " << ec;
        uz.closeArchive();
        QString errorCode = tr("Error Code: %1").arg(QString::number(ec));
        QMessageBox::critical(0, tr("Critical"),
                              tr("<p>There was a problem extracting ebook archive.</p>")
                              +errorCode ,
                              QMessageBox::Cancel,
                              QMessageBox::NoButton);              
        statusBar()->showMessage(tr("Could not extra file."));
        return false;
    }

    return true;
}


void Guten::addFile()
{
#ifdef Q_WS_QWS
#else
    QStringList slist = QFileDialog::getOpenFileNames( this,
                                                       "Choose a file",
                                                       gutenLibraryDir,
                                                       "txt (*.txt);; html (*.html | *.htm);; zip (*.zip)");
    QStringList::Iterator it = slist.begin();
    while(it != slist.end()) {
        QString fileName = (*it);

        qWarning(fileName.toLocal8Bit());
        QFileInfo fi(fileName);

        QString filePath = fi.absoluteFilePath();
        QString base = fi.baseName();
        qWarning() << base;
        
        QList<QStringList> dblist = gutdb->find("id", base.simplified());
        if(dblist.isEmpty()) {
            dblist = gutdb->find("id", base.left(base.length()-2).simplified());
        }
        if(!dblist.isEmpty()) {
            for (int i = 0; i < dblist.size(); ++i) {
                gutdb->installEtext(dblist.at(i)[4]);
                gutdb->setEtextPath(dblist.at(i)[4],fi.absoluteFilePath());
            }
        }
        ++it;
    }
#endif
    showInstalled();
}


void Guten::addFiles()
{
#ifdef Q_WS_QWS
#else
    QString dirStr = QFileDialog::getExistingDirectory( this,
                                                        "Choose a directory",
                                                        gutenLibraryDir);
    QDir dir(dirStr);

    QDirIterator it(dirStr, QDirIterator::Subdirectories);
    while (it.hasNext()) {
        QFileInfo fi = it.next();
        QString base = fi.baseName();
        qWarning() << "find" << fi.fileName();
        if(fi.completeSuffix() == "txt" || fi.completeSuffix() == "htm") { 
            
            QList<QStringList> dblist = gutdb->find("id", base.simplified());
            if(dblist.isEmpty()) {
                dblist = gutdb->find("id",base.left(base.length()-2).simplified());
            }
            if(!dblist.isEmpty()) {
                for (int i = 0; i < dblist.size(); ++i) {
                    gutdb->installEtext(dblist.at(i)[4]);
                    gutdb->setEtextPath(dblist.at(i)[4], fi.absoluteFilePath());
                }
            }
        }
#endif
    }
    showInstalled();
}

void Guten::doNewFont()
{
#ifdef Q_WS_QWS
#else
      //    webView->setFont(QFontDialog::getFont(0, webView->font()));
    bool ok;
    QFont newFont = QFontDialog::getFont(&ok, webView->font(), this);
    if (ok) {
        webView->page()->settings()->setFontFamily(QWebSettings::FixedFont, newFont.family());
          //        webView->setFont(newFont);
        QSettings *settings = getSettings();
        settings->beginGroup("General");
        settings->setValue("font", newFont.toString());
        qWarning("write font is "+newFont.toString().toLocal8Bit());
    } else {

    }

#endif
}



void Guten::keyReleaseEvent( QKeyEvent * e )
{
    switch(e->key()) {
      case Qt::Key_Return:
          qWarning("key");
          if(!comboBox->currentText().isEmpty())
              doSearch();
          break;

    };
}

void Guten::writeSettings()
{
    QSettings *settings = getSettings();

    settings->beginGroup("General");
    settings->setValue("size", size());
    settings->setValue("pos", pos());

    QFont font = webView->font();
    settings->setValue("font",font.toString());
    qWarning("write font is "+font.toString().toLocal8Bit());

    settings->endGroup();
}

QSettings *Guten::getSettings()
{
    return  new QSettings("llornkcor", "Gutenbrowser");
}

void Guten::readSettings()
{
    QSettings *settings = getSettings();
    settings->beginGroup("General");

    resize(settings->value("size", QSize(720, 610)).toSize());
    move(settings->value("pos", QPoint(200, 200)).toPoint());

//     QString fontStr =settings->value("font").toString();
//     qWarning("set font is "+fontStr.toLocal8Bit());
//     QFont font(fontStr.simplified());
//     webView->setFont(font);

    settings->endGroup();
}


void Guten::showSearchTab()
{
    tabWidget->setCurrentIndex(0);
}

void Guten::mousePressEvent( QMouseEvent * event )
{
    switch(event->button()) {
      case  Qt::RightButton:
          QMenu *treeMenu;
          treeMenu = new QMenu(this);
          treeMenu->addAction(tr("More information"), this,SLOT(searchNet()));
          treeMenu->exec(QCursor::pos());
          break;
      default:
          break;
    };
}

void Guten::searchNet()
{
    if (!treeWidget->currentItem())
        return;
 
      //  QString text = "\"" + treeWidget->currentItem()->text(1)+"\"";
      //author only
    
    QString text;
    if(! webView->selectedText().isEmpty()) {
        text = webView->selectedText();
        qWarning() <<  text;
    } else {
       
        text= "\""+treeWidget->currentItem()->text(0) + "\"\"" + treeWidget->currentItem()->text(1)+"\"";
          // title and author
    
        text = text.replace(" ", "+");
    }

    QString url = "http://google.com/search?q="+text+"&num=30&sa=Google+Search";
    new QAction( tr("Local"), this);
    resultsTab->show();
    webView_2->load(QUrl(url));
    webView_2->show();
    tabWidget->setCurrentIndex(3);
}

void Guten::treePressed( const QModelIndex &/*index*/)
{
    qWarning(treeWidget->currentItem()->text(1).toLocal8Bit());
    QMenu *treeMenu;
    treeMenu = new QMenu(this);
    treeMenu->addAction(tr("Open"), this,SLOT(treeClicked()));
    treeMenu->addAction(tr("More information"), this,SLOT(searchNet()));
    treeMenu->exec(QCursor::pos());
    
}

void Guten::treeClicked()
{

    treeClicked(treeWidget->currentItem(),0);

}

QStringList Guten::findFilesRecursively( QStringList paths, QStringList fileTypes ) 
{ //thanks Nathan Carter!
    if ( fileTypes.size() == 0)
        fileTypes << "*";


    
    QStringList result, more;
    QStringList::Iterator it;
    for ( int i = 0 ; i < paths.size() ; i++ ) { // inefficient...whatever
        qWarning("%d", i);
        QDir dir( paths[i] );
        more = dir.entryList( fileTypes, QDir::Files );
        for ( it = more.begin() ; it != more.end() ; ++it )
            result.append( paths[i] + "/" + *it );
          // reg exp in next line excludes . and .. dirs (and .* actually)
        more = dir.entryList( QDir::Dirs | QDir::NoDotAndDotDot );
        for ( it = more.begin() ; it != more.end() ; ++it )
            *it = paths[i] + "/" + *it;
        more = findFilesRecursively( more, fileTypes );
        for ( it = more.begin() ; it != more.end() ; ++it )
            result.append( *it );
    }    
    return result; // yields absolute paths
}


void Guten::moveToTop()
{
   
}


void Guten::moveToStart()
{
    qWarning() << "moveToSTart";
    if(!webView->findText("*END*THE SMALL PRINT" /*,QTextDocument::FindWholeWords)*/))
        webView->findText("***START OF THE PROJECT GUTENBERG EBOOK"/*,QTextDocument::FindWholeWords*/);

//  webView->textCursor().clearSelection();
//  QKeyEvent event(QEvent::KeyPress, Qt::Key_PageDown,Qt::NoModifier);
//  QApplication::sendEvent(webView, &event);

    webView->setFocus();
  
}


void Guten::startAutoScroll()
{
    int scrollTime = 600;
    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(timerScroll()));
    timer->start(scrollTime); 
}

void Guten::timerScroll()
{
    QKeyEvent event(QEvent::KeyPress, Qt::Key_Down,Qt::NoModifier);
    QApplication::sendEvent(webView, &event);    
}

void Guten::lookupClicked()
{
    QMessageBox::warning(this,"Blah",treeWidget->currentItem()->text(0));
}

void Guten::fetchEbook(const QString & ebook, int id)
{
    qWarning() << __PRETTY_FUNCTION__;
    
    QString filePath = ebook;
    QString fileName = filePath.right(filePath.length() - filePath.lastIndexOf('/') - 1);
      //   filePath = filePath.left(filePath.lastIndexOf('/'));
    
    qWarning() << fileName << filePath << id;
    
    QSettings *settings = getSettings();
    QString currentServer = settings->value("network/server").toString();

    if(currentServer.isEmpty()) {
        QMessageBox::information(this,tr("Server Settings"),
                                 tr("<p>Please select a Gutenberg server mirror to download your etext.</p>"),0);
        showMirrorSettings();
        currentServer =settings->value("network/server").toString();
        statusBar()->showMessage(tr("Please select a Gutenberg server mirror to download your etext."));
        return;
    }
    if(currentServer.right(1) == "/")
        currentServer.chop(1);
    filePath.simplified();
    
    if(id < 10001 && filePath.startsWith("/etext")) { // different directory scheme for older etexts
        qWarning() << "download directory listing first";
        QString directory = filePath.left( filePath.length() - fileName.length());
          //        directory.
          // directory.chop(1);
        qWarning() << directory;
        
//        qWarning("downloading "+ currentServer +filePath.toLocal8Bit());
        NetworkDialog *net1;
        net1 = new NetworkDialog();
        net1->setWindowTitle(tr("Downloading directory listing."));
        statusBar()->showMessage(tr("Downloading directory listing."));
        net1->setUrl( currentServer + directory );
        net1->setFile(TEMPFILE);
        if(net1->exec() == QDialog::Accepted) {
            fileName = findFile(fileName, preferredExtension);
            filePath = directory;
        }
    } 
    

    if(fileName.isEmpty()) {
        return;
    }
    qWarning() << "downloading" << "Server:" << currentServer << "Filename:"<< fileName <<"Filepath:" << filePath;
    if( !fileName.contains("."))
        fileName += preferredExtension;
    
    
      // filePath = currentServer + filePath + "/"+ fileName;
    QDir path(QDir::home());
    if(path.mkpath("Gutenberg"+filePath)) {
        QString tFileName = QDir::homePath()+"/Gutenberg"+filePath+"/"+fileName;
        NetworkDialog *net;
        net = new NetworkDialog(this);
        net->setWindowTitle(tr("Downloading Etext"));
        net->setUrl( currentServer + filePath + "/"+ fileName);
        net->setFile( tFileName);
        connect(net, SIGNAL(downloadFinished()), this,SLOT(fetchEbookDone()));
        connect(net, SIGNAL(downloadFailed()), this,SLOT(fetchEbookFailed()));
        downloadId = id;
        
        if(net->exec() == QDialog::Accepted) {
            
            if(QFileInfo(tFileName).exists()) {
                
                gutdb->setEtextPath( QString::number(id),tFileName );
            }
        }
        
    } else {
        QMessageBox::information(this,tr("Warning"),
                                 tr("<p>Could not make directory path.</p>"),0);
        return;
    }
}



/*!
  \fn Guten::loadStarted()
*/
void Guten::loadStarted()
{
    setCursor(Qt::WaitCursor /*Qt::BusyCursor*/);
}


/*!
  \fn Guten::loadProgress(int progress)
*/
void Guten::loadProgress(int progress)
{
    qWarning() << "Loading progress" << progress;
}


/*!
  \fn Guten::loadFinished(bool ok)
*/
void Guten::loadFinished(bool ok)
{
    Q_UNUSED(ok);
    setCursor(Qt::ArrowCursor);
}


/*!
  \fn Guten::treeSelectionChanged()
*/
void Guten::treeSelectionChanged()
{
    qWarning() << __PRETTY_FUNCTION__;
}


/*!
  \fn Guten::showDatabaseWarning()
*/
void Guten::showDatabaseWarning()
{
    QMessageBox::information(this,tr("Database"),
                             tr("<p>It looks like you need to update the Gutenberg Library Index."
                                "<br>Please chose Update from the Library menu.</p>"),0);  
}

void Guten::removeBook()
{
    if(!treeWidget->currentItem() || treeWidget->currentItem()->text(2) != "X")
        return;
    
    switch(QMessageBox::warning(this, "Gutenbrowser",
                                tr("<p>Really remove <br><b>%1?</b></p>").arg(treeWidget->currentItem()->text(0)),
                                tr("Ok"),
                                tr("Cancel"), 0, 0, 1)) {
      case 0:
          gutdb->removeBook( treeWidget->currentItem()->text(4));
          showInstalled();
          break;
      case 1:
          break;
    }          
}


/*!
  \fn Guten::removeFile(const QString &str)
*/
void Guten::removeFile(const QString &str)
{
    QFile file(str);
    if(file.exists())
        if(!file.remove()) {
            QMessageBox::information(this,tr("Gutenbrowser"), tr("Could not remove file"));
        }
}


/*!
  \fn Guten::findFileName(const QString &path, const QString &extension)
*/
QString Guten::findFileName(const QString &path, const QString &extension)
{
    qWarning() << __PRETTY_FUNCTION__ << path << extension;
    
    QDirIterator it(path, QDirIterator::Subdirectories);
    while (it.hasNext()) {
        QFileInfo fi = it.next();
        QString base = fi.baseName();
        qWarning() << "Found" << base << fi.completeSuffix();
        if(fi.completeSuffix() == extension) {
            qWarning() << __PRETTY_FUNCTION__ <<  fi.absoluteFilePath();
            return fi.absoluteFilePath();
        } else if(fi.completeSuffix() == "txt") {
            qWarning() << __PRETTY_FUNCTION__ <<  fi.absoluteFilePath();
            return fi.absoluteFilePath();
        }
        
    }
    return "";
}


/*!
  \fn Guten::about()
*/
void Guten::about()
{ 
    QMessageBox::about(this, tr("About"), tr("<P><b>Gutenbrowser</b> is copyright 1998-2008 by<br> lpotter &lt;lorn.potter@gmail.com&gt;</P>"
                                             "<p>\"Free flow of information is the only safeguard against tyranny. The once-chained people who's leaders at last lose their grip on information will soon burst with freedom and vitality, but the free nation gradually constricting its grip on public discourse has begun its rapid slide into despotism. Beware he who would deny you access to information, for in his heart he dreams himself your master.\" - Unknown</P>" 
                                             
                                             "<p>Gutenbrowser is a multi platform reader for etexts republished by Project Gutenberg. Project Gutenberg (http://www.gutenberg.org) aims to make public domain literature available to the world.</P>"
                                             
                                             "<P>This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or  (at your option) any later version.</P>"
                                             
                                             "<P>This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.</P>"
                                             
                                             "<P>You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA</P>"
                                            
                                             ));
}


/*!
  \fn Guten::fetchEbookDone()
*/
void Guten::fetchEbookDone()
{
    if(gutdb->installEtext( QString::number(downloadId))) {
        treeWidget->currentItem()->setText(2,"X");
        QMessageBox::information(this,tr("Gutenbrowser Install"), tr("Now installed"));
        emit openSuccess();
    }
}


/*!
  \fn Guten::fetchEbookFailed(const QString & str)
*/
void Guten::fetchEbookFailed()
{
    switch(QMessageBox::warning(this, tr("Gutenbrowser Download"),
                                tr("<p>Some books do not come in html, nor in zip files. This may be one reason your download failed.<br>"
                                   "Try downloading the same book but in a different form?</p>"),
                                tr("Ok"),
                                tr("Cancel"), 0, 0, 1)) {
      case 0:
          preferredExtension = ".txt";
          fetchEbook(treeWidget->currentItem()->text(3), treeWidget->currentItem()->text(4).toInt());
                                    
          break;
      case 1:
          break;
    }    
}


/*!
  \fn Guten::openOk()
*/
void Guten::openOk()
{
    tabWidget->setTabText(2, treeWidget->currentItem()->text(0));
    statusBar()->showMessage(treeWidget->currentItem()->text(0));
}
