/* gtd_ring.c - Handle the sending of an announcement to a ringer process
 *
 * Copyright (C) 1997, 98 Free Software Foundation
 * Copyright (C) 1994, 1995, 1998 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *  Handles the sending of an announcement message.  It does this by
 * loading a ".ringer" file from the users home account, and using
 * it's specification to send a message to a given user.
 * 
 * $Log: gtd_ring.c,v $
 * Revision 1.19  1998/10/17 21:59:52  zappo
 * Upgraded the filling of the ringer message.
 *
 * Revision 1.18  1998/02/15 13:58:41  zappo
 * Use extended name buffers which prevent buffer overflows.
 *
 * Revision 1.17  1998/01/04 13:33:41  zappo
 * Fixed warnings
 *
 * Revision 1.16  1997/12/14 19:19:48  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.15  1997/10/08 00:50:20  zappo
 * Cast localtime parameter to the correct type.
 *
 * Revision 1.14  1997/02/21 04:03:02  zappo
 * More comments
 *
 * Revision 1.13  1996/06/25 03:03:29  zappo
 * Updated how success/failure is reported back so it is more stable.
 *
 * Revision 1.12  1996/03/02  03:32:58  zappo
 * Fixed some warnings
 *
 * Revision 1.11  1995/12/10  03:56:42  zappo
 * error messages now use DISP_message The LOOK_HERE response was being
 * generated with the CALLER's ringer was on a different machine from the
 * daemon, not when the CALLEE was on a different machine from the daemon.
 *
 * Revision 1.10  1995/09/29  09:10:03  zappo
 * Removed traces of ringer response message
 *
 * Revision 1.9  1995/09/22  13:46:36  zappo
 * Client recieving a ringer message no longer needs to send an
 * acknowledgement back, as there was never a "failed" anyway.
 *
 * Revision 1.8  1995/07/23  21:51:54  zappo
 * In previous fix forgot to change sizeof the address being used.
 *
 * Revision 1.7  1995/07/23  21:19:50  zappo
 * Forwarding can mess up the printed name of hosts.  Grab the TARGET
 * address for the name first, and if that fails, then grab the return
 * address.
 *
 * Revision 1.6  1995/07/15  19:51:01  zappo
 * Replaced constant with a define.
 *
 * Revision 1.5  1995/04/05  00:57:44  zappo
 * In the case of really long application names being inserted, use
 * strncpy to copy contents into message buffer.
 *
 * Revision 1.4  1995/04/01  18:58:22  zappo
 * Removed &'s in array indicies
 *
 * Revision 1.3  1995/03/25  04:23:38  zappo
 * Updated copyright
 *
 * Revision 1.2  1995/02/11  17:06:35  zappo
 * Added forwarding capability when ringer read in is not the same as
 * current host
 *
 * Revision 1.1  1995/02/01  03:50:25  zappo
 * Initial revision
 *
 * History:
 * zappo   11/20/94    Created
 *
 * Tokens: ::Header:: gtalkd.h 
 */

#include "gtalklib.h"
#include "gtalkd.h"

static GNU_RING_CONTROL   rc;


/*
 * Function: RINGER_sendto
 *
 *   Sends an announcement message to a process designated by a user
 * as a "ringer".
 *
 * Returns:     int  - 
 * Parameters:  uo   - Pointer to user object
 *              ro   - Pointer to request object
 *              Ctxt - Context
 * History:
 * zappo   11/20/94   Created
 */
int RINGER_sendto(uo, ro, Ctxt)
     struct UserObject    *uo;
     struct RequestObject *ro;
     struct DaemonContext *Ctxt;
{
  static char *ampm[] = { "am", "pm" };
  char                buff[300];
  int                 ap, hr;
  struct HostObject  *host;
  struct tm          *clock;
  struct InputDevice *rdev;
  int                 ans;
  char               *extendfetch;
  int                 fetchlength;

  rdev = RING_open(uo->ringf, Ctxt->udp_ring);

  if(! rdev)
    {
      /* In this case, no ringer file available */
      uo->ringto = NULL;
      return FAILED;
    }

  /* Set who the ringer is set to here... */
  uo->ringto = rdev;

  /* Get the hostname for the message we are sending. */
  host = HOST_gen_host_by_addr((struct sockaddr *)&ro->request.addr,
			       sizeof(ro->request.addr));
  /* We just grabbed the TARGET host.  This is the address the
     other guy wants us to announce as.  If it fails, then grab
     the name of the machine which just sent us the address.  This
     could be wrong due to forwarding, but we need to advertise
     something. */
  if(!host)
    host = HOST_gen_host_by_addr((struct sockaddr *)&ro->sender->raddr,
				 sizeof(ro->sender->raddr));
  if(!host)
    {
      if(verbose)
	printf("Error finding printable host name!\n");
      return BADADDR;
    }

  /* At this point:  host  = host id of request sender
   *                 rdev  = udp socket id of ringer host
   *             Ctxt->me  = My host structure
   *
   * to make sure that we are on the same host as the sender... */
  if(verbose)
    printf("Is ring local? Only if (%d == %d)\n", 
	   rdev->raddr.sin_addr.s_addr,
	   Ctxt->me->addr.sin_addr.s_addr);

  if(rdev->raddr.sin_addr.s_addr != Ctxt->me->addr.sin_addr.s_addr)
    {
      /* Return the new error code (Not here but...) */
      return TRY_HERE;
    }

  /* Okie dokie.  Let write out a nice little message.
   */
  clock = localtime((time_t *)&ro->stamp.tv_sec);
  
  /* Find the 12 hour clock time.
   */
  if(clock->tm_hour > 12)
    {
      ap = 1;
      hr = clock->tm_hour - 12;
    }
  else
    {
      ap = 0;
      hr = clock->tm_hour;
    }

  /* This is where we protect from bomb-characters in gtd_ann.  Instead
   * we let them through because we expect the app recieving this ring
   * to be smart enough to handle it.  It might even contain useful
   * bytes in some far off future day.
   */
  sprintf((char *)rc.msg1, "Message from GNU Talk Daemon: %d/%d/%d %d:%02d%s",
	  clock->tm_mon+1, clock->tm_mday, clock->tm_year,
	  hr, clock->tm_min, ampm[ap]);

  if((extendfetch = (char *)
      EXT_fetch_extension_data(EXTENSION_PERSONAL_NAME,
			       ro->extend, &fetchlength)))
    {
      sprintf(rc.msg2, "gtalk: connection requested by %*s at %s\n\r",
	      fetchlength-1, extendfetch, host->name);
    }
  else
    {
      sprintf(rc.msg2, "gtalk: connection requested by %s@%s\n\r",
	      ro->l_name, host->name);
    }
  
  /* We must be careful to tell the user what kind of response to use.
   */
  if(ro->request.vers == 0)
    {
      sprintf(buff, "gtalk: respond with \"otalk %s@%s\"\n\n\r",
	      ro->l_name, host->name);
    }
  else if((ro->request.vers == TALK_VERSION) ||
	  !(extendfetch = (char *)
	    EXT_fetch_extension_data(EXTENSION_APPLICATION_NAME,
				     ro->extend, &fetchlength)))
    {
      sprintf(buff, "gtalk: respond with \"talk %s@%s\"\n\n\r",
	      ro->l_name, host->name);
    }
  else
    {
      sprintf(buff, "gtalk: respond with \"%*s %s@%s\"\n\n\r",
	      fetchlength-1, extendfetch, /* extension data */
	      ro->l_name, host->name);
    }
    
  strncpy((char *)rc.msg3, buff, RINGER_MESSAGE_LEN);
  buff[RINGER_MESSAGE_LEN - 1] = 0;

  rc.vers = TALK_VERSION_GNU;
  rc.type = RINGER_CALL;
  rc.addr = *((struct sockaddr *)&Ctxt->udp_ring->laddr);
  strcpy(rc.name, ro->l_name);
  rc.raddr = *((struct sockaddr *)&host->addr);
  /* Get stuff into network order for the family. */
  ((struct sockaddr_in *)&rc.raddr)->sin_family = 
    htons(((struct sockaddr_in *)&rc.raddr)->sin_family);

  if(verbose)
    {
      printf("Sending ringer request to read in address: ");
      print_sockaddr((struct sockaddr *)&rdev->raddr);
      printf("\n");
    }

  /* Send the message */
  ans = GT_send(rdev, (char *)&rc, sizeof(rc));

  /* Translate return code into talk.h codes. */
  if(ans == Success)
    return SUCCESS;
  else
    return FAILED;
}
